const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');
const dotenv = require('dotenv');

// Get database URLs from .env files
function getDbUrls() {
  const envPaths = [
    { path: '.env', name: 'Default .env' },
    { path: '.env.local', name: 'Local .env.local' }
  ];
  
  const dbUrls = [];
  
  for (const { path, name } of envPaths) {
    if (fs.existsSync(path)) {
      const envContent = fs.readFileSync(path, 'utf8');
      const envConfig = dotenv.parse(envContent);
      
      if (envConfig.DATABASE_URL) {
        // Parse the URL to get database name
        const urlPattern = /mysql:\/\/([^:]+):?([^@]*)@([^:]+):(\d+)\/([^?]+)/;
        const match = envConfig.DATABASE_URL.match(urlPattern);
        
        if (match) {
          const [, user, password, host, port, dbName] = match;
          
          dbUrls.push({
            source: name,
            url: envConfig.DATABASE_URL,
            config: {
              user,
              password,
              host,
              port: parseInt(port, 10),
              database: dbName
            }
          });
          
          console.log(`Found database URL in ${name}: ${dbName}`);
        }
      }
    }
  }
  
  return dbUrls;
}

// Run Prisma db push for a specific database
async function runDbPush(dbUrl) {
  console.log(`\nRunning Prisma db push for ${dbUrl}...`);
  try {
    // Run database migrations
    console.log('Pushing schema to database...');
    execSync('npx prisma db push', { 
      stdio: 'inherit',
      env: { ...process.env, DATABASE_URL: dbUrl }
    });
    
    console.log('Schema update complete.');
    return true;
  } catch (error) {
    console.error('Error running schema update:', error);
    return false;
  }
}

// Main function
async function main() {
  try {
    console.log('Updating database schema...');
    
    // Get database URLs from .env files
    const dbUrls = getDbUrls();
    
    if (dbUrls.length === 0) {
      console.error('No database URLs found in .env files.');
      return;
    }
    
    // Process each database
    for (const { source, url, config } of dbUrls) {
      console.log(`\n========================================`);
      console.log(`Processing database from ${source}`);
      console.log(`Database: ${config.database}`);
      console.log(`========================================`);
      
      // Run migrations
      const schemaUpdateSuccessful = await runDbPush(url);
      if (!schemaUpdateSuccessful) {
        console.error(`Failed to update schema for ${config.database}. Continuing with next database.`);
        continue;
      }
      
      console.log(`\n✅ Successfully updated schema for database: ${config.database}`);
    }
    
    console.log('\n========================================');
    console.log('All database schemas updated successfully!');
    console.log('========================================');
    console.log('Added new timeline field to AuditFinding model');
    console.log('========================================');
    
  } catch (error) {
    console.error('Error updating database schema:', error);
  }
}

// Add the migration for required_threshold to the update function
async function updateSchema() {
  const queries = [
    // ... existing migrations
    
    // Add required_threshold to standard_compliance table
    `ALTER TABLE standard_compliance ADD COLUMN required_threshold FLOAT NOT NULL DEFAULT 90;`
  ];
  
  // ... rest of the function
}

main(); 