const { execSync } = require('child_process');
const { PrismaClient } = require('@prisma/client');
const { hash } = require('bcrypt');
const mysql = require('mysql2');

// Database connection parameters
const DB_HOST = 'localhost';
const DB_USER = 'root';
const DB_PASSWORD = '';
const DB_NAME = 'health_audit';
const DB_URL = `mysql://${DB_USER}:${DB_PASSWORD}@${DB_HOST}:3306/${DB_NAME}`;

// Create the database if it doesn't exist
async function createDatabase() {
  console.log('Checking if database exists...');
  
  const connection = mysql.createConnection({
    host: DB_HOST,
    user: DB_USER,
    password: DB_PASSWORD
  });
  
  return new Promise((resolve, reject) => {
    connection.query(`CREATE DATABASE IF NOT EXISTS ${DB_NAME}`, (err) => {
      connection.end();
      
      if (err) {
        console.error('Error creating database:', err);
        reject(err);
        return;
      }
      
      console.log(`Database '${DB_NAME}' created or already exists.`);
      resolve();
    });
  });
}

// Run Prisma migrations
async function runMigrations() {
  console.log('\nRunning Prisma migrations...');
  try {
    // Generate Prisma client
    console.log('Generating Prisma client...');
    execSync('npx prisma generate', { 
      stdio: 'inherit',
      env: { ...process.env, DATABASE_URL: DB_URL }
    });
    
    // Run database migrations
    console.log('Pushing schema to database...');
    execSync('npx prisma db push --force-reset', { 
      stdio: 'inherit',
      env: { ...process.env, DATABASE_URL: DB_URL }
    });
    
    console.log('Migrations complete.');
    return true;
  } catch (error) {
    console.error('Error running migrations:', error);
    return false;
  }
}

// Create users and test data
async function createTestData() {
  console.log('\nCreating test data...');
  
  // Set the DATABASE_URL for the PrismaClient
  process.env.DATABASE_URL = DB_URL;
  
  const prisma = new PrismaClient();
  
  try {
    // Create hospitals
    console.log('Creating hospitals...');
    const hospital1 = await prisma.hospital.create({
      data: {
        name: 'City General Hospital',
        location: 'New York, NY',
        logo_url: '/images/hospitals/city-general.png',
      }
    });
    console.log(`Hospital created: ${hospital1.name} (ID: ${hospital1.id})`);
    
    const hospital2 = await prisma.hospital.create({
      data: {
        name: 'County Medical Center',
        location: 'Los Angeles, CA',
        logo_url: '/images/hospitals/county-medical.png',
      }
    });
    console.log(`Hospital created: ${hospital2.name} (ID: ${hospital2.id})`);
    
    // Create super admin user
    console.log('\nCreating super admin user...');
    const adminPassword = await hash('admin123', 10);
    const admin = await prisma.user.create({
      data: {
        username: 'superadmin',
        email: 'superadmin@healthguardian.com',
        password: adminPassword,
        role: 'super_admin',
      }
    });
    console.log('Super admin created:');
    console.log(`Email: ${admin.email}`);
    console.log(`Password: admin123`);
    
    // Create hospital user
    console.log('\nCreating hospital user...');
    const hospitalPassword = await hash('hospital123', 10);
    const hospitalUser = await prisma.user.create({
      data: {
        username: 'hospital',
        email: 'hospital@healthguardian.com',
        password: hospitalPassword,
        role: 'hospital',
        hospital_id: hospital1.id,
      }
    });
    console.log('Hospital user created:');
    console.log(`Email: ${hospitalUser.email}`);
    console.log(`Password: hospital123`);
    console.log(`Hospital: ${hospital1.name} (ID: ${hospital1.id})`);
    
    console.log('\nTest data creation complete!');
    return true;
  } catch (error) {
    console.error('Error creating test data:', error);
    return false;
  } finally {
    await prisma.$disconnect();
  }
}

// Main function
async function main() {
  try {
    console.log('Setting up database for authentication testing...');
    
    // Create the database
    await createDatabase();
    
    // Run migrations
    const migrationsSuccessful = await runMigrations();
    if (!migrationsSuccessful) {
      console.error('Failed to run migrations. Aborting.');
      return;
    }
    
    // Create test data
    const dataCreationSuccessful = await createTestData();
    if (!dataCreationSuccessful) {
      console.error('Failed to create test data. Aborting.');
      return;
    }
    
    console.log('\n========================================');
    console.log('Database setup complete!');
    console.log('========================================');
    console.log('You can now use the following credentials:');
    console.log('\nSuper Admin User:');
    console.log('- Email: superadmin@healthguardian.com');
    console.log('- Password: admin123');
    console.log('\nHospital User:');
    console.log('- Email: hospital@healthguardian.com');
    console.log('- Password: hospital123');
    console.log('========================================');
    
  } catch (error) {
    console.error('Error setting up database:', error);
  }
}

main(); 