const { execSync } = require('child_process');
const { PrismaClient } = require('@prisma/client');
const { hash } = require('bcrypt');
const mysql = require('mysql2');
const fs = require('fs');
const dotenv = require('dotenv');

// Get database URLs from .env files
function getDbUrls() {
  const envPaths = [
    { path: '.env', name: 'Default .env' },
    { path: '.env.local', name: 'Local .env.local' }
  ];
  
  const dbUrls = [];
  
  for (const { path, name } of envPaths) {
    if (fs.existsSync(path)) {
      const envContent = fs.readFileSync(path, 'utf8');
      const envConfig = dotenv.parse(envContent);
      
      if (envConfig.DATABASE_URL) {
        // Parse the URL to get database name
        const urlPattern = /mysql:\/\/([^:]+):?([^@]*)@([^:]+):(\d+)\/([^?]+)/;
        const match = envConfig.DATABASE_URL.match(urlPattern);
        
        if (match) {
          const [, user, password, host, port, dbName] = match;
          
          dbUrls.push({
            source: name,
            url: envConfig.DATABASE_URL,
            config: {
              user,
              password,
              host,
              port: parseInt(port, 10),
              database: dbName
            }
          });
          
          console.log(`Found database URL in ${name}: ${dbName}`);
        }
      }
    }
  }
  
  return dbUrls;
}

// Create the database if it doesn't exist
async function createDatabase(dbConfig) {
  console.log(`Checking if database ${dbConfig.database} exists...`);
  
  const connection = mysql.createConnection({
    host: dbConfig.host,
    user: dbConfig.user,
    password: dbConfig.password
  });
  
  return new Promise((resolve, reject) => {
    connection.query(`CREATE DATABASE IF NOT EXISTS ${dbConfig.database}`, (err) => {
      connection.end();
      
      if (err) {
        console.error(`Error creating database ${dbConfig.database}:`, err);
        reject(err);
        return;
      }
      
      console.log(`Database '${dbConfig.database}' created or already exists.`);
      resolve();
    });
  });
}

// Run Prisma migrations for a specific database
async function runMigrations(dbUrl) {
  console.log(`\nRunning Prisma migrations for ${dbUrl}...`);
  try {
    // Generate Prisma client
    console.log('Generating Prisma client...');
    execSync('npx prisma generate', { 
      stdio: 'inherit',
      env: { ...process.env, DATABASE_URL: dbUrl }
    });
    
    // Run database migrations
    console.log('Pushing schema to database...');
    execSync('npx prisma db push --force-reset', { 
      stdio: 'inherit',
      env: { ...process.env, DATABASE_URL: dbUrl }
    });
    
    console.log('Migrations complete.');
    return true;
  } catch (error) {
    console.error('Error running migrations:', error);
    return false;
  }
}

// Create users and test data
async function createTestData(dbUrl) {
  console.log(`\nCreating test data in ${dbUrl}...`);
  
  // Set the DATABASE_URL for the PrismaClient
  process.env.DATABASE_URL = dbUrl;
  
  const prisma = new PrismaClient();
  
  try {
    // Create hospitals
    console.log('Creating hospitals...');
    const hospital1 = await prisma.hospital.create({
      data: {
        name: 'City General Hospital',
        location: 'New York, NY',
        logo_url: '/images/hospitals/city-general.png',
      }
    });
    console.log(`Hospital created: ${hospital1.name} (ID: ${hospital1.id})`);
    
    const hospital2 = await prisma.hospital.create({
      data: {
        name: 'County Medical Center',
        location: 'Los Angeles, CA',
        logo_url: '/images/hospitals/county-medical.png',
      }
    });
    console.log(`Hospital created: ${hospital2.name} (ID: ${hospital2.id})`);
    
    // Create super admin user
    console.log('\nCreating super admin user...');
    const adminPassword = await hash('admin123', 10);
    const admin = await prisma.user.create({
      data: {
        username: 'superadmin',
        email: 'superadmin@healthguardian.com',
        password: adminPassword,
        role: 'super_admin',
      }
    });
    console.log('Super admin created:');
    console.log(`Email: ${admin.email}`);
    console.log(`Password: admin123`);
    
    // Create hospital user
    console.log('\nCreating hospital user...');
    const hospitalPassword = await hash('hospital123', 10);
    const hospitalUser = await prisma.user.create({
      data: {
        username: 'hospital',
        email: 'hospital@healthguardian.com',
        password: hospitalPassword,
        role: 'hospital',
        hospital_id: hospital1.id,
      }
    });
    console.log('Hospital user created:');
    console.log(`Email: ${hospitalUser.email}`);
    console.log(`Password: hospital123`);
    console.log(`Hospital: ${hospital1.name} (ID: ${hospital1.id})`);
    
    console.log('\nTest data creation complete!');
    return true;
  } catch (error) {
    console.error('Error creating test data:', error);
    return false;
  } finally {
    await prisma.$disconnect();
  }
}

// Main function
async function main() {
  try {
    console.log('Setting up databases for authentication testing...');
    
    // Get database URLs from .env files
    const dbUrls = getDbUrls();
    
    if (dbUrls.length === 0) {
      console.error('No database URLs found in .env files.');
      return;
    }
    
    // Process each database
    for (const { source, url, config } of dbUrls) {
      console.log(`\n========================================`);
      console.log(`Processing database from ${source}`);
      console.log(`Database: ${config.database}`);
      console.log(`========================================`);
      
      // Create the database
      await createDatabase(config);
      
      // Run migrations
      const migrationsSuccessful = await runMigrations(url);
      if (!migrationsSuccessful) {
        console.error(`Failed to run migrations for ${config.database}. Continuing with next database.`);
        continue;
      }
      
      // Create test data
      const dataCreationSuccessful = await createTestData(url);
      if (!dataCreationSuccessful) {
        console.error(`Failed to create test data for ${config.database}. Continuing with next database.`);
        continue;
      }
      
      console.log(`\n✅ Successfully set up database: ${config.database}`);
    }
    
    console.log('\n========================================');
    console.log('All database setup complete!');
    console.log('========================================');
    console.log('You can now use the following credentials:');
    console.log('\nSuper Admin User:');
    console.log('- Email: superadmin@healthguardian.com');
    console.log('- Password: admin123');
    console.log('\nHospital User:');
    console.log('- Email: hospital@healthguardian.com');
    console.log('- Password: hospital123');
    console.log('========================================');
    
  } catch (error) {
    console.error('Error setting up databases:', error);
  }
}

main(); 