// This script will help debug the authentication process
const { PrismaClient } = require('@prisma/client');
const { compare, hash } = require('bcrypt');

const prisma = new PrismaClient();

async function main() {
  try {
    console.log('Debug Authentication Process');
    
    // 1. Check if the database is connected
    console.log('\n1. Testing database connection...');
    try {
      // Try a simple query to check the connection
      const userCount = await prisma.user.count();
      console.log(`✅ Database connection successful. Found ${userCount} users in the database.`);
    } catch (error) {
      console.error('❌ Database connection failed:', error.message);
      
      // Show environment info
      console.log('\nEnvironment variables:');
      console.log(`DATABASE_URL: ${process.env.DATABASE_URL || 'Not set'}`);
      console.log(`NODE_ENV: ${process.env.NODE_ENV || 'Not set'}`);
      return;
    }
    
    // 2. Check for the superadmin user
    console.log('\n2. Checking for superadmin user...');
    const superAdmin = await prisma.user.findUnique({
      where: { email: 'superadmin@healthguardian.com' },
      include: { hospital: true },
    });
    
    if (superAdmin) {
      console.log('✅ Super admin found:');
      console.log(`   ID: ${superAdmin.id}`);
      console.log(`   Username: ${superAdmin.username}`);
      console.log(`   Email: ${superAdmin.email}`);
      console.log(`   Role: ${superAdmin.role}`);
      
      // Test the password
      const testPassword = 'admin123';
      try {
        const passwordValid = await compare(testPassword, superAdmin.password);
        console.log(`   Password '${testPassword}' valid: ${passwordValid ? '✅ Yes' : '❌ No'}`);
        
        if (!passwordValid) {
          // Create a new password hash for troubleshooting
          const newHash = await hash(testPassword, 10);
          console.log(`   Debug: New hash for '${testPassword}': ${newHash}`);
          
          // Update the password
          console.log('   Updating superadmin password...');
          await prisma.user.update({
            where: { id: superAdmin.id },
            data: { password: newHash },
          });
          console.log('   ✅ Password updated successfully.');
        }
      } catch (error) {
        console.error(`   ❌ Error checking password: ${error.message}`);
      }
    } else {
      console.log('❌ Super admin not found. Let\'s create one:');
      
      try {
        // Create the super admin
        const hashedPassword = await hash('admin123', 10);
        const newAdmin = await prisma.user.create({
          data: {
            username: 'superadmin',
            password: hashedPassword,
            email: 'superadmin@healthguardian.com',
            role: 'super_admin',
          },
        });
        
        console.log('✅ Super admin created:');
        console.log(`   ID: ${newAdmin.id}`);
        console.log(`   Username: ${newAdmin.username}`);
        console.log(`   Email: ${newAdmin.email}`);
        console.log(`   Role: ${newAdmin.role}`);
      } catch (error) {
        console.error(`❌ Error creating super admin: ${error.message}`);
      }
    }
    
    // 3. Check for the hospital user
    console.log('\n3. Checking for hospital user...');
    const hospitalUser = await prisma.user.findUnique({
      where: { email: 'hospital@healthguardian.com' },
      include: { hospital: true },
    });
    
    if (hospitalUser) {
      console.log('✅ Hospital user found:');
      console.log(`   ID: ${hospitalUser.id}`);
      console.log(`   Username: ${hospitalUser.username}`);
      console.log(`   Email: ${hospitalUser.email}`);
      console.log(`   Role: ${hospitalUser.role}`);
      console.log(`   Hospital ID: ${hospitalUser.hospital_id || 'None'}`);
      
      if (hospitalUser.hospital) {
        console.log(`   Hospital: ${hospitalUser.hospital.name}`);
      } else {
        console.log('   ❌ Not associated with any hospital');
      }
      
      // Test the password
      const testPassword = 'hospital123';
      try {
        const passwordValid = await compare(testPassword, hospitalUser.password);
        console.log(`   Password '${testPassword}' valid: ${passwordValid ? '✅ Yes' : '❌ No'}`);
        
        if (!passwordValid) {
          // Create a new password hash for troubleshooting
          const newHash = await hash(testPassword, 10);
          console.log(`   Debug: New hash for '${testPassword}': ${newHash}`);
          
          // Update the password
          console.log('   Updating hospital user password...');
          await prisma.user.update({
            where: { id: hospitalUser.id },
            data: { password: newHash },
          });
          console.log('   ✅ Password updated successfully.');
        }
      } catch (error) {
        console.error(`   ❌ Error checking password: ${error.message}`);
      }
    } else {
      console.log('❌ Hospital user not found. Checking for hospitals first...');
      
      // Check if any hospitals exist
      const hospitals = await prisma.hospital.findMany({ take: 1 });
      
      if (hospitals.length === 0) {
        console.log('   Creating a hospital first...');
        const hospital = await prisma.hospital.create({
          data: {
            name: 'City General Hospital',
            location: 'New York, NY',
            logo_url: '/images/hospitals/city-general.png',
          },
        });
        console.log(`   ✅ Hospital created: ${hospital.name} (ID: ${hospital.id})`);
        
        // Now create the hospital user
        try {
          const hashedPassword = await hash('hospital123', 10);
          const newHospitalUser = await prisma.user.create({
            data: {
              username: 'hospital',
              password: hashedPassword,
              email: 'hospital@healthguardian.com',
              role: 'hospital',
              hospital_id: hospital.id,
            },
          });
          
          console.log('✅ Hospital user created:');
          console.log(`   ID: ${newHospitalUser.id}`);
          console.log(`   Username: ${newHospitalUser.username}`);
          console.log(`   Email: ${newHospitalUser.email}`);
          console.log(`   Role: ${newHospitalUser.role}`);
          console.log(`   Hospital: ${hospital.name} (ID: ${hospital.id})`);
        } catch (error) {
          console.error(`❌ Error creating hospital user: ${error.message}`);
        }
      } else {
        const hospital = hospitals[0];
        console.log(`   Using existing hospital: ${hospital.name} (ID: ${hospital.id})`);
        
        try {
          const hashedPassword = await hash('hospital123', 10);
          const newHospitalUser = await prisma.user.create({
            data: {
              username: 'hospital',
              password: hashedPassword,
              email: 'hospital@healthguardian.com',
              role: 'hospital',
              hospital_id: hospital.id,
            },
          });
          
          console.log('✅ Hospital user created:');
          console.log(`   ID: ${newHospitalUser.id}`);
          console.log(`   Username: ${newHospitalUser.username}`);
          console.log(`   Email: ${newHospitalUser.email}`);
          console.log(`   Role: ${newHospitalUser.role}`);
          console.log(`   Hospital: ${hospital.name} (ID: ${hospital.id})`);
        } catch (error) {
          console.error(`❌ Error creating hospital user: ${error.message}`);
        }
      }
    }
    
    // 4. Authentication simulation
    console.log('\n4. Simulating login authentication...');
    const credentials = [
      { email: 'superadmin@healthguardian.com', password: 'admin123' },
      { email: 'hospital@healthguardian.com', password: 'hospital123' },
      { email: 'wrong@example.com', password: 'wrongpassword' },
    ];
    
    for (const cred of credentials) {
      console.log(`\nTesting login with: ${cred.email} / ${cred.password}`);
      
      try {
        const user = await prisma.user.findUnique({
          where: { email: cred.email },
          include: { hospital: true },
        });
        
        if (!user) {
          console.log(`❌ User not found: ${cred.email}`);
          continue;
        }
        
        const isPasswordValid = await compare(cred.password, user.password);
        
        if (isPasswordValid) {
          console.log('✅ Authentication successful!');
          console.log(`   User: ${user.username} (${user.email})`);
          console.log(`   Role: ${user.role}`);
          console.log(`   Hospital ID: ${user.hospital_id || 'None'}`);
          
          if (user.hospital) {
            console.log(`   Hospital: ${user.hospital.name}`);
          }
        } else {
          console.log('❌ Password invalid');
        }
      } catch (error) {
        console.error(`❌ Authentication error: ${error.message}`);
      }
    }
    
    console.log('\nDebug completed!');
  } catch (error) {
    console.error('Error during debug:', error);
  } finally {
    await prisma.$disconnect();
  }
}

main(); 