import { PrismaClient } from '@prisma/client';
import { hash } from 'bcryptjs';

const prisma = new PrismaClient();

async function main() {
  try {
    // Seed hospitals
    const hospital1 = await prisma.hospital.upsert({
      where: { id: 1 },
      update: {},
      create: {
        name: 'City General Hospital',
        location: 'New York, NY',
        logo_url: '/images/hospitals/city-general.png',
      },
    });
    console.log(`Created hospital: ${hospital1.name}`);

    const hospital2 = await prisma.hospital.upsert({
      where: { id: 2 },
      update: {},
      create: {
        name: 'County Medical Center',
        location: 'Los Angeles, CA',
        logo_url: '/images/hospitals/county-medical.png',
      },
    });
    console.log(`Created hospital: ${hospital2.name}`);

    // Seed users
    const adminPassword = await hash('password123', 10);
    const admin = await prisma.user.upsert({
      where: { email: 'admin@example.com' },
      update: {},
      create: {
        username: 'admin',
        password: adminPassword,
        email: 'admin@example.com',
        role: 'super_admin',
      },
    });
    console.log(`Created super_admin user: ${admin.email}`);

    const userPassword = await hash('user123', 10);
    const user = await prisma.user.upsert({
      where: { email: 'user@example.com' },
      update: {},
      create: {
        username: 'user',
        password: userPassword,
        email: 'user@example.com',
        role: 'hospital',
        hospital_id: hospital1.id,
      },
    });
    console.log(`Created hospital user: ${user.email}`);

    // Seed compliance data
    await prisma.$transaction(async (tx) => {
      // Clear existing data
      await prisma.complianceData.deleteMany();
      
      // Seed compliance data
      const complianceItems = [
        { name: 'ISO 9001', compliance: 87, required: 90, hospital_id: hospital1.id },
        { name: 'HIPAA', compliance: 94, required: 90, hospital_id: hospital1.id },
        { name: 'JCI', compliance: 78, required: 90, hospital_id: hospital1.id },
        { name: 'GDPR', compliance: 82, required: 90, hospital_id: hospital1.id },
        { name: 'FDA 21 CFR', compliance: 76, required: 90, hospital_id: hospital1.id },
        { name: 'OSHA', compliance: 88, required: 90, hospital_id: hospital1.id },
        { name: 'ISO 27001', compliance: 81, required: 90, hospital_id: hospital1.id },
        { name: 'ISO 14001', compliance: 79, required: 85, hospital_id: hospital1.id },
        
        { name: 'ISO 9001', compliance: 91, required: 90, hospital_id: hospital2.id },
        { name: 'HIPAA', compliance: 89, required: 90, hospital_id: hospital2.id },
        { name: 'JCI', compliance: 82, required: 90, hospital_id: hospital2.id },
        { name: 'GDPR', compliance: 77, required: 90, hospital_id: hospital2.id },
        { name: 'FDA 21 CFR', compliance: 84, required: 90, hospital_id: hospital2.id },
        { name: 'OSHA', compliance: 92, required: 90, hospital_id: hospital2.id },
        { name: 'ISO 27001', compliance: 86, required: 90, hospital_id: hospital2.id },
        { name: 'ISO 14001', compliance: 81, required: 85, hospital_id: hospital2.id },
      ];
      
      for (const item of complianceItems) {
        await prisma.complianceData.create({ data: item });
      }
      console.log(`Created ${complianceItems.length} compliance items`);
      
      // Clear existing standards data
      await prisma.standardCompliance.deleteMany();
      
      // Seed standards compliance data
      const standardItems = [
        { standard: 'HIPAA Privacy Rule', category: 'Data Privacy', compliance: 94, last_assessed: new Date('2023-05-28'), hospital_id: hospital1.id },
        { standard: 'ISO 9001:2015', category: 'Quality Management', compliance: 82, last_assessed: new Date('2023-06-10'), hospital_id: hospital1.id },
        { standard: 'JCI 7th Edition', category: 'Patient Safety', compliance: 68, last_assessed: new Date('2023-06-15'), hospital_id: hospital1.id },
        { standard: 'HIPAA Security Rule', category: 'Data Security', compliance: 88, last_assessed: new Date('2023-05-15'), hospital_id: hospital1.id },
        { standard: 'GDPR', category: 'Data Privacy', compliance: 76, last_assessed: new Date('2023-04-20'), hospital_id: hospital1.id },
        
        { standard: 'HIPAA Privacy Rule', category: 'Data Privacy', compliance: 91, last_assessed: new Date('2023-05-28'), hospital_id: hospital2.id },
        { standard: 'ISO 9001:2015', category: 'Quality Management', compliance: 87, last_assessed: new Date('2023-06-10'), hospital_id: hospital2.id },
        { standard: 'JCI 7th Edition', category: 'Patient Safety', compliance: 78, last_assessed: new Date('2023-06-15'), hospital_id: hospital2.id },
      ];
      
      for (const item of standardItems) {
        await prisma.standardCompliance.create({ data: item });
      }
      console.log(`Created ${standardItems.length} standard compliance items`);
      
      // Clear existing reports data
      await prisma.complianceReport.deleteMany();
      
      // Seed reports data
      const reportItems = [
        { title: 'HIPAA Quarterly Assessment - Q2 2023', type: 'compliance_summary', hospital_id: hospital1.id },
        { title: 'Staff Training Completion Report - June 2023', type: 'training_status', hospital_id: hospital1.id },
        { title: 'Medical Equipment Validation - May 2023', type: 'audit_findings', hospital_id: hospital1.id },
        
        { title: 'HIPAA Quarterly Assessment - Q1 2023', type: 'compliance_summary', hospital_id: hospital2.id },
        { title: 'Staff Training Completion Report - May 2023', type: 'training_status', hospital_id: hospital2.id },
      ];
      
      for (const item of reportItems) {
        await prisma.complianceReport.create({ data: item });
      }
      console.log(`Created ${reportItems.length} compliance reports`);
      
      // Clear existing policies data
      await prisma.compliancePolicy.deleteMany();
      
      // Seed policies data
      const policyItems = [
        { name: 'Patient Data Access', category: 'Data Privacy Policies', status: 'Active', hospital_id: hospital1.id },
        { name: 'Electronic Records Security', category: 'Data Privacy Policies', status: 'Active', hospital_id: hospital1.id },
        { name: 'Data Breach Response', category: 'Data Privacy Policies', status: 'Active', hospital_id: hospital1.id },
        
        { name: 'Quality Assurance', category: 'Quality Management Policies', status: 'Active', hospital_id: hospital1.id },
        { name: 'Risk Management', category: 'Quality Management Policies', status: 'Review Due', hospital_id: hospital1.id },
        { name: 'Corrective Actions', category: 'Quality Management Policies', status: 'Active', hospital_id: hospital1.id },
        
        { name: 'Emergency Procedures', category: 'Facility Safety Policies', status: 'Active', hospital_id: hospital1.id },
        { name: 'Infection Control', category: 'Facility Safety Policies', status: 'Review Due', hospital_id: hospital1.id },
        { name: 'Equipment Safety', category: 'Facility Safety Policies', status: 'Update Required', hospital_id: hospital1.id },
        
        { name: 'Patient Data Access', category: 'Data Privacy Policies', status: 'Active', hospital_id: hospital2.id },
        { name: 'Electronic Records Security', category: 'Data Privacy Policies', status: 'Review Due', hospital_id: hospital2.id },
        { name: 'Emergency Procedures', category: 'Facility Safety Policies', status: 'Active', hospital_id: hospital2.id },
      ];
      
      for (const item of policyItems) {
        await prisma.compliancePolicy.create({ data: item });
      }
      console.log(`Created ${policyItems.length} compliance policies`);
      
      // Clear existing findings data
      await prisma.auditFinding.deleteMany();
      
      // Seed audit findings data
      const findingItems = [
        { 
          title: 'Missing documentation for equipment maintenance',
          department: 'Facilities',
          severity: 'high' as const,
          status: 'open' as const,
          description: 'Regular maintenance records for radiology equipment are missing for the last quarter.',
          date_reported: new Date('2023-05-10'),
          hospital_id: hospital1.id 
        },
        { 
          title: 'Staff training records incomplete',
          department: 'Human Resources',
          severity: 'medium' as const,
          status: 'in_progress' as const,
          description: 'Several staff members are missing required annual compliance training documentation.',
          date_reported: new Date('2023-05-08'),
          hospital_id: hospital1.id 
        },
        { 
          title: 'Patient data access logs not maintained',
          department: 'IT Security',
          severity: 'critical' as const,
          status: 'open' as const,
          description: 'Access logs for patient records are not being properly maintained and reviewed.',
          date_reported: new Date('2023-05-05'),
          hospital_id: hospital1.id 
        },
        { 
          title: 'Medication storage temperature log inconsistent',
          department: 'Pharmacy',
          severity: 'medium' as const,
          status: 'resolved' as const,
          description: 'Temperature logs for medication storage were not consistently maintained.',
          date_reported: new Date('2023-05-03'),
          hospital_id: hospital1.id 
        },
        { 
          title: 'Emergency evacuation plan outdated',
          department: 'Safety',
          severity: 'high' as const,
          status: 'open' as const,
          description: 'The emergency evacuation plan has not been updated to reflect recent building renovations.',
          date_reported: new Date('2023-05-02'),
          hospital_id: hospital1.id 
        },
        
        { 
          title: 'Incomplete informed consent documentation',
          department: 'Legal',
          severity: 'high' as const,
          status: 'in_progress' as const,
          description: 'Several patient files have incomplete informed consent documentation.',
          date_reported: new Date('2023-05-12'),
          hospital_id: hospital2.id 
        },
        { 
          title: 'Expired supplies found in storage',
          department: 'Inventory',
          severity: 'medium' as const,
          status: 'resolved' as const,
          description: 'Multiple expired medical supplies were found during inventory audit.',
          date_reported: new Date('2023-05-01'),
          hospital_id: hospital2.id 
        },
      ];
      
      for (const item of findingItems) {
        await prisma.auditFinding.create({ data: item });
      }
      console.log(`Created ${findingItems.length} audit findings`);
    });

    // Add training events seed data
    await seedTrainingEvents();

    console.log('Database seeded!');
  } catch (error) {
    console.error('Error seeding database:', error);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}

// Add training events seed data
async function seedTrainingEvents() {
  console.log("Creating training events...");
  
  // First check if hospital exists
  const hospital = await prisma.hospital.findFirst();
  
  if (!hospital) {
    console.log("No hospital found. Please seed hospitals first.");
    return;
  }
  
  // Sample training events
  const trainingEvents = [
    {
      title: "HIPAA Compliance Workshop",
      date: new Date("2023-06-15"),
      instructor: "Dr. Sarah Johnson",
      location: "Conference Room A",
      department: "All Staff",
      duration: "3 hours",
      capacity: 30,
      enrolled: 22,
      required: true,
      hospital_id: hospital.id
    },
    {
      title: "Basic Life Support Training",
      date: new Date("2023-06-25"),
      instructor: "Dr. Michael Chen",
      location: "Simulation Lab",
      department: "Clinical Staff",
      duration: "4 hours",
      capacity: 15,
      enrolled: 12,
      required: true,
      hospital_id: hospital.id
    },
    {
      title: "Medical Records Documentation",
      date: new Date("2023-07-10"),
      instructor: "Emily Rodriguez",
      location: "Training Room B",
      department: "Administrative Staff",
      duration: "2 hours",
      capacity: 25,
      enrolled: 18,
      required: false,
      hospital_id: hospital.id
    },
    {
      title: "Patient Safety Protocols",
      date: new Date("2023-07-20"),
      instructor: "Dr. James Wilson",
      location: "Auditorium",
      department: "All Staff",
      duration: "2.5 hours",
      capacity: 50,
      enrolled: 32,
      required: true,
      hospital_id: hospital.id
    },
    {
      title: "Infection Control Procedures",
      date: new Date("2023-08-05"),
      instructor: "Dr. Lisa Taylor",
      location: "Conference Room C",
      department: "Clinical Staff",
      duration: "3 hours",
      capacity: 20,
      enrolled: 15,
      required: true,
      hospital_id: hospital.id
    }
  ];
  
  // Clear existing training events
  await prisma.trainingEvent.deleteMany({
    where: {
      hospital_id: hospital.id
    }
  });
  
  // Create new training events
  for (const event of trainingEvents) {
    await prisma.trainingEvent.create({
      data: event
    });
  }
  
  console.log(`Created ${trainingEvents.length} training events.`);
}

main()
  .then(async () => {
    await prisma.$disconnect();
  })
  .catch(async (e) => {
    console.error(e);
    await prisma.$disconnect();
    process.exit(1);
  }); 