import { getToken } from "next-auth/jwt";
import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";

export async function middleware(request: NextRequest) {
  // Get the pathname
  const path = request.nextUrl.pathname;
  console.log(`Middleware processing path: ${path}`);

  // Define public paths that don't require authentication
  const isPublicPath = path === '/login' || 
                        path === '/direct-login' ||
                        path === '/api/auth/signin' || 
                        path === '/api/auth/callback' ||
                        path === '/api/test-login';
  
  const isNextAuthPath = path.startsWith('/api/auth');
  const isTestPath = path === '/access-test';
  
  const isApiPath = path.startsWith('/api');
  const isStaticPath = path.startsWith('/_next') || 
                       path.includes('favicon.ico') ||
                       path.includes('.jpg') ||
                       path.includes('.png') ||
                       path.includes('.svg');

  const isDevelopment = process.env.NODE_ENV === 'development';
  
  // Allow access to /documents in development mode
  if (isDevelopment && path === '/documents') {
    console.log('Development mode: allowing access to /documents without authentication');
    return NextResponse.next();
  }

  // Skip middleware for NextAuth routes, static assets, API routes except auth, and test paths
  if (isNextAuthPath || isStaticPath || isTestPath || (isApiPath && !path.includes('/api/auth'))) {
    console.log(`Skipping middleware for: ${path}`);
    return NextResponse.next();
  }

  // Get the token with proper secret configuration
  try {
    const token = await getToken({ 
      req: request,
      secret: process.env.NEXTAUTH_SECRET || process.env.AUTH_SECRET
    });
    console.log(`Token for ${path}:`, token ? 'exists' : 'null');
    
    const isAuthenticated = !!token;

    // Also check for our custom cookie
    const userSession = request.cookies.get('user-session');
    const hasCustomSession = !!userSession;
    
    if (hasCustomSession) {
      console.log('Custom session found');
    }

    // Redirect authenticated users trying to access login page to dashboard
    if (isPublicPath && (isAuthenticated || hasCustomSession)) {
      console.log(`Redirecting authenticated user from ${path} to /dashboard`);
      return NextResponse.redirect(new URL('/dashboard', request.url));
    }

    // Redirect unauthenticated users to login page
    if (!isPublicPath && !isAuthenticated && !hasCustomSession) {
      console.log(`Redirecting unauthenticated user from ${path} to /login`);
      return NextResponse.redirect(new URL('/login', request.url));
    }

    console.log(`Allowing access to: ${path}`);

    // Only process API requests for hospital ID header
    if (request.nextUrl.pathname.startsWith('/api/')) {
      // Get hospital ID from request header, URL param, or cookie
      const hospitalIdFromHeader = request.headers.get('x-hospital-id');
      const hospitalIdFromParam = request.nextUrl.searchParams.get('hospitalId');
      const hospitalIdFromCookie = request.cookies.get('selectedHospitalId')?.value;
      
      // Use header first, then URL param, then cookie
      const hospitalId = hospitalIdFromHeader || hospitalIdFromParam || hospitalIdFromCookie;
      
      if (hospitalId) {
        // Create a new response with the hospital ID header
        const response = NextResponse.next();
        response.headers.set('x-hospital-id', hospitalId);
        
        // For debugging
        console.log(`Middleware: Setting hospital ID ${hospitalId} for ${request.nextUrl.pathname}`);
        
        // Return the modified response
        return response;
      }
    }

    // Redirect /documentation to /documents
    if (path === '/documentation' || path.startsWith('/documentation/')) {
      const newUrl = request.nextUrl.clone();
      newUrl.pathname = '/documents';
      return NextResponse.redirect(newUrl);
    }

    return NextResponse.next();
  } catch (error) {
    console.error(`Middleware error for ${path}:`, error);
    // If there's an error, proceed to avoid blocking the request
    return NextResponse.next();
  }
}

export const config = {
  matcher: [
    '/((?!_next/static|_next/image|favicon.ico).*)',
  ],
}; 