'use client'

import React, { createContext, useContext, useState, ReactNode, useEffect } from 'react';
import { useUser } from '@/lib/context/UserContext';
import { toast } from 'sonner';

export interface Hospital {
  id: string;
  name: string;
  location: string;
  logo_url?: string | null;
}

interface HospitalContextType {
  currentHospital: Hospital | null;
  hospitals: Hospital[];
  setCurrentHospital: (hospital: Hospital) => void;
  accessibleHospitals: Hospital[];
  getSelectedHospitalId: () => string | undefined;
  loading: boolean;
  error: string | null;
  refetchHospitals: () => Promise<void>;
}

const SELECTED_HOSPITAL_KEY = 'selectedHospitalId';

const HospitalContext = createContext<HospitalContextType | undefined>(undefined);

export const HospitalProvider = ({ children }: { children: ReactNode }) => {
  const [hospitals, setHospitals] = useState<Hospital[]>([]);
  const [currentHospital, setCurrentHospital] = useState<Hospital | null>(null);
  const [accessibleHospitals, setAccessibleHospitals] = useState<Hospital[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const userContext = useUser();

  // Function to fetch hospitals from the API
  const fetchHospitals = async () => {
    if (!userContext.isLoggedIn) {
      return;
    }

    setLoading(true);
    setError(null);

    try {
      // For super admins, fetch from the admin API
      if (userContext.isSuperAdmin) {
        const response = await fetch('/api/admin/hospitals');
        
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || 'Failed to fetch hospitals');
        }
        
        const data = await response.json();
        setHospitals(data);
        
        // If no hospitals found, show message
        if (data.length === 0) {
          setError('No hospitals found in database');
        }
      } else {
        // For regular hospital users, get their assigned hospital
        const userHospital = userContext.currentUser?.permissions.find(p => p.accessLevel === 'full')?.hospitalId;
        
        if (userHospital) {
          // Fetch the actual hospital data from the API
          try {
            console.log('Fetching hospital data from API...');
            const response = await fetch(`/api/admin/hospitals`);
            
            if (!response.ok) {
              console.error('Hospital API error:', response.status, response.statusText);
              // Don't throw immediately, try to extract more error details
              const errorText = await response.text();
              console.error('Error response:', errorText);
              throw new Error(`Hospital API error: ${response.status} ${response.statusText}`);
            }
            
            const allHospitals = await response.json();
            console.log('Fetched hospitals:', allHospitals?.length || 0);
            
            const userHospitalObj = allHospitals.find((h: any) => h.id === userHospital || h.id === parseInt(userHospital));
            
            if (userHospitalObj) {
              setHospitals([userHospitalObj]);
            } else {
              // If the hospital isn't found, use fallback method instead of throwing
              console.warn('Hospital not found in API results, trying fallback');
              throw new Error('Hospital not found in API results');
            }
          } catch (hospitalError) {
            // Handle the error gracefully and try fallback method
            console.error('Error fetching hospital data:', hospitalError);
            console.log('Trying fallback to hospital profile API');
            
            // Fallback to the hospital profile API
            try {
              console.log('Fetching from hospital profile API');
              const response = await fetch('/api/hospital/profile');
              
              if (!response.ok) {
                // Instead of throwing, create a default hospital object
                console.warn('Hospital profile API failed, using default hospital');
                
                const defaultHospital = {
                  id: '1',
                  name: 'Default Hospital',
                  location: 'System Default',
                  logo_url: null
                };
                
                setHospitals([defaultHospital]);
                setCurrentHospital(defaultHospital);
                return;
              }
              
              const data = await response.json();
              console.log('Hospital profile data retrieved');
              setHospitals([data]);
            } catch (profileError) {
              console.error('Error fetching hospital profile:', profileError);
              
              // Create a minimal hospital object to avoid breaking the UI
              const fallbackHospital = {
                id: '0',
                name: 'System Hospital',
                location: 'Default',
                logo_url: null
              };
              
              setHospitals([fallbackHospital]);
              setCurrentHospital(fallbackHospital);
              setError('Using fallback hospital data. Some features may be limited.');
            }
          }
        } else {
          // Fallback to get user's hospital information
          try {
            console.log('Regular user, fetching hospital profile');
            const response = await fetch('/api/hospital/profile');
            
            if (!response.ok) {
              console.warn('Hospital profile API failed for regular user, using default');
              
              const defaultHospital = {
                id: '1',
                name: 'Default Hospital',
                location: 'System Default',
                logo_url: null
              };
              
              setHospitals([defaultHospital]);
              setCurrentHospital(defaultHospital);
              return;
            }
            
            const data = await response.json();
            console.log('Hospital profile retrieved for regular user');
            setHospitals([data]);
          } catch (hospitalError) {
            console.error('Error fetching hospital profile for regular user:', hospitalError);
            
            // Create a minimal hospital object to avoid breaking the UI
            const fallbackHospital = {
              id: '0',
              name: 'System Hospital',
              location: 'Default',
              logo_url: null
            };
            
            setHospitals([fallbackHospital]);
            setCurrentHospital(fallbackHospital);
            setError('Using fallback hospital data. Some features may be limited.');
          }
        }
      }
    } catch (err) {
      console.error('Error fetching hospitals:', err);
      
      // Create a minimal hospital object to avoid breaking the UI
      const fallbackHospital = {
        id: '0',
        name: 'System Hospital',
        location: 'Default',
        logo_url: null
      };
      
      setHospitals([fallbackHospital]);
      setCurrentHospital(fallbackHospital);
      setError('Using fallback hospital data. Some features may be limited.');
    } finally {
      setLoading(false);
    }
  };

  // Fetch hospitals when user context changes
  useEffect(() => {
    if (userContext.isLoggedIn) {
      fetchHospitals();
    }
  }, [userContext.isLoggedIn]);

  // Helper function to get hospital ID from various sources
  const getPersistedHospitalId = (): string | null => {
    // Check localStorage first (client-side only)
    if (typeof window !== 'undefined') {
      const localStorageId = localStorage.getItem(SELECTED_HOSPITAL_KEY);
      if (localStorageId) {
        return localStorageId;
      }
    }
    
    // Check cookies as fallback
    if (typeof document !== 'undefined') {
      const cookies = document.cookie.split(';').reduce((acc, cookie) => {
        const [name, value] = cookie.trim().split('=').map(c => c.trim());
        if (name && value) acc[name] = value;
        return acc;
      }, {} as Record<string, string>);
      
      const cookieId = cookies[SELECTED_HOSPITAL_KEY];
      if (cookieId) {
        return cookieId;
      }
    }
    
    return null;
  };

  // Persist hospital ID to both localStorage and cookie
  const persistHospitalId = (hospitalId: string) => {
    if (typeof window === 'undefined') return;
    
    // Save to localStorage
    localStorage.setItem(SELECTED_HOSPITAL_KEY, hospitalId);
    
    // Save to cookie for server-side access
    document.cookie = `${SELECTED_HOSPITAL_KEY}=${hospitalId}; path=/; max-age=2592000`; // 30 days
  };

  // Initialize selected hospital from localStorage if available
  useEffect(() => {
    if (hospitals.length === 0) return;
    
    const savedHospitalId = getPersistedHospitalId();
    
    if (savedHospitalId) {
      // Try to find the hospital with this ID (as string)
      let savedHospital = hospitals.find(h => h.id === savedHospitalId);
      
      // If not found, try with numeric comparison in case of type mismatch
      if (!savedHospital) {
        const numericId = parseInt(savedHospitalId);
        if (!isNaN(numericId)) {
          savedHospital = hospitals.find(h => parseInt(h.id) === numericId);
        }
      }
      
      if (savedHospital) {
        setCurrentHospital(savedHospital);
        return;
      }
    }
    
    // For hospital users, find their assigned hospital
    if (!userContext.isSuperAdmin) {
      const userHospitalId = userContext.currentUser?.permissions.find(p => p.accessLevel === 'full')?.hospitalId;
      if (userHospitalId) {
        const userHospital = hospitals.find(h => h.id === userHospitalId || h.id === String(userHospitalId));
        if (userHospital) {
          setCurrentHospital(userHospital);
          persistHospitalId(userHospital.id);
          return;
        }
      }
    }
    
    // Default to first hospital if no saved hospital or saved hospital not found
    if (hospitals.length > 0) {
      setCurrentHospital(hospitals[0]);
      
      // Persist the default selection
      persistHospitalId(hospitals[0].id);
    }
  }, [hospitals, userContext.currentUser]);

  // Filter hospitals based on user permissions
  useEffect(() => {
    if (!hospitals.length) return;
    
    if (userContext.isSuperAdmin) {
      // Superadmin has access to all hospitals
      setAccessibleHospitals(hospitals);
    } else {
      const accessibleHospitalIds = userContext.getAccessibleHospitals();
      const filteredHospitals = hospitals.filter(hospital => 
        accessibleHospitalIds.includes(hospital.id)
      );
      setAccessibleHospitals(filteredHospitals);
      
      // If current hospital is no longer accessible, switch to the first accessible one
      if (currentHospital && filteredHospitals.length > 0 && !userContext.hasAccessToHospital(currentHospital.id)) {
        setCurrentHospital(filteredHospitals[0]);
      }
    }
  }, [userContext, hospitals, currentHospital?.id]);
  
  // Custom hospital setter that also updates localStorage
  const handleSetCurrentHospital = (hospital: Hospital) => {
    setCurrentHospital(hospital);
    
    // Ensure hospital ID is stored as a string
    const hospitalId = String(hospital.id);
    
    // Persist the selection
    persistHospitalId(hospitalId);
    
    // Show toast notification
    toast.success(`Hospital Changed: ${hospital.name}`, {
      description: 'All dashboard data now shows information for this hospital.',
      duration: 3000,
    });
    
    // Refresh the page to reload data for the new hospital
    // Use a slight delay to allow the toast to be seen
    setTimeout(() => {
      if (typeof window !== 'undefined') {
        window.location.reload();
      }
    }, 500);
  };

  // Helper function to get the selected hospital ID for API calls
  const getSelectedHospitalId = (): string | undefined => {
    if (userContext.isSuperAdmin) {
      // Super admins need to explicitly select a hospital
      return currentHospital?.id;
    } else {
      // For regular hospital users, use their assigned hospital from permissions
      const hospitalId = userContext.currentUser?.permissions.find(p => p.accessLevel === 'full')?.hospitalId;
      
      // If there's no assigned hospital with full access, use the current selected one
      return hospitalId || currentHospital?.id;
    }
  };

  return (
    <HospitalContext.Provider
      value={{
        currentHospital,
        hospitals,
        setCurrentHospital: handleSetCurrentHospital,
        accessibleHospitals,
        getSelectedHospitalId,
        loading,
        error,
        refetchHospitals: fetchHospitals
      }}
    >
      {children}
    </HospitalContext.Provider>
  );
};

export const useHospital = () => {
  const context = useContext(HospitalContext);
  if (context === undefined) {
    throw new Error('useHospital must be used within a HospitalProvider');
  }
  return context;
}; 