import { compare } from "bcrypt";
import NextAuth from "next-auth";
import CredentialsProvider from "next-auth/providers/credentials";
import { db } from "./db";

// Extend the default session and JWT types
declare module "next-auth" {
  interface User {
    id: string;
    email: string;
    name: string;
    role: string;
    hospitalId: string | null;
    hospitalName?: string;
  }
  
  interface Session {
    user: {
      id: string;
      name: string;
      email: string;
      role: string;
      hospitalId: string | null;
      hospitalName?: string;
    };
  }
}

declare module "next-auth/jwt" {
  interface JWT {
    id: string;
    role: string;
    hospitalId: string | null;
    hospitalName?: string;
  }
}

const authConfig = {
  session: {
    strategy: "jwt" as const,
    maxAge: 30 * 24 * 60 * 60, // 30 days
  },
  pages: {
    signIn: "/login",
    signOut: "/login",
    error: "/login",
  },
  secret: process.env.NEXTAUTH_SECRET || "your-default-secret-do-not-use-in-production",
  debug: true, // Enable debug mode in development
  providers: [
    CredentialsProvider({
      name: "credentials",
      credentials: {
        email: { label: "Email", type: "email" },
        password: { label: "Password", type: "password" },
      },
      async authorize(credentials) {
        console.log("Authorizing with credentials:", credentials?.email);

        if (!credentials?.email || !credentials?.password) {
          console.log("Missing credentials");
          return null;
        }

        try {
          const user = await db.user.findUnique({
            where: {
              email: credentials.email as string,
            },
            include: {
              hospital: true,
            },
          });

          console.log("User found:", user ? "Yes" : "No");

          if (!user) {
            return null;
          }

          const isPasswordValid = await compare(credentials.password as string, user.password);
          console.log("Password valid:", isPasswordValid);

          if (!isPasswordValid) {
            return null;
          }

          const authUser = {
            id: String(user.id),
            email: user.email,
            name: user.username,
            role: user.role,
            hospitalId: user.hospital_id ? String(user.hospital_id) : null,
            hospitalName: user.hospital?.name,
          };

          console.log("Authentication successful for user:", authUser.email);
          return authUser;
        } catch (error) {
          console.error("Auth error:", error);
          return null;
        }
      },
    }),
  ],
  callbacks: {
    async jwt({ token, user }: any) {
      if (user) {
        console.log("JWT callback with user:", user.email);
        token.id = user.id;
        token.role = user.role;
        token.hospitalId = user.hospitalId;
        token.hospitalName = user.hospitalName;
      }
      return token;
    },
    async session({ session, token }: any) {
      console.log("Session callback for user:", token.email);
      if (session.user && token) {
        session.user.id = token.id;
        session.user.role = token.role;
        session.user.hospitalId = token.hospitalId;
        session.user.hospitalName = token.hospitalName;
      }
      return session;
    },
  },
};

export const { handlers, auth, signIn, signOut } = NextAuth(authConfig);

// Keep the legacy export for backward compatibility if needed
export const authOptions = authConfig; 