import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';

interface ApiRequestContext {
  req: NextRequest | Request;
  session: any;
  user: any;
  isSuperAdmin: boolean;
  hospitalId: number | null;
}

/**
 * Gets the hospital ID from various sources (header, URL, user record)
 * Ensures consistent hospital ID handling across API routes
 */
export async function getApiContext(req: NextRequest | Request | any): Promise<ApiRequestContext> {
  // Get the session
  const session = await auth();
  
  if (!session || !session.user?.email) {
    throw new Error('Unauthorized');
  }
  
  // Get the user from the database
  const user = await prisma.user.findUnique({
    where: { email: session.user.email as string },
    select: { id: true, username: true, email: true, role: true, hospital_id: true }
  });
  
  if (!user) {
    throw new Error('User not found');
  }
  
  const isSuperAdmin = user.role === 'super_admin';
  
  // Get the hospital ID
  let hospitalId: number | null = null;
  
  if (isSuperAdmin) {
    // For super admins, try to get hospital ID from header, URL, or body
    let headerHospitalId: string | null = null;
    let urlHospitalId: string | null = null;
    
    // Handle case where req might be empty or undefined (for GET requests without a request object)
    if (!req || typeof req !== 'object') {
      console.log("WARNING: Request object is empty or undefined. Using cookies from server request.");
      // Continue without trying to access req - we'll use cookies instead
    } 
    // If it's a proper Request object
    else if (req instanceof Request) {
      const headers = new Headers(req.headers);
      headerHospitalId = headers.get('x-hospital-id');
      
      // Get hospital ID from URL query parameter
      const url = new URL(req.url);
      urlHospitalId = url.searchParams.get('hospitalId');
    } 
    // If it's a NextRequest
    else if (req.headers && typeof req.headers.get === 'function') {
      headerHospitalId = req.headers.get('x-hospital-id');
      
      // Get hospital ID from URL query parameter if nextUrl exists
      if (req.nextUrl && typeof req.nextUrl.searchParams?.get === 'function') {
        urlHospitalId = req.nextUrl.searchParams.get('hospitalId');
      }
    }
    
    // Get hospital ID from cookies
    let cookieHospitalId: string | null = null;
    
    // Try to get from cookies in the request
    if (req instanceof Request) {
      // Parse cookies from Request header
      const cookieHeader = new Headers(req.headers).get('cookie');
      if (cookieHeader) {
        const cookies = parseCookieString(cookieHeader);
        cookieHospitalId = cookies.selectedHospitalId || null;
      }
    } 
    // If it has a cookies property with a get method
    else if (req && req.cookies && typeof req.cookies.get === 'function') {
      cookieHospitalId = req.cookies.get('selectedHospitalId')?.value || null;
    }
    
    // If we couldn't get from request, try the server request object
    if (!cookieHospitalId) {
      // Try to get from global request context
      const { cookies } = require('next/headers');
      try {
        const cookieStore = cookies();
        if (cookieStore && typeof cookieStore.get === 'function') {
          const cookie = cookieStore.get('selectedHospitalId');
          cookieHospitalId = cookie?.value || null;
        }
      } catch (e) {
        console.error('Error accessing server cookies:', e);
      }
    }
    
    // Try to get hospital ID from body if it's a POST or PUT request
    let bodyHospitalId: string | null = null;
    if (req instanceof Request && req.method && ['POST', 'PUT'].includes(req.method)) {
      try {
        const clone = req.clone(); // Clone the request to avoid consuming the body
        const body = await clone.json();
        bodyHospitalId = body.hospitalId?.toString() || null;
      } catch (e) {
        // Ignore error if body can't be parsed
      }
    }
    
    // Use the first available hospital ID
    const hospitalIdStr = headerHospitalId || urlHospitalId || cookieHospitalId || bodyHospitalId;
    
    // Debug log all sources
    console.log('Hospital ID sources:', {
      header: headerHospitalId,
      url: urlHospitalId,
      cookie: cookieHospitalId,
      body: bodyHospitalId,
      selected: hospitalIdStr
    });
    
    if (hospitalIdStr) {
      const parsed = parseInt(hospitalIdStr);
      if (!isNaN(parsed)) {
        hospitalId = parsed;
      }
    }
  } else {
      // For regular users, always use their assigned hospital
  if (user.hospital_id) {
    hospitalId = user.hospital_id;
    console.log(`Using assigned hospital ID for regular user: ${hospitalId}`);
  } else {
    console.log('Warning: User has no assigned hospital_id');
    throw new Error('User has no assigned hospital. Please contact an administrator.');
  }
  }
  
  return { req, session, user, isSuperAdmin, hospitalId };
}

/**
 * Helper function to parse cookies from a cookie header string
 */
function parseCookieString(cookieHeader: string): Record<string, string> {
  return cookieHeader.split(';').reduce((cookies, cookie) => {
    const [name, value] = cookie.trim().split('=').map(c => c.trim());
    if (name && value) cookies[name] = value;
    return cookies;
  }, {} as Record<string, string>);
}

/**
 * Helper function to verify if a user has access to a specific hospital
 */
export async function hasHospitalAccess(userEmail: string, hospitalId: number): Promise<boolean> {
  const user = await prisma.user.findUnique({
    where: { email: userEmail },
    select: { role: true, hospital_id: true }
  });
  
  if (!user) return false;
  
  // Super admins have access to all hospitals
  if (user.role === 'super_admin') return true;
  
  // Regular users only have access to their assigned hospital
  return user.hospital_id === hospitalId;
}

/**
 * Standard error handler for API routes
 */
export function handleApiError(error: any, statusCode = 500): NextResponse {
  console.error('API error:', error);
  
  return NextResponse.json({ 
    error: error instanceof Error ? error.message : 'An unexpected error occurred',
    details: error instanceof Error ? error.stack : String(error)
  }, { status: statusCode });
}

export interface ApiResponse<T = any> {
  data?: T;
  error?: string;
  message?: string;
  success: boolean;
}

export class ApiError extends Error {
  constructor(public message: string, public status: number = 500) {
    super(message);
    this.name = 'ApiError';
  }
}

export async function getAuthenticatedSession() {
  try {
    const session = await auth();
    
    if (!session?.user) {
      throw new ApiError('Unauthorized access', 401);
    }
    
    return session;
  } catch (error) {
    console.error('Session error:', error);
    throw new ApiError('Authentication failed', 401);
  }
}

export function createApiResponse<T>(data: T, message?: string): ApiResponse<T> {
  return {
    data,
    message,
    success: true,
  };
}

export function createErrorResponse(error: string, status: number = 500): NextResponse {
  return NextResponse.json(
    {
      error,
      success: false,
    },
    { status }
  );
}

export async function withAuth<T>(
  handler: (session: any) => Promise<T>
): Promise<NextResponse> {
  try {
    const session = await getAuthenticatedSession();
    const result = await handler(session);
    return NextResponse.json(createApiResponse(result));
  } catch (error) {
    if (error instanceof ApiError) {
      return createErrorResponse(error.message, error.status);
    }
    console.error('Unexpected error:', error);
    return createErrorResponse('Internal server error', 500);
  }
} 