import { useState, useEffect } from 'react'
import { toast } from 'sonner'

export interface AuditSchedule {
  id: string
  title: string
  date: string
  department: string
  auditor: string
  status: string
}

export interface TrainingEvent {
  id: string
  title: string
  date: string
  instructor: string
  location: string
  department: string
  duration: string
  capacity: number
  enrolled: number
  required: boolean
}

export interface RegulatoryDeadline {
  id: string
  title: string
  dueDate: string
  authority: string
  criticalLevel: string
  status: string
  assignedTo: string
  description: string
}

export interface ResourceAllocation {
  id: string
  resourceName: string
  role: string
  allocatedTo: string
  startDate: string
  endDate: string
  utilizationPercentage: number
  department: string
}

interface ScheduleState {
  auditSchedules: AuditSchedule[]
  trainingEvents: TrainingEvent[]
  regulatoryDeadlines: RegulatoryDeadline[]
  resourceAllocations: ResourceAllocation[]
  loading: {
    audits: boolean
    training: boolean
    deadlines: boolean
    resources: boolean
  }
  error: {
    audits: string | null
    training: string | null
    deadlines: string | null
    resources: string | null
  }
}

const initialState: ScheduleState = {
  auditSchedules: [],
  trainingEvents: [],
  regulatoryDeadlines: [],
  resourceAllocations: [],
  loading: {
    audits: false,
    training: false,
    deadlines: false,
    resources: false,
  },
  error: {
    audits: null,
    training: null,
    deadlines: null,
    resources: null,
  },
}

export const useScheduleData = () => {
  const [state, setState] = useState<ScheduleState>(initialState)

  // Fetch all schedule data
  useEffect(() => {
    fetchAuditSchedules()
    fetchTrainingEvents()
    fetchRegulatoryDeadlines()
    fetchResourceAllocations()
  }, [])
  
  // Fetch audit schedules
  const fetchAuditSchedules = async () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, audits: true },
      error: { ...prev.error, audits: null },
    }))

    let retryCount = 0;
    const maxRetries = 3;
    const retryDelay = 1000; // 1 second

    const attemptFetch = async (): Promise<boolean> => {
    try {
        const response = await fetch('/api/schedule/audits');
      
      if (!response.ok) {
          if (response.status === 401) {
            throw new Error('Authentication error - please log in again');
          } else if (response.status === 404) {
            throw new Error('No hospital associated with your account');
          } else if (response.status === 400) {
            const errorData = await response.json().catch(() => ({}));
            // If this is the hospital selection error, display a user-friendly message
            if (errorData.error === 'Hospital selection required') {
              throw new Error('Please select a hospital to view audit schedules');
            } else {
              throw new Error(errorData.message || 'Failed to fetch audit schedules');
            }
          } else {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.message || 'Failed to fetch audit schedules');
          }
      }
      
        const data = await response.json();
      
      // Format dates for display
      const formattedData = data.map((item: any) => ({
        ...item,
        date: item.date ? new Date(item.date).toISOString().split('T')[0] : '',
        }));
      
      setState(prev => ({
        ...prev,
        auditSchedules: formattedData,
        loading: { ...prev.loading, audits: false },
        }));
        return true;
    } catch (error) {
        console.error('Error fetching audit schedules:', error);
        
        // Check if we should retry
        if (retryCount < maxRetries && (error instanceof Error && error.message.includes('fetch'))) {
          retryCount++;
          console.log(`Retrying fetch attempt ${retryCount}/${maxRetries}...`);
          await new Promise(resolve => setTimeout(resolve, retryDelay));
          return attemptFetch();
        }
        
      setState(prev => ({
        ...prev,
        loading: { ...prev.loading, audits: false },
        error: { ...prev.error, audits: error instanceof Error ? error.message : 'Unknown error' },
        }));
        return false;
    }
    };

    return attemptFetch();
  }

  // Fetch training events
  const fetchTrainingEvents = async () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, training: true },
      error: { ...prev.error, training: null },
    }))

    try {
      const response = await fetch('/api/schedule/training')
      
      if (!response.ok) {
        throw new Error('Failed to fetch training events')
      }
      
      const data = await response.json()
      
      // Format dates for display
      const formattedData = data.map((item: any) => ({
        ...item,
        date: item.date ? new Date(item.date).toISOString().split('T')[0] : '',
      }))
      
      setState(prev => ({
        ...prev,
        trainingEvents: formattedData,
        loading: { ...prev.loading, training: false },
      }))
    } catch (error) {
      console.error('Error fetching training events:', error)
      setState(prev => ({
        ...prev,
        loading: { ...prev.loading, training: false },
        error: { ...prev.error, training: error instanceof Error ? error.message : 'Unknown error' },
      }))
    }
  }

  // Fetch regulatory deadlines
  const fetchRegulatoryDeadlines = async () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, deadlines: true },
      error: { ...prev.error, deadlines: null },
    }))

    try {
      const response = await fetch('/api/schedule/deadlines')
      
      if (!response.ok) {
        throw new Error('Failed to fetch regulatory deadlines')
      }
      
      const data = await response.json()
      
      // Format dates for display
      const formattedData = data.map((item: any) => ({
        ...item,
        dueDate: item.dueDate ? new Date(item.dueDate).toISOString().split('T')[0] : '',
      }))
      
      setState(prev => ({
        ...prev,
        regulatoryDeadlines: formattedData,
        loading: { ...prev.loading, deadlines: false },
      }))
    } catch (error) {
      console.error('Error fetching regulatory deadlines:', error)
      setState(prev => ({
        ...prev,
        loading: { ...prev.loading, deadlines: false },
        error: { ...prev.error, deadlines: error instanceof Error ? error.message : 'Unknown error' },
      }))
    }
  }

  // Fetch resource allocations
  const fetchResourceAllocations = async () => {
    setState(prev => ({
      ...prev,
      loading: { ...prev.loading, resources: true },
      error: { ...prev.error, resources: null },
    }))

    try {
      const response = await fetch('/api/schedule/resources')
      
      if (!response.ok) {
        throw new Error('Failed to fetch resource allocations')
      }
      
      const data = await response.json()
      
      // Format dates for display
      const formattedData = data.map((item: any) => ({
        ...item,
        startDate: item.startDate ? new Date(item.startDate).toISOString().split('T')[0] : '',
        endDate: item.endDate ? new Date(item.endDate).toISOString().split('T')[0] : '',
      }))
      
      setState(prev => ({
        ...prev,
        resourceAllocations: formattedData,
        loading: { ...prev.loading, resources: false },
      }))
    } catch (error) {
      console.error('Error fetching resource allocations:', error)
      setState(prev => ({
        ...prev,
        loading: { ...prev.loading, resources: false },
        error: { ...prev.error, resources: error instanceof Error ? error.message : 'Unknown error' },
      }))
    }
  }

  // Create a new audit schedule
  const createAuditSchedule = async (audit: Omit<AuditSchedule, 'id'>): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/audits', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(audit),
      })

      if (!response.ok) {
        throw new Error('Failed to create audit schedule')
      }

      // Refresh audit schedules after creating
      await fetchAuditSchedules()
      toast.success('Audit schedule created successfully')
      return true
    } catch (error) {
      console.error('Error creating audit schedule:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create audit schedule')
      return false
    }
  }

  // Update an existing audit schedule
  const updateAuditSchedule = async (audit: AuditSchedule): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/audits', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(audit),
      })

      if (!response.ok) {
        throw new Error('Failed to update audit schedule')
      }

      // Refresh audit schedules after updating
      await fetchAuditSchedules()
      toast.success('Audit schedule updated successfully')
      return true
    } catch (error) {
      console.error('Error updating audit schedule:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update audit schedule')
      return false
    }
  }

  // Delete an audit schedule
  const deleteAuditSchedule = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/schedule/audits?id=${id}`, {
        method: 'DELETE',
      })

      if (!response.ok) {
        throw new Error('Failed to delete audit schedule')
      }

      // Refresh audit schedules after deleting
      await fetchAuditSchedules()
      toast.success('Audit schedule deleted successfully')
      return true
    } catch (error) {
      console.error('Error deleting audit schedule:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete audit schedule')
      return false
    }
  }

  // Create a new training event
  const createTrainingEvent = async (training: Omit<TrainingEvent, 'id'>): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/training', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(training),
      })

      if (!response.ok) {
        throw new Error('Failed to create training event')
      }

      // Refresh training events after creating
      await fetchTrainingEvents()
      toast.success('Training event created successfully')
      return true
    } catch (error) {
      console.error('Error creating training event:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create training event')
      return false
    }
  }

  // Update an existing training event
  const updateTrainingEvent = async (training: TrainingEvent): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/training', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(training),
      })

      if (!response.ok) {
        throw new Error('Failed to update training event')
      }

      // Refresh training events after updating
      await fetchTrainingEvents()
      toast.success('Training event updated successfully')
      return true
    } catch (error) {
      console.error('Error updating training event:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update training event')
      return false
    }
  }

  // Delete a training event
  const deleteTrainingEvent = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/schedule/training?id=${id}`, {
        method: 'DELETE',
      })

      if (!response.ok) {
        throw new Error('Failed to delete training event')
      }

      // Refresh training events after deleting
      await fetchTrainingEvents()
      toast.success('Training event deleted successfully')
      return true
    } catch (error) {
      console.error('Error deleting training event:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete training event')
      return false
    }
  }

  // Create a new regulatory deadline
  const createRegulatoryDeadline = async (deadline: Omit<RegulatoryDeadline, 'id'>): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/deadlines', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(deadline),
      })

      if (!response.ok) {
        throw new Error('Failed to create regulatory deadline')
      }

      // Refresh regulatory deadlines after creating
      await fetchRegulatoryDeadlines()
      toast.success('Regulatory deadline created successfully')
      return true
    } catch (error) {
      console.error('Error creating regulatory deadline:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create regulatory deadline')
      return false
    }
  }

  // Update an existing regulatory deadline
  const updateRegulatoryDeadline = async (deadline: RegulatoryDeadline): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/deadlines', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(deadline),
      })

      if (!response.ok) {
        throw new Error('Failed to update regulatory deadline')
      }

      // Refresh regulatory deadlines after updating
      await fetchRegulatoryDeadlines()
      toast.success('Regulatory deadline updated successfully')
      return true
    } catch (error) {
      console.error('Error updating regulatory deadline:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update regulatory deadline')
      return false
    }
  }

  // Delete a regulatory deadline
  const deleteRegulatoryDeadline = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/schedule/deadlines?id=${id}`, {
        method: 'DELETE',
      })

      if (!response.ok) {
        throw new Error('Failed to delete regulatory deadline')
      }

      // Refresh regulatory deadlines after deleting
      await fetchRegulatoryDeadlines()
      toast.success('Regulatory deadline deleted successfully')
      return true
    } catch (error) {
      console.error('Error deleting regulatory deadline:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete regulatory deadline')
      return false
    }
  }

  // Create a new resource allocation
  const createResourceAllocation = async (resource: Omit<ResourceAllocation, 'id'>): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/resources', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(resource),
      })

      if (!response.ok) {
        throw new Error('Failed to create resource allocation')
      }

      // Refresh resource allocations after creating
      await fetchResourceAllocations()
      toast.success('Resource allocation created successfully')
      return true
    } catch (error) {
      console.error('Error creating resource allocation:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create resource allocation')
      return false
    }
  }

  // Update an existing resource allocation
  const updateResourceAllocation = async (resource: ResourceAllocation): Promise<boolean> => {
    try {
      const response = await fetch('/api/schedule/resources', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(resource),
      })

      if (!response.ok) {
        throw new Error('Failed to update resource allocation')
      }

      // Refresh resource allocations after updating
      await fetchResourceAllocations()
      toast.success('Resource allocation updated successfully')
      return true
    } catch (error) {
      console.error('Error updating resource allocation:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update resource allocation')
      return false
    }
  }

  // Delete a resource allocation
  const deleteResourceAllocation = async (id: string): Promise<boolean> => {
    try {
      const response = await fetch(`/api/schedule/resources?id=${id}`, {
        method: 'DELETE',
      })

      if (!response.ok) {
        throw new Error('Failed to delete resource allocation')
      }

      // Refresh resource allocations after deleting
      await fetchResourceAllocations()
      toast.success('Resource allocation deleted successfully')
      return true
    } catch (error) {
      console.error('Error deleting resource allocation:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete resource allocation')
      return false
    }
  }

  // Refresh all schedule data
  const refreshAll = async () => {
    await Promise.all([
      fetchAuditSchedules(),
      fetchTrainingEvents(),
      fetchRegulatoryDeadlines(),
      fetchResourceAllocations()
    ])
  }

  return {
    ...state,
    // CRUD operations
    createAuditSchedule,
    updateAuditSchedule,
    deleteAuditSchedule,
    createTrainingEvent,
    updateTrainingEvent,
    deleteTrainingEvent,
    createRegulatoryDeadline,
    updateRegulatoryDeadline,
    deleteRegulatoryDeadline,
    createResourceAllocation,
    updateResourceAllocation,
    deleteResourceAllocation,
    // Refresh methods
    refreshAuditSchedules: fetchAuditSchedules,
    refreshTrainingEvents: fetchTrainingEvents,
    refreshRegulatoryDeadlines: fetchRegulatoryDeadlines,
    refreshResourceAllocations: fetchResourceAllocations,
    refreshAll
  }
} 