import { useState, useEffect, useCallback } from 'react';
import { useApiWithHospital } from '@/hooks/useApiWithHospital';

export interface Report {
  id: number;
  title: string;
  type: string;
  file_url?: string | null;
  hospital_id: number;
  created_at: string;
  author?: string; // Derived value for display
  date?: string; // Formatted date for display
}

export interface ReportType {
  value: string;
  label: string;
  count: number;
}

export interface ScheduledReport {
  id: string;
  title: string;
  frequency: string;
  nextRun: string;
  format: string;
  recipients: number;
  status: string;
}

export interface ReportSummary {
  totalReports: number;
  byType: { [key: string]: number };
  recent: Report[];
  complianceMetrics: {
    hipaa: number;
    iso: number;
    jci: number;
  };
  riskAssessment: Array<{
    area: string;
    priority: string;
    risk: 'high' | 'medium' | 'low';
    details: string;
  }>;
}

export const useReportsData = () => {
  const [recentReports, setRecentReports] = useState<Report[]>([]);
  const [reportTypes, setReportTypes] = useState<ReportType[]>([]);
  const [summaryData, setSummaryData] = useState<ReportSummary | null>(null);
  const [scheduledReports, setScheduledReports] = useState<ScheduledReport[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const { fetchWithHospital } = useApiWithHospital();

  const fetchReports = useCallback(async () => {
    setLoading(true);
    try {
      // Try to fetch detailed report data from API
      let data;
      try {
        data = await fetchWithHospital<any>('/api/compliance/reports?detailed=true');
        console.log('Fetched report data:', data);
      } catch (fetchError) {
        console.error('Error fetching detailed reports:', fetchError);
        // Try a simpler request as fallback
        try {
          data = await fetchWithHospital<any>('/api/compliance/reports');
          console.log('Fetched basic report data as fallback:', data);
        } catch (basicFetchError) {
          console.error('Error fetching basic reports:', basicFetchError);
          throw new Error('Failed to fetch any report data');
        }
      }
      
      // Ensure data is an array (for basic endpoint) or has reports property (for detailed endpoint)
      const reportData = Array.isArray(data) ? data : (data?.reports || []);
      
      if (!reportData || reportData.length === 0) {
        console.warn('No report data returned from API');
      }
      
      // Process the raw data
      const processedReports = reportData.map((report: Report) => {
        // Check if this is a pharmacy report and assign a demo file URL if needed
        let reportWithUrl = {
          ...report,
          // Format created_at as a readable date
          date: new Date(report.created_at).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
          }),
          // Set default author based on report type
          author: getAuthorByReportType(report.type)
        };
        
        // Special handling for pharmacy reports - add a sample file URL
        if (
          (report.title && report.title.toLowerCase().includes('pharmacy')) || 
          (report.type && report.type.toLowerCase().includes('pharmacy'))
        ) {
          // If it's a pharmacy report but doesn't have a file URL, add one
          if (!report.file_url) {
            const sampleFiles = [
              'sample-reports/pharmacy_inventory.txt',
              'sample-reports/pharmacy_compliance.txt',
              'sample-reports/medication_safety.txt',
              'sample-reports/pharmacy_report.html'
            ];
            
            // Assign a consistent file for the same report ID
            const fileIndex = (report.id % sampleFiles.length) || 0;
            reportWithUrl.file_url = sampleFiles[fileIndex];
          }
        }
        
        return reportWithUrl;
      });
      
      setRecentReports(processedReports);
      
      // If we have detailed data from the API
      if (data && data.reportTypes) {
        // Use the report types from the API
        const typesList = Object.entries(data.reportTypes).map(([type, count]) => ({
          value: type,
          label: formatReportType(type),
          count: Number(count)
        }));
        
        setReportTypes(typesList);
        
        // Use API risk assessment and metrics if available
        if (data.riskAssessment && data.complianceMetrics) {
          const apiSummaryData: ReportSummary = {
            totalReports: data.totalReports || processedReports.length,
            byType: data.reportTypes,
            recent: processedReports.slice(0, 5),
            complianceMetrics: data.complianceMetrics,
            riskAssessment: data.riskAssessment
          };
          
          setSummaryData(apiSummaryData);
        } else {
          // Generate fallback summary data if needed
          generateFallbackSummaryData(processedReports);
        }
      } else {
        // Generate types from the data if not provided by API
        const types: Record<string, number> = {};
        processedReports.forEach((report: Report) => {
          const type = report.type;
          if (types[type]) {
            types[type]++;
          } else {
            types[type] = 1;
          }
        });
        
        const typesList = Object.entries(types).map(([type, count]) => ({
          value: type,
          label: formatReportType(type),
          count
        }));
        
        setReportTypes(typesList);
        
        // Generate fallback summary data
        generateFallbackSummaryData(processedReports);
      }
      
      // Generate scheduled reports data
      // In a real implementation, this would come from the API
      generateScheduledReportsData();
      
      setError(null);
    } catch (err) {
      console.error('Error in useReportsData hook:', err);
      setError(err instanceof Error ? err.message : 'Failed to load report data. Please try again later.');
      
      // Set fallback data in case of error
      const fallbackReports = [
        {
          id: 1,
          title: "HIPAA Compliance Q2 2023",
          date: "Jun 30, 2023",
          author: "Dr. Sarah Williams",
          type: "compliance_summary",
          created_at: new Date().toISOString(),
          hospital_id: 1
        },
        {
          id: 2,
          title: "ISO 9001 Audit Findings",
          date: "Jun 15, 2023",
          author: "Quality Management Team",
          type: "audit_findings",
          created_at: new Date().toISOString(),
          hospital_id: 1
        }
      ];
      
      setRecentReports(fallbackReports);
      
      // Generate fallback types from our static reports
      setReportTypes([
        { value: "compliance_summary", label: "Compliance Summary Reports", count: 1 },
        { value: "audit_findings", label: "Audit Findings Reports", count: 1 }
      ]);
      
      // Generate fallback summary data
      generateFallbackSummaryData(fallbackReports);
      
      // Generate scheduled reports fallback data
      generateScheduledReportsData();
    } finally {
      setLoading(false);
    }
  }, [fetchWithHospital]);

  // Function to manually refresh data
  const refreshData = useCallback(() => {
    setLoading(true);
    setError(null);
    
    fetchReports()
      .catch(err => {
        console.error('Error refreshing report data:', err);
        setError(err instanceof Error ? err.message : 'Failed to refresh data');
      })
      .finally(() => {
        // Loading will be set to false by fetchReports, but add this as a safety
        setTimeout(() => {
          if (loading) {
            setLoading(false);
          }
        }, 500);
      });
  }, [fetchReports, loading]);

  // Function to delete a report
  const deleteReport = useCallback(async (reportId: number) => {
    try {
      const response = await fetchWithHospital<{ success: boolean; message: string }>(`/api/compliance/reports?id=${reportId}`, {
        method: 'DELETE',
      });
      
      if (response.success) {
        // Update the reports list by removing the deleted report
        setRecentReports(prev => prev.filter(report => report.id !== reportId));
        
        // Update summary data if available
        if (summaryData) {
          setSummaryData(prev => {
            if (!prev) return null;
            
            return {
              ...prev,
              totalReports: prev.totalReports - 1,
            };
          });
        }
        
        return true;
      }
      
      return false;
    } catch (error) {
      console.error('Error deleting report:', error);
      return false;
    }
  }, [fetchWithHospital, summaryData]);

  useEffect(() => {
    fetchReports();
  }, [fetchReports]);

  // Generate scheduled reports data (would be API-based in production)
  const generateScheduledReportsData = () => {
    const today = new Date();
    
    // Calculate first day of next month
    const nextMonth = new Date(today);
    nextMonth.setMonth(nextMonth.getMonth() + 1);
    nextMonth.setDate(1);
    
    // Calculate last day of current quarter
    const currentQuarter = Math.floor(today.getMonth() / 3);
    const lastDayOfQuarter = new Date(today.getFullYear(), (currentQuarter + 1) * 3, 0);
    
    // Calculate next Friday
    const nextFriday = new Date(today);
    nextFriday.setDate(today.getDate() + ((7 - today.getDay() + 5) % 7 || 7));
    
    setScheduledReports([
      {
        id: '1',
        title: 'Monthly HIPAA Compliance',
        frequency: 'Monthly (1st)',
        nextRun: nextMonth.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
        format: 'PDF',
        recipients: 5,
        status: 'active'
      },
      {
        id: '2',
        title: 'Quarterly Regulatory Review',
        frequency: 'Quarterly',
        nextRun: lastDayOfQuarter.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
        format: 'Excel',
        recipients: 8,
        status: 'active'
      },
      {
        id: '3',
        title: 'Weekly Staff Training Progress',
        frequency: 'Weekly (Friday)',
        nextRun: nextFriday.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }),
        format: 'PDF',
        recipients: 3,
        status: 'active'
      }
    ]);
  };

  // Generate fallback summary data when API doesn't provide it
  const generateFallbackSummaryData = (reports: Report[]) => {
    // Check if we have any pharmacy reports, if not, add some
    const hasPharmacyReports = reports.some(
      report => 
        (report.title && report.title.toLowerCase().includes('pharmacy')) ||
        (report.type && report.type.toLowerCase().includes('pharmacy'))
    );
    
    if (!hasPharmacyReports) {
      // Add some pharmacy reports to the dataset
      const pharmacyReports: Report[] = [
        {
          id: Math.floor(Math.random() * 1000) + 500,
          title: "Pharmacy Inventory Audit Q2",
          type: "pharmacy_inventory",
          file_url: "sample-reports/pharmacy_inventory.txt",
          hospital_id: 1,
          created_at: new Date().toISOString(),
          author: "Pharmacy Department",
          date: new Date().toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
          })
        },
        {
          id: Math.floor(Math.random() * 1000) + 501,
          title: "Medication Safety Compliance",
          type: "pharmacy_compliance",
          file_url: "sample-reports/pharmacy_compliance.txt",
          hospital_id: 1,
          created_at: new Date().toISOString(),
          author: "Pharmacy Department",
          date: new Date().toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
          })
        },
        {
          id: Math.floor(Math.random() * 1000) + 502,
          title: "Controlled Substances Review",
          type: "pharmacy_safety",
          file_url: "sample-reports/medication_safety.txt",
          hospital_id: 1,
          created_at: new Date().toISOString(),
          author: "Pharmacy Department",
          date: new Date().toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
          })
        }
      ];
      
      // Add pharmacy reports to recentReports
      setRecentReports([...reports, ...pharmacyReports]);
      
      // Update reports list for summary data
      reports = [...reports, ...pharmacyReports];
    }
    
    const summaryData: ReportSummary = {
      totalReports: reports.length,
      byType: {},
      recent: reports.slice(0, 5),
      complianceMetrics: {
        hipaa: Math.floor(Math.random() * 15) + 80, // Simulated values 80-95%
        iso: Math.floor(Math.random() * 20) + 70, // Simulated values 70-90%
        jci: Math.floor(Math.random() * 25) + 65, // Simulated values 65-90%
      },
      riskAssessment: [
        {
          area: "Patient Data Security",
          priority: "High priority",
          risk: "high",
          details: "Data encryption and access controls need review"
        },
        {
          area: "Staff Training Completion",
          priority: "Due in 14 days",
          risk: "medium",
          details: "75% of staff have completed required training"
        },
        {
          area: "Equipment Calibration",
          priority: "4 items pending",
          risk: "low",
          details: "Routine calibration needed for 4 devices"
        }
      ]
    };
    
    setSummaryData(summaryData);
  };

  // Helper function to get an author name based on report type
  const getAuthorByReportType = (type: string): string => {
    switch (type) {
      case 'compliance_summary':
        return 'Compliance Officer';
      case 'audit_findings':
        return 'Audit Team';
      case 'training_status':
        return 'Training Department';
      default:
        return 'System Generated';
    }
  };

  // Helper function to format report type as a readable label
  const formatReportType = (type: string): string => {
    return type
      .split('_')
      .map(word => word.charAt(0).toUpperCase() + word.slice(1))
      .join(' ') + ' Reports';
  };

  return { 
    recentReports, 
    reportTypes, 
    summaryData,
    scheduledReports,
    loading, 
    error,
    refreshData,
    deleteReport 
  };
}; 