import { useState, useEffect } from 'react';

export interface RegulatoryStandard {
  id: string;
  name: string;
  category: string;
  relevance: string; // high, medium, low
  lastUpdated: string;
  description?: string | null;
}

export const useRegulatoryStandards = () => {
  const [standards, setStandards] = useState<RegulatoryStandard[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  
  const fetchStandards = async () => {
    setLoading(true);
    setError(null);
    
    try {
      const response = await fetch('/api/regulations/standards');
      
      if (!response.ok) {
        throw new Error('Failed to fetch regulatory standards');
      }
      
      const data = await response.json();
      setStandards(data);
    } catch (error) {
      console.error('Error fetching regulatory standards:', error);
      setError(error instanceof Error ? error.message : 'An unknown error occurred');
      // Fallback to sample data in case of error
      setStandards([
        { 
          id: "iso9001", 
          name: "ISO 9001:2015", 
          category: "Quality Management",
          lastUpdated: "2023-01-15",
          relevance: "high"
        },
        { 
          id: "hipaa", 
          name: "HIPAA", 
          category: "Data Privacy",
          lastUpdated: "2023-03-22",
          relevance: "high"
        },
        { 
          id: "gdpr", 
          name: "GDPR", 
          category: "Data Protection",
          lastUpdated: "2023-04-10",
          relevance: "medium"
        },
        { 
          id: "jci", 
          name: "JCI 7th Edition", 
          category: "Healthcare Quality",
          lastUpdated: "2023-05-05", 
          relevance: "high"
        },
        { 
          id: "fda21cfr", 
          name: "FDA 21 CFR Part 11", 
          category: "Electronic Records",
          lastUpdated: "2023-02-18",
          relevance: "medium"
        }
      ]);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchStandards();
  }, []);
  
  // Create a new standard
  const createStandard = async (standardData: Omit<RegulatoryStandard, 'id'>) => {
    setLoading(true);
    
    try {
      const response = await fetch('/api/regulations/standards', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(standardData),
      });
      
      if (!response.ok) {
        throw new Error('Failed to create regulatory standard');
      }
      
      await fetchStandards(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error creating regulatory standard:', error);
      setError(error instanceof Error ? error.message : 'Failed to create standard');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Update a standard
  const updateStandard = async (standardData: RegulatoryStandard) => {
    setLoading(true);
    
    try {
      const response = await fetch('/api/regulations/standards', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(standardData),
      });
      
      if (!response.ok) {
        throw new Error('Failed to update regulatory standard');
      }
      
      await fetchStandards(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error updating regulatory standard:', error);
      setError(error instanceof Error ? error.message : 'Failed to update standard');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Delete a standard
  const deleteStandard = async (id: string) => {
    setLoading(true);
    
    try {
      const response = await fetch(`/api/regulations/standards?id=${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        throw new Error('Failed to delete regulatory standard');
      }
      
      await fetchStandards(); // Refresh the list
      return true;
    } catch (error) {
      console.error('Error deleting regulatory standard:', error);
      setError(error instanceof Error ? error.message : 'Failed to delete standard');
      return false;
    } finally {
      setLoading(false);
    }
  };
  
  // Get details of a single standard
  const getStandardById = async (id: string) => {
    // First try to find in current state
    const existing = standards.find(standard => standard.id === id);
    if (existing) {
      return existing;
    }
    
    setLoading(true);
    
    try {
      const response = await fetch(`/api/regulations/standards?id=${id}`);
      
      if (!response.ok) {
        throw new Error('Failed to fetch regulatory standard');
      }
      
      return await response.json();
    } catch (error) {
      console.error('Error fetching regulatory standard:', error);
      setError(error instanceof Error ? error.message : 'Failed to fetch standard details');
      return null;
    } finally {
      setLoading(false);
    }
  };
  
  // Format a date to a nice readable format
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };
  
  return {
    standards,
    loading,
    error,
    createStandard,
    updateStandard,
    deleteStandard,
    getStandardById,
    refreshStandards: fetchStandards,
    formatDate,
  };
}; 