import { useState, useEffect } from 'react';
import { useHospital } from '@/lib/context/HospitalContext';

export interface DocumentItem {
  id: number;
  title: string;
  category: string;
  lastUpdated: string;
  updatedBy: string;
  status: 'active' | 'review' | 'archived';
  department?: string;
  fileName?: string;
  description?: string;
  fileSize?: string;
  fileType?: string;
}

export interface DocumentCategory {
  name: string;
  icon: string;
  count: number;
  description: string;
}

// Default categories structure
const defaultCategories: Omit<DocumentCategory, 'count'>[] = [
  {
    name: "Policies",
    icon: "FileText",
    description: "Organizational policies for compliance and governance"
  },
  {
    name: "Procedures",
    icon: "FileCheck",
    description: "Step-by-step procedures for clinical operations"
  },
  {
    name: "Forms",
    icon: "FileText",
    description: "Standard forms for patient care and administration"
  },
  {
    name: "Training",
    icon: "GraduationCap",
    description: "Staff training resources and compliance education"
  },
  {
    name: "Certifications",
    icon: "Award",
    description: "Certifications and accreditations"
  },
  {
    name: "Reports",
    icon: "BarChart",
    description: "Audit reports and compliance assessments"
  }
];

export const useDocumentsData = () => {
  const [documents, setDocuments] = useState<DocumentItem[]>([]);
  const [categories, setCategories] = useState<DocumentCategory[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const { currentHospital, getSelectedHospitalId } = useHospital();
  
  // Fetch documents from API
  const fetchDocuments = async () => {
    if (!currentHospital) {
      setDocuments([]);
      setLoading(false);
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const hospitalId = getSelectedHospitalId();
      const response = await fetch(`/api/documentation/documents?hospitalId=${hospitalId}`);
      
      if (!response.ok) {
        throw new Error('Failed to fetch documents');
      }
      
      const data = await response.json();
      
      // Transform API data to match our interface if needed
      const transformedDocuments = data.map((doc: any) => ({
        id: doc.id,
        title: doc.title,
        category: doc.category,
        lastUpdated: new Date(doc.updated_at || doc.lastUpdated).toLocaleDateString('en-US', {
          year: 'numeric',
          month: 'long',
          day: 'numeric'
        }),
        updatedBy: doc.updated_by || doc.updatedBy || 'Unknown',
        status: doc.status || 'active',
        department: doc.department,
        fileName: doc.file_name || doc.fileName,
        description: doc.description,
        fileSize: doc.file_size || doc.fileSize,
        fileType: doc.file_type || doc.fileType
      }));
      
      setDocuments(transformedDocuments);
      
    } catch (err) {
      console.error('Error fetching documents:', err);
      setError(err instanceof Error ? err.message : 'Failed to load documents');
      setDocuments([]);
    } finally {
      setLoading(false);
    }
  };

  // Fetch categories from API
  const fetchCategories = async () => {
    try {
      const response = await fetch('/api/documentation/categories');
      
      if (!response.ok) {
        // If categories API fails, use default categories
        console.warn('Failed to fetch categories, using defaults');
        throw new Error('Failed to fetch categories');
      }
      
      const data = await response.json();
      setCategories(data);
    } catch (err) {
      console.error('Error fetching categories:', err);
      // Use default categories with calculated counts
      const categoriesWithCounts = defaultCategories.map(category => ({
        ...category,
        count: documents.filter(doc => doc.category === category.name).length
      }));
      setCategories(categoriesWithCounts);
    }
  };
  
  // Fetch data when hospital changes
  useEffect(() => {
    fetchDocuments();
  }, [currentHospital]);

  // Update category counts when documents change
  useEffect(() => {
    if (documents.length > 0) {
      fetchCategories();
    } else {
      // Reset categories when no documents
      const categoriesWithCounts = defaultCategories.map(category => ({
        ...category,
        count: 0
      }));
      setCategories(categoriesWithCounts);
    }
  }, [documents]);
  
  // Get documents by category
  const getDocumentsByCategory = (category: string) => {
    return documents.filter(doc => doc.category === category);
  };
  
  // Get recent documents (last 5)
  const getRecentDocuments = () => {
    return [...documents]
      .sort((a, b) => new Date(b.lastUpdated).getTime() - new Date(a.lastUpdated).getTime())
      .slice(0, 5);
  };
  
  // Get documents by department
  const getDocumentsByDepartment = (department: string) => {
    return documents.filter(doc => doc.department === department);
  };
  
  // Get documents by status
  const getDocumentsByStatus = (status: 'active' | 'review' | 'archived') => {
    return documents.filter(doc => doc.status === status);
  };

  // Refresh documents
  const refetchDocuments = async () => {
    await fetchDocuments();
  };
  
  return {
    documents,
    categories,
    loading,
    error,
    hospitalName: currentHospital?.name || 'Unknown Hospital',
    getDocumentsByCategory,
    getRecentDocuments,
    getDocumentsByDepartment,
    getDocumentsByStatus,
    refetchDocuments
  };
}; 