import { useState, useEffect } from 'react';

interface DocumentOption {
  id: number;
  value: string;
  label: string;
  description?: string;
  color_class?: string;
  sort_order: number;
}

interface UseDocumentOptionsReturn {
  statuses: DocumentOption[];
  statusesLoading: boolean;
  statusesError: string | null;
  getStatusOptions: () => { value: string; label: string }[];
  getStatusColorClass: (value: string) => string;
  getStatusLabel: (value: string) => string;
  refetch: () => void;
}

export const useDocumentOptions = (): UseDocumentOptionsReturn => {
  const [statuses, setStatuses] = useState<DocumentOption[]>([]);
  const [statusesLoading, setStatusesLoading] = useState(true);
  const [statusesError, setStatusesError] = useState<string | null>(null);

  const fetchStatuses = async () => {
    setStatusesLoading(true);
    setStatusesError(null);
    try {
      const response = await fetch('/api/documentation/statuses');
      if (!response.ok) {
        throw new Error('Failed to fetch statuses');
      }
      const data = await response.json();
      setStatuses(data);
    } catch (error) {
      console.error('Error fetching statuses:', error);
      setStatusesError(error instanceof Error ? error.message : 'Unknown error');
      // Fallback to static data
      setStatuses([
        { id: 1, value: 'active', label: 'Active', sort_order: 1, color_class: 'text-green-600 border-green-300' },
        { id: 2, value: 'review', label: 'Review Needed', sort_order: 2, color_class: 'text-amber-600 border-amber-300' },
        { id: 3, value: 'archived', label: 'Archived', sort_order: 3, color_class: 'text-gray-600 border-gray-300' },
        { id: 4, value: 'draft', label: 'Draft', sort_order: 4, color_class: 'text-blue-600 border-blue-300' },
      ]);
    } finally {
      setStatusesLoading(false);
    }
  };

  useEffect(() => {
    fetchStatuses();
  }, []);

  const getStatusOptions = () => {
    return statuses.map(status => ({
      value: status.value,
      label: status.label
    }));
  };

  const getStatusColorClass = (value: string) => {
    const status = statuses.find(s => s.value === value);
    return status?.color_class || 'text-gray-600 border-gray-300';
  };

  const getStatusLabel = (value: string) => {
    const status = statuses.find(s => s.value === value);
    return status?.label || value.charAt(0).toUpperCase() + value.slice(1);
  };

  const refetch = () => {
    fetchStatuses();
  };

  return {
    statuses,
    statusesLoading,
    statusesError,
    getStatusOptions,
    getStatusColorClass,
    getStatusLabel,
    refetch
  };
}; 