import { useEffect, useState } from 'react';
import { useSession } from 'next-auth/react';

// Utility function to get the selected hospital ID from various sources
export const getSelectedHospitalIdFromBrowser = (): string | null => {
  // Try to get from cookie first
  const cookieHospitalId = typeof document !== 'undefined' 
    ? document.cookie
        .split('; ')
        .find(row => row.startsWith('selectedHospitalId='))
        ?.split('=')[1]
    : null;
      
  // Fall back to localStorage
  const localStorageId = typeof localStorage !== 'undefined'
    ? localStorage.getItem('selectedHospitalId')
    : null;

  return cookieHospitalId || localStorageId;
};

// Utility function to add hospital ID header for super admins
export const addHospitalIdHeader = (headers: HeadersInit = {}, isSuperAdmin: boolean): HeadersInit => {
  if (isSuperAdmin) {
    const hospitalId = getSelectedHospitalIdFromBrowser();
    if (hospitalId) {
      return {
        ...headers,
        'x-hospital-id': hospitalId
      };
    }
  }
  return headers;
};

export interface ComplianceItem {
  id: number;
  name: string;
  compliance: number;
  required: number;
  hospital_id: number;
  created_at: string;
  updated_at: string;
}

export interface HospitalComplianceStats {
  overallCompliance: number;
  atRiskCount: number;
  compliantCount: number;
  remediationNeeded: ComplianceItem[];
}

export interface StandardItem {
  id: number;
  standard: string;
  category: string;
  compliance: number;
  required_threshold: number;
  last_assessed: string;
  hospital_id: number;
  created_at: string;
  updated_at: string;
}

export interface ReportItem {
  id: number;
  title: string;
  type: string;
  file_url: string | null;
  hospital_id: number;
  created_at: string;
}

export interface PolicyItem {
  id: number;
  name: string;
  category: string;
  status: string;
  hospital_id: number;
  created_at: string;
  updated_at: string;
}

export interface ComplianceData {
  id: number;
  name: string;
  compliance: number;
  required: number;
  hospital_id: number;
  created_at: string;
  updated_at: string;
}

export const useComplianceData = () => {
  const { data: session } = useSession();
  const [complianceData, setComplianceData] = useState<ComplianceData[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [deleting, setDeleting] = useState(false);
  const [updating, setUpdating] = useState(false);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch('/api/compliance', {
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch compliance data');
      }
      
      const data = await response.json();
      setComplianceData(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching compliance data:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    if (session) {
      fetchData();
    }
  }, [session]);
  
  const deleteComplianceItem = async (id: number) => {
    setDeleting(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch(`/api/compliance/${id}`, {
        method: 'DELETE',
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to delete compliance item');
      }
      
      // Remove the item from the local state
      setComplianceData(complianceData.filter(item => item.id !== id));
      return true;
    } catch (err) {
      console.error('Error deleting compliance item:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setDeleting(false);
    }
  };
  
  const updateComplianceItem = async (id: number, updatedData: Partial<ComplianceData>) => {
    setUpdating(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({ 'Content-Type': 'application/json' }, isSuperAdmin);
      
      const response = await fetch(`/api/compliance/${id}`, {
        method: 'PUT',
        headers,
        body: JSON.stringify(updatedData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to update compliance item');
      }
      
      const updatedItem = await response.json();
      
      // Update the item in the local state
      setComplianceData(complianceData.map(item => 
        item.id === id ? updatedItem : item
      ));
      
      return true;
    } catch (err) {
      console.error('Error updating compliance item:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setUpdating(false);
    }
  };
  
  const refetch = () => {
    fetchData();
  };
  
  return { 
    complianceData, 
    loading, 
    error, 
    refetch, 
    deleteComplianceItem, 
    updateComplianceItem,
    deleting,
    updating 
  };
};

export const useStandardsData = () => {
  const { data: session } = useSession();
  const [standards, setStandards] = useState<StandardItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [deleting, setDeleting] = useState(false);
  const [updating, setUpdating] = useState(false);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch('/api/compliance/standards', {
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch standards data');
      }
      
      const data = await response.json();
      setStandards(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching standards data:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    if (session) {
      fetchData();
    }
  }, [session]);
  
  const deleteStandard = async (id: number) => {
    setDeleting(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch(`/api/compliance/standards/${id}`, {
        method: 'DELETE',
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to delete standard');
      }
      
      // Remove the standard from the local state
      setStandards(standards.filter(standard => standard.id !== id));
      return true;
    } catch (err) {
      console.error('Error deleting standard:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setDeleting(false);
    }
  };
  
  const updateStandard = async (id: number, updatedData: Partial<StandardItem>) => {
    setUpdating(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({ 'Content-Type': 'application/json' }, isSuperAdmin);
      
      const response = await fetch(`/api/compliance/standards/${id}`, {
        method: 'PUT',
        headers,
        body: JSON.stringify(updatedData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to update standard');
      }
      
      const updatedStandard = await response.json();
      
      // Update the standard in the local state
      setStandards(standards.map(standard => 
        standard.id === id ? updatedStandard : standard
      ));
      
      return true;
    } catch (err) {
      console.error('Error updating standard:', err);
      setError((err as Error).message);
      return false;
    } finally {
      setUpdating(false);
    }
  };
  
  const refetch = () => {
    fetchData();
  };
  
  return { 
    standards, 
    loading, 
    error, 
    refetch, 
    deleteStandard, 
    updateStandard,
    deleting,
    updating 
  };
};

export const useReportsData = () => {
  const { data: session } = useSession();
  const [reports, setReports] = useState<ReportItem[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch('/api/compliance/reports', {
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch reports data');
      }
      
      const data = await response.json();
      setReports(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching reports data:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    if (session) {
      fetchData();
    }
  }, [session]);
  
  const refetch = () => {
    fetchData();
  };
  
  return { reports, loading, error, refetch };
};

export const usePoliciesData = () => {
  const { data: session } = useSession();
  const [policies, setPolicies] = useState<Record<string, PolicyItem[]>>({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      const headers = addHospitalIdHeader({}, isSuperAdmin);
      
      const response = await fetch('/api/compliance/policies', {
        headers
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch policies data');
      }
      
      const data = await response.json();
      setPolicies(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching policies data:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    if (session) {
      fetchData();
    }
  }, [session]);
  
  const refetch = () => {
    fetchData();
  };
  
  return { policies, loading, error, refetch };
}; 

// New hooks for dynamic data
export const usePolicyStatuses = () => {
  const [statuses, setStatuses] = useState<{id: number, value: string, label: string}[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/compliance/policy-statuses');
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch policy statuses');
      }
      
      const data = await response.json();
      setStatuses(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching policy statuses:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchData();
  }, []);
  
  return { statuses, loading, error };
};

export const usePolicyCategories = () => {
  const [categories, setCategories] = useState<{id: number, value: string, label: string}[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/compliance/policy-categories');
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch policy categories');
      }
      
      const data = await response.json();
      setCategories(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching policy categories:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchData();
  }, []);
  
  return { categories, loading, error };
};

export const useReportTypes = () => {
  const [reportTypes, setReportTypes] = useState<{id: number, value: string, label: string}[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const fetchData = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/compliance/report-types');
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || 'Failed to fetch report types');
      }
      
      const data = await response.json();
      setReportTypes(data);
      setError(null);
    } catch (err) {
      console.error('Error fetching report types:', err);
      setError((err as Error).message);
    } finally {
      setLoading(false);
    }
  };
  
  useEffect(() => {
    fetchData();
  }, []);
  
  return { reportTypes, loading, error };
}; 