'use client'

import { useCallback } from 'react';
import { useHospital } from '@/lib/context/HospitalContext';
import { useUser } from '@/lib/context/UserContext';

/**
 * Custom hook for making API calls with the current hospital ID for super admins
 * Regular users will use their assigned hospital ID handled by the server
 */
export const useApiWithHospital = () => {
  const { currentHospital } = useHospital();
  const { isSuperAdmin } = useUser();

  const fetchWithHospital = useCallback(
    async <T>(
      url: string,
      options?: RequestInit
    ): Promise<T> => {
      try {
        // For superadmins, add the hospital ID as a query parameter if available
        const urlWithHospital = isSuperAdmin && currentHospital
          ? `${url}${url.includes('?') ? '&' : '?'}hospitalId=${currentHospital.id}`
          : url;

        const response = await fetch(urlWithHospital, {
          ...options,
          headers: {
            'Content-Type': 'application/json',
            ...(options?.headers || {}),
          },
        });

        if (!response.ok) {
          // Try to parse error JSON but don't fail if it's not valid JSON
          let errorMessage = `API error: ${response.status} ${response.statusText}`;
          
          try {
            const errorData = await response.json();
            if (errorData && (errorData.message || errorData.error)) {
              errorMessage = errorData.message || errorData.error || errorMessage;
            }
          } catch (jsonError) {
            // If we can't parse JSON, use the status text
            console.warn('Failed to parse error response as JSON:', jsonError);
          }
          
          // Create a more detailed error with the status code
          const error = new Error(errorMessage);
          (error as any).status = response.status;
          throw error;
        }

        // Try to parse JSON response but handle non-JSON responses gracefully
        try {
          return await response.json();
        } catch (jsonError) {
          console.warn('Response is not valid JSON:', jsonError);
          // Return empty object for non-JSON responses
          return {} as T;
        }
      } catch (error) {
        console.error('API request failed:', error);
        // Add more context to the error
        if (error instanceof Error) {
          error.message = `Request to ${url} failed: ${error.message}`;
        }
        throw error;
      }
    },
    [currentHospital, isSuperAdmin]
  );

  const postWithHospital = useCallback(
    async <T>(
      url: string,
      data: any,
      options?: RequestInit
    ): Promise<T> => {
      return fetchWithHospital<T>(url, {
        ...options,
        method: 'POST',
        body: JSON.stringify(data),
      });
    },
    [fetchWithHospital]
  );

  return {
    fetchWithHospital,
    postWithHospital,
  };
};

export default useApiWithHospital; 