'use client'

import { useState } from 'react'
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { 
  Dialog, 
  DialogContent, 
  DialogDescription, 
  DialogFooter, 
  DialogHeader, 
  DialogTitle, 
  DialogTrigger,
  DialogClose
} from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { TrainingEvent } from '@/hooks/useScheduleData'

interface TrainingCalendarTabProps {
  trainingEvents: TrainingEvent[]
  loading: boolean
  error: string | null
  onAdd: (training: Omit<TrainingEvent, 'id'>) => Promise<boolean>
  onEdit: (training: TrainingEvent) => Promise<boolean>
  onDelete: (id: string) => Promise<boolean>
}

export function TrainingCalendarTab({
  trainingEvents,
  loading,
  error,
  onAdd,
  onEdit,
  onDelete
}: TrainingCalendarTabProps) {
  const [selectedTraining, setSelectedTraining] = useState<TrainingEvent | null>(null)
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false)
  
  // Form state for add/edit
  const [formData, setFormData] = useState<Omit<TrainingEvent, 'id'>>({
    title: '',
    date: '',
    instructor: '',
    location: '',
    department: '',
    duration: '',
    capacity: 0,
    enrolled: 0,
    required: false
  })
  
  // Reset form data
  const resetFormData = () => {
    setFormData({
      title: '',
      date: '',
      instructor: '',
      location: '',
      department: '',
      duration: '',
      capacity: 0,
      enrolled: 0,
      required: false
    })
  }
  
  // Open edit dialog with selected training data
  const handleEditClick = (training: TrainingEvent) => {
    setSelectedTraining(training)
    setFormData({
      title: training.title,
      date: training.date,
      instructor: training.instructor,
      location: training.location,
      department: training.department,
      duration: training.duration,
      capacity: training.capacity,
      enrolled: training.enrolled,
      required: training.required
    })
    setIsEditDialogOpen(true)
  }
  
  // Open delete dialog with selected training
  const handleDeleteClick = (training: TrainingEvent) => {
    setSelectedTraining(training)
    setIsDeleteDialogOpen(true)
  }
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type } = e.target
    
    if (type === 'number') {
      setFormData(prev => ({
        ...prev,
        [name]: parseInt(value)
      }))
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }))
    }
  }
  
  // Handle switch change
  const handleSwitchChange = (checked: boolean) => {
    setFormData(prev => ({
      ...prev,
      required: checked
    }))
  }
  
  // Handle form submission for adding a new training
  const handleAddSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    const success = await onAdd(formData)
    if (success) {
      setIsAddDialogOpen(false)
      resetFormData()
    }
  }
  
  // Handle form submission for editing a training
  const handleEditSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!selectedTraining) return
    
    const success = await onEdit({
      id: selectedTraining.id,
      ...formData
    })
    
    if (success) {
      setIsEditDialogOpen(false)
      setSelectedTraining(null)
      resetFormData()
    }
  }
  
  // Handle confirmation of delete
  const handleDeleteConfirm = async () => {
    if (!selectedTraining) return
    
    const success = await onDelete(selectedTraining.id)
    if (success) {
      setIsDeleteDialogOpen(false)
      setSelectedTraining(null)
    }
  }
  
  // Calculate enrollment percentage
  const calculateEnrollmentPercentage = (enrolled: number, capacity: number) => {
    if (capacity === 0) return 0
    return Math.round((enrolled / capacity) * 100)
  }
  
  // Render enrollment status with color-coded badge
  const renderEnrollmentStatus = (enrolled: number, capacity: number) => {
    const percentage = calculateEnrollmentPercentage(enrolled, capacity)
    
    if (percentage >= 90) {
      return <Badge variant="outline" className="bg-red-50 text-red-700 border-red-200">Full ({percentage}%)</Badge>
    } else if (percentage >= 70) {
      return <Badge variant="outline" className="bg-yellow-50 text-yellow-700 border-yellow-200">Limited ({percentage}%)</Badge>
    } else {
      return <Badge variant="outline" className="bg-green-50 text-green-700 border-green-200">Available ({percentage}%)</Badge>
    }
  }
  
  // Loading state
  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
      </div>
    )
  }
  
  // Error state
  if (error) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="text-red-500">Error: {error}</div>
      </div>
    )
  }
  
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-xl font-semibold">Training Calendar</h2>
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button>Add New Training</Button>
          </DialogTrigger>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Add New Training Event</DialogTitle>
              <DialogDescription>
                Enter the details for the new training event
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleAddSubmit}>
              <div className="grid gap-4 py-4">
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="title" className="text-right">Title</Label>
                  <Input
                    id="title"
                    name="title"
                    value={formData.title}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="date" className="text-right">Date</Label>
                  <Input
                    id="date"
                    name="date"
                    type="date"
                    value={formData.date}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="instructor" className="text-right">Instructor</Label>
                  <Input
                    id="instructor"
                    name="instructor"
                    value={formData.instructor}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="location" className="text-right">Location</Label>
                  <Input
                    id="location"
                    name="location"
                    value={formData.location}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="department" className="text-right">Department</Label>
                  <Input
                    id="department"
                    name="department"
                    value={formData.department}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="duration" className="text-right">Duration</Label>
                  <Input
                    id="duration"
                    name="duration"
                    value={formData.duration}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                    placeholder="e.g., 2 hours"
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="capacity" className="text-right">Capacity</Label>
                  <Input
                    id="capacity"
                    name="capacity"
                    type="number"
                    min="1"
                    value={formData.capacity}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="enrolled" className="text-right">Enrolled</Label>
                  <Input
                    id="enrolled"
                    name="enrolled"
                    type="number"
                    min="0"
                    value={formData.enrolled}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="required" className="text-right">Required</Label>
                  <div className="col-span-3 flex items-center space-x-2">
                    <Switch 
                      id="required" 
                      checked={formData.required} 
                      onCheckedChange={handleSwitchChange} 
                    />
                    <Label htmlFor="required">Mandatory training</Label>
                  </div>
                </div>
              </div>
              <DialogFooter>
                <DialogClose asChild>
                  <Button variant="outline">Cancel</Button>
                </DialogClose>
                <Button type="submit">Save</Button>
              </DialogFooter>
            </form>
          </DialogContent>
        </Dialog>
      </div>
      
      <div className="rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Title</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Instructor</TableHead>
              <TableHead>Department</TableHead>
              <TableHead>Enrollment</TableHead>
              <TableHead>Required</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {trainingEvents.length === 0 ? (
              <TableRow>
                <TableCell colSpan={7} className="text-center h-32">
                  No training events found
                </TableCell>
              </TableRow>
            ) : (
              trainingEvents.map((training) => (
                <TableRow key={training.id}>
                  <TableCell className="font-medium">{training.title}</TableCell>
                  <TableCell>{training.date}</TableCell>
                  <TableCell>{training.instructor}</TableCell>
                  <TableCell>{training.department}</TableCell>
                  <TableCell>
                    {renderEnrollmentStatus(training.enrolled, training.capacity)}
                    <div className="text-xs text-gray-500 mt-1">
                      {training.enrolled} / {training.capacity}
                    </div>
                  </TableCell>
                  <TableCell>
                    {training.required ? (
                      <Badge className="bg-blue-500">Required</Badge>
                    ) : (
                      <Badge variant="outline">Optional</Badge>
                    )}
                  </TableCell>
                  <TableCell className="text-right">
                    <div className="flex justify-end gap-2">
                      <Button variant="outline" size="sm" onClick={() => handleEditClick(training)}>
                        Edit
                      </Button>
                      <Button variant="destructive" size="sm" onClick={() => handleDeleteClick(training)}>
                        Delete
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>
      
      {/* Edit Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Edit Training Event</DialogTitle>
            <DialogDescription>
              Update the details for this training event
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleEditSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-title" className="text-right">Title</Label>
                <Input
                  id="edit-title"
                  name="title"
                  value={formData.title}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-date" className="text-right">Date</Label>
                <Input
                  id="edit-date"
                  name="date"
                  type="date"
                  value={formData.date}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-instructor" className="text-right">Instructor</Label>
                <Input
                  id="edit-instructor"
                  name="instructor"
                  value={formData.instructor}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-location" className="text-right">Location</Label>
                <Input
                  id="edit-location"
                  name="location"
                  value={formData.location}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-department" className="text-right">Department</Label>
                <Input
                  id="edit-department"
                  name="department"
                  value={formData.department}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-duration" className="text-right">Duration</Label>
                <Input
                  id="edit-duration"
                  name="duration"
                  value={formData.duration}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-capacity" className="text-right">Capacity</Label>
                <Input
                  id="edit-capacity"
                  name="capacity"
                  type="number"
                  min="1"
                  value={formData.capacity}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-enrolled" className="text-right">Enrolled</Label>
                <Input
                  id="edit-enrolled"
                  name="enrolled"
                  type="number"
                  min="0"
                  value={formData.enrolled}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-required" className="text-right">Required</Label>
                <div className="col-span-3 flex items-center space-x-2">
                  <Switch 
                    id="edit-required" 
                    checked={formData.required} 
                    onCheckedChange={handleSwitchChange} 
                  />
                  <Label htmlFor="edit-required">Mandatory training</Label>
                </div>
              </div>
            </div>
            <DialogFooter>
              <DialogClose asChild>
                <Button variant="outline">Cancel</Button>
              </DialogClose>
              <Button type="submit">Update</Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
      
      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This will permanently delete the training event{' '}
              <span className="font-semibold">{selectedTraining?.title}</span>.
              This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction onClick={handleDeleteConfirm} className="bg-red-600 hover:bg-red-700">
              Delete
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  )
} 