'use client'

import { useState } from 'react'
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { 
  Dialog, 
  DialogContent, 
  DialogDescription, 
  DialogFooter, 
  DialogHeader, 
  DialogTitle, 
  DialogTrigger,
  DialogClose
} from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { ResourceAllocation } from '@/hooks/useScheduleData'
import { Progress } from "@/components/ui/progress"

interface ResourcePlanningTabProps {
  resourceAllocations: ResourceAllocation[]
  loading: boolean
  error: string | null
  onAdd: (resource: Omit<ResourceAllocation, 'id'>) => Promise<boolean>
  onEdit: (resource: ResourceAllocation) => Promise<boolean>
  onDelete: (id: string) => Promise<boolean>
}

export function ResourcePlanningTab({
  resourceAllocations,
  loading,
  error,
  onAdd,
  onEdit,
  onDelete
}: ResourcePlanningTabProps) {
  const [selectedResource, setSelectedResource] = useState<ResourceAllocation | null>(null)
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false)
  
  // Form state for add/edit
  const [formData, setFormData] = useState<Omit<ResourceAllocation, 'id'>>({
    resourceName: '',
    role: '',
    allocatedTo: '',
    startDate: '',
    endDate: '',
    utilizationPercentage: 0,
    department: ''
  })
  
  // Reset form data
  const resetFormData = () => {
    setFormData({
      resourceName: '',
      role: '',
      allocatedTo: '',
      startDate: '',
      endDate: '',
      utilizationPercentage: 0,
      department: ''
    })
  }
  
  // Open edit dialog with selected resource data
  const handleEditClick = (resource: ResourceAllocation) => {
    setSelectedResource(resource)
    setFormData({
      resourceName: resource.resourceName,
      role: resource.role,
      allocatedTo: resource.allocatedTo,
      startDate: resource.startDate,
      endDate: resource.endDate,
      utilizationPercentage: resource.utilizationPercentage,
      department: resource.department
    })
    setIsEditDialogOpen(true)
  }
  
  // Open delete dialog with selected resource
  const handleDeleteClick = (resource: ResourceAllocation) => {
    setSelectedResource(resource)
    setIsDeleteDialogOpen(true)
  }
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type } = e.target
    
    if (type === 'number') {
      setFormData(prev => ({
        ...prev,
        [name]: parseInt(value)
      }))
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }))
    }
  }
  
  // Handle form submission for adding a new resource
  const handleAddSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    const success = await onAdd(formData)
    if (success) {
      setIsAddDialogOpen(false)
      resetFormData()
    }
  }
  
  // Handle form submission for editing a resource
  const handleEditSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!selectedResource) return
    
    const success = await onEdit({
      id: selectedResource.id,
      ...formData
    })
    
    if (success) {
      setIsEditDialogOpen(false)
      setSelectedResource(null)
      resetFormData()
    }
  }
  
  // Handle confirmation of delete
  const handleDeleteConfirm = async () => {
    if (!selectedResource) return
    
    const success = await onDelete(selectedResource.id)
    if (success) {
      setIsDeleteDialogOpen(false)
      setSelectedResource(null)
    }
  }
  
  // Render utilization with progress bar and color coding
  const renderUtilization = (percentage: number) => {
    let color = "bg-green-500"
    if (percentage > 90) {
      color = "bg-red-500"
    } else if (percentage > 75) {
      color = "bg-amber-500"
    } else if (percentage > 50) {
      color = "bg-blue-500"
    }
    
    return (
      <div className="w-full space-y-1">
        <div className="flex justify-between text-xs">
          <span>{percentage}%</span>
        </div>
        <Progress value={percentage} className={`h-2 ${color}`} />
      </div>
    )
  }
  
  // Loading state
  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
      </div>
    )
  }
  
  // Error state
  if (error) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="text-red-500">Error: {error}</div>
      </div>
    )
  }
  
  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-xl font-semibold">Resource Planning</h2>
        <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
          <DialogTrigger asChild>
            <Button>Add New Resource</Button>
          </DialogTrigger>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Add New Resource Allocation</DialogTitle>
              <DialogDescription>
                Enter the details for the new resource allocation
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleAddSubmit}>
              <div className="grid gap-4 py-4">
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="resourceName" className="text-right">Resource Name</Label>
                  <Input
                    id="resourceName"
                    name="resourceName"
                    value={formData.resourceName}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="role" className="text-right">Role</Label>
                  <Input
                    id="role"
                    name="role"
                    value={formData.role}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="department" className="text-right">Department</Label>
                  <Input
                    id="department"
                    name="department"
                    value={formData.department}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="allocatedTo" className="text-right">Allocated To</Label>
                  <Input
                    id="allocatedTo"
                    name="allocatedTo"
                    value={formData.allocatedTo}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="startDate" className="text-right">Start Date</Label>
                  <Input
                    id="startDate"
                    name="startDate"
                    type="date"
                    value={formData.startDate}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="endDate" className="text-right">End Date</Label>
                  <Input
                    id="endDate"
                    name="endDate"
                    type="date"
                    value={formData.endDate}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
                <div className="grid grid-cols-4 items-center gap-4">
                  <Label htmlFor="utilizationPercentage" className="text-right">Utilization %</Label>
                  <Input
                    id="utilizationPercentage"
                    name="utilizationPercentage"
                    type="number"
                    min="0"
                    max="100"
                    value={formData.utilizationPercentage}
                    onChange={handleInputChange}
                    className="col-span-3"
                    required
                  />
                </div>
              </div>
              <DialogFooter>
                <Button type="submit">Add Resource</Button>
              </DialogFooter>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      <div className="rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Resource Name</TableHead>
              <TableHead>Role</TableHead>
              <TableHead>Department</TableHead>
              <TableHead>Allocated To</TableHead>
              <TableHead>Period</TableHead>
              <TableHead className="w-[140px]">Utilization</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {resourceAllocations.length === 0 ? (
              <TableRow>
                <TableCell colSpan={7} className="text-center h-32">
                  No resource allocations found
                </TableCell>
              </TableRow>
            ) : (
              resourceAllocations.map((resource) => (
                <TableRow key={resource.id}>
                  <TableCell className="font-medium">{resource.resourceName}</TableCell>
                  <TableCell>{resource.role}</TableCell>
                  <TableCell>{resource.department}</TableCell>
                  <TableCell>{resource.allocatedTo}</TableCell>
                  <TableCell>{resource.startDate} - {resource.endDate}</TableCell>
                  <TableCell>{renderUtilization(resource.utilizationPercentage)}</TableCell>
                  <TableCell className="text-right">
                    <div className="flex justify-end gap-2">
                      <Button 
                        variant="ghost" 
                        size="sm"
                        onClick={() => handleEditClick(resource)}
                      >
                        Edit
                      </Button>
                      <Button 
                        variant="ghost" 
                        size="sm" 
                        className="text-red-500"
                        onClick={() => handleDeleteClick(resource)}
                      >
                        Delete
                      </Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>

      {/* Edit Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Edit Resource Allocation</DialogTitle>
            <DialogDescription>
              Update the details for this resource allocation
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleEditSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-resourceName" className="text-right">Resource Name</Label>
                <Input
                  id="edit-resourceName"
                  name="resourceName"
                  value={formData.resourceName}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-role" className="text-right">Role</Label>
                <Input
                  id="edit-role"
                  name="role"
                  value={formData.role}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-department" className="text-right">Department</Label>
                <Input
                  id="edit-department"
                  name="department"
                  value={formData.department}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-allocatedTo" className="text-right">Allocated To</Label>
                <Input
                  id="edit-allocatedTo"
                  name="allocatedTo"
                  value={formData.allocatedTo}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-startDate" className="text-right">Start Date</Label>
                <Input
                  id="edit-startDate"
                  name="startDate"
                  type="date"
                  value={formData.startDate}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-endDate" className="text-right">End Date</Label>
                <Input
                  id="edit-endDate"
                  name="endDate"
                  type="date"
                  value={formData.endDate}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
              <div className="grid grid-cols-4 items-center gap-4">
                <Label htmlFor="edit-utilizationPercentage" className="text-right">Utilization %</Label>
                <Input
                  id="edit-utilizationPercentage"
                  name="utilizationPercentage"
                  type="number"
                  min="0"
                  max="100"
                  value={formData.utilizationPercentage}
                  onChange={handleInputChange}
                  className="col-span-3"
                  required
                />
              </div>
            </div>
            <DialogFooter>
              <Button type="submit">Update Resource</Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      {/* Delete Alert Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Are you sure?</AlertDialogTitle>
            <AlertDialogDescription>
              This will permanently delete the resource allocation. This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Cancel</AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleDeleteConfirm}
              className="bg-red-500 hover:bg-red-600"
            >
              Delete
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  )
} 