import { ElementType, useMemo, useCallback } from 'react'
import Link from 'next/link'
import { usePathname } from 'next/navigation'
import { useSession } from 'next-auth/react'
import { cn } from '@/lib/utils'

interface NavigationItem {
  title: string;
  path: string;
  icon: () => React.ReactElement;
  badge?: string;
}

interface SuperadminNavigationProps {
  collapsed: boolean;
}

// Enhanced NavigationLink component with prefetching
const SuperadminNavigationLink = ({ 
  item, 
  pathname, 
  collapsed 
}: { 
  item: NavigationItem, 
  pathname: string, 
  collapsed: boolean 
}) => {
  const isActive = pathname === item.path
  
  return (
    <Link
      href={item.path}
      prefetch={true} // Enable prefetching for faster navigation
      className={cn(
        "flex items-center rounded-md px-3 py-2 text-sm font-medium relative transition-all duration-150", // Faster transition
        isActive 
          ? "bg-blue-100 text-blue-700" 
          : "text-gray-700 hover:bg-gray-100",
        collapsed ? "justify-center" : ""
      )}
    >
      <item.icon />
      {!collapsed && <span className="ml-3">{item.title}</span>}
      {item.badge && !collapsed && (
        <span className="ml-auto bg-blue-600 text-white text-[10px] py-0.5 px-1 rounded">
          {item.badge}
        </span>
      )}
    </Link>
  )
}

const SuperadminNavigation = ({ collapsed }: SuperadminNavigationProps) => {
  const { data: session } = useSession()
  const pathname = usePathname()
  
  // Memoized user role check
  const isSuperAdmin = useMemo(() => 
    session?.user?.role === 'super_admin', 
    [session?.user?.role]
  )
  
  // Memoized admin navigation items to prevent re-creation
  const adminItems = useMemo(() => [
    { title: 'Hospitals', path: '/admin/hospitals', icon: HospitalIcon },
    { title: 'Settings', path: '/admin/settings', icon: SettingsIcon },
    { title: 'Security', path: '/admin/security', icon: SecurityIcon, badge: 'New' },
    { title: 'Database', path: '/admin/database', icon: DatabaseIcon },
    { title: 'Access', path: '/admin/access', icon: AccessIcon },
  ], [])

  // Memoized navigation list to prevent unnecessary re-renders
  const navigationList = useMemo(() => 
    adminItems.map((item) => (
      <SuperadminNavigationLink 
        key={item.path}
        item={item}
        pathname={pathname}
        collapsed={collapsed}
      />
    )), [adminItems, pathname, collapsed])
  
  // Only render if the user is a superadmin
  if (!isSuperAdmin) return null

  return (
    <div className="mt-6 pt-6 border-t border-gray-200">
      <h3 className={cn(
        "px-3 text-xs font-semibold text-gray-500 uppercase tracking-wider",
        collapsed ? "sr-only" : ""
      )}>
        SUPERADMIN
      </h3>
      <nav className="mt-2 space-y-1">
        {navigationList}
      </nav>
    </div>
  )
}

// Icon Components
const HospitalIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M8 3v2" />
    <path d="M16 3v2" />
    <path d="M21 12h-3a1 1 0 0 1-1-1V7a1 1 0 0 0-1-1H8a1 1 0 0 0-1 1v4a1 1 0 0 1-1 1H3" />
    <path d="M7 14a1 1 0 0 1 1-1h8a1 1 0 0 1 1 1v7H7z" />
    <path d="M12 3v16" />
  </svg>
)

const UsersIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2" />
    <circle cx="9" cy="7" r="4" />
    <path d="M22 21v-2a4 4 0 0 0-3-3.87" />
    <path d="M16 3.13a4 4 0 0 1 0 7.75" />
  </svg>
)

const SettingsIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <path d="M12.22 2h-.44a2 2 0 0 0-2 2v.18a2 2 0 0 1-1 1.73l-.43.25a2 2 0 0 1-2 0l-.15-.08a2 2 0 0 0-2.73.73l-.22.38a2 2 0 0 0 .73 2.73l.15.1a2 2 0 0 1 1 1.72v.51a2 2 0 0 1-1 1.74l-.15.09a2 2 0 0 0-.73 2.73l.22.38a2 2 0 0 0 2.73.73l.15-.08a2 2 0 0 1 2 0l.43.25a2 2 0 0 1 1 1.73V20a2 2 0 0 0 2 2h.44a2 2 0 0 0 2-2v-.18a2 2 0 0 1 1-1.73l.43-.25a2 2 0 0 1 2 0l.15.08a2 2 0 0 0 2.73-.73l.22-.39a2 2 0 0 0-.73-2.73l-.15-.08a2 2 0 0 1-1-1.74v-.5a2 2 0 0 1 1-1.74l.15-.09a2 2 0 0 0 .73-2.73l-.22-.38a2 2 0 0 0-2.73-.73l-.15.08a2 2 0 0 1-2 0l-.43-.25a2 2 0 0 1-1-1.73V4a2 2 0 0 0-2-2z" />
    <circle cx="12" cy="12" r="3" />
  </svg>
)

const SecurityIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <circle cx="12" cy="12" r="10" />
    <path d="M12 2a4.5 4.5 0 0 0 0 9 4.5 4.5 0 0 1 0 9 10 10 0 0 0 0-18Z" />
    <path d="M12 2c-1.4 0-2.8.5-4 1.5" />
  </svg>
)

const DatabaseIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <ellipse cx="12" cy="5" rx="9" ry="3" />
    <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3" />
    <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5" />
  </svg>
)

const AccessIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <rect width="18" height="11" x="3" y="11" rx="2" ry="2" />
    <path d="M7 11V7a5 5 0 0 1 10 0v4" />
    <circle cx="12" cy="16" r="1" />
  </svg>
)

const BuildingIcon = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="w-5 h-5">
    <rect width="16" height="20" x="4" y="2" rx="2" ry="2" />
    <path d="M9 22v-4h6v4" />
    <path d="M8 6h.01" />
    <path d="M16 6h.01" />
    <path d="M12 6h.01" />
    <path d="M12 10h.01" />
    <path d="M12 14h.01" />
    <path d="M16 10h.01" />
    <path d="M16 14h.01" />
    <path d="M8 10h.01" />
    <path d="M8 14h.01" />
  </svg>
)

export default SuperadminNavigation 