import { useState, useEffect } from 'react';
import Link from 'next/link';
import { StandardItem, useStandardsData } from '@/hooks/useComplianceData';
import { cn } from '@/lib/utils';
import { useSession } from 'next-auth/react';

interface SidebarRecentStandardsProps {
  collapsed: boolean;
}

const SidebarRecentStandards = ({ collapsed }: SidebarRecentStandardsProps) => {
  const { data: session } = useSession();
  const [recentStandards, setRecentStandards] = useState<StandardItem[]>([]);
  const { standards, loading, error } = useStandardsData();

  useEffect(() => {
    if (standards && standards.length > 0) {
      // Sort by created_at date in descending order and take the 3 most recent
      const sortedStandards = [...standards].sort((a, b) => 
        new Date(b.created_at).getTime() - new Date(a.created_at).getTime()
      ).slice(0, 3);
      
      setRecentStandards(sortedStandards);
    }
  }, [standards]);

  // If session is loading or not available, show loading state
  if (!session) {
    return (
      <div className="mt-6 pt-6 border-t border-gray-200">
        <h3 className={cn(
          "px-3 text-xs font-semibold text-gray-500 uppercase tracking-wider",
          collapsed ? "sr-only" : ""
        )}>
          Recent Standards
        </h3>
        <nav className="mt-2 space-y-1">
          {/* Loading skeleton */}
          {collapsed ? (
            <>
              <div className="flex justify-center items-center rounded-md p-2">
                <div className="h-4 w-4 bg-gray-200 rounded-full animate-pulse"></div>
              </div>
              <div className="flex justify-center items-center rounded-md p-2">
                <div className="h-4 w-4 bg-gray-200 rounded-full animate-pulse"></div>
              </div>
            </>
          ) : (
            <>
              <div className="px-3 py-1">
                <div className="h-4 w-3/4 bg-gray-200 rounded animate-pulse"></div>
              </div>
              <div className="px-3 py-1">
                <div className="h-4 w-3/4 bg-gray-200 rounded animate-pulse"></div>
              </div>
            </>
          )}
        </nav>
      </div>
    );
  }

  return (
    <div className="mt-6 pt-6 border-t border-gray-200">
      <h3 className={cn(
        "px-3 text-xs font-semibold text-gray-500 uppercase tracking-wider",
        collapsed ? "sr-only" : ""
      )}>
        Recent Standards
      </h3>
      <nav className="mt-2 space-y-1">
        {loading ? (
          collapsed ? (
            <>
              <div className="flex justify-center items-center rounded-md p-2">
                <div className="h-4 w-4 bg-gray-200 rounded-full animate-pulse"></div>
              </div>
              <div className="flex justify-center items-center rounded-md p-2">
                <div className="h-4 w-4 bg-gray-200 rounded-full animate-pulse"></div>
              </div>
              <div className="flex justify-center items-center rounded-md p-2">
                <div className="h-4 w-4 bg-gray-200 rounded-full animate-pulse"></div>
              </div>
            </>
          ) : (
            <>
              <div className="px-3 py-1">
                <div className="h-4 w-3/4 bg-gray-200 rounded animate-pulse"></div>
              </div>
              <div className="px-3 py-1">
                <div className="h-4 w-3/4 bg-gray-200 rounded animate-pulse"></div>
              </div>
              <div className="px-3 py-1">
                <div className="h-4 w-3/4 bg-gray-200 rounded animate-pulse"></div>
              </div>
            </>
          )
        ) : error || recentStandards.length === 0 ? (
          <div className={cn(
            "text-xs text-gray-500",
            collapsed ? "text-center px-1" : "px-3"
          )}>
            {collapsed ? "-" : "No standards available"}
          </div>
        ) : (
          collapsed ? (
            <>
              {recentStandards.map((standard) => (
                <Link
                  key={standard.id}
                  href={`/compliance?standard=${standard.id}`}
                  className="flex justify-center items-center rounded-md p-2 text-gray-700 hover:text-blue-600"
                  title={standard.standard}
                >
                  {standard.standard.substring(0, 3)}
                </Link>
              ))}
            </>
          ) : (
            recentStandards.map((standard) => (
              <Link
                key={standard.id}
                href={`/compliance?standard=${standard.id}`}
                className="block px-3 py-1 text-sm text-gray-600 hover:text-blue-600"
              >
                {standard.standard}
              </Link>
            ))
          )
        )}
      </nav>
    </div>
  );
};

export default SidebarRecentStandards; 