import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Download, FileText, Info, Pencil, Trash2, Loader2, Plus } from "lucide-react";
import { useState, useEffect } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger, DialogClose } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { toast } from "sonner";
import { useHospital } from "@/lib/context/HospitalContext";

interface DocumentItem {
  id: string;
  title: string;
  fileName?: string;
  fileUrl?: string;
  fileSize?: number;
  fileType?: string;
  status: string;
  department?: string;
  updatedBy: string;
  category_id: string;
  hospital_id?: number;
  createdAt: string;
  updatedAt: string;
  category: {
    id: string;
    name: string;
    icon: string;
    description: string;
  };
}

interface DocumentCategory {
  id: string;
  name: string;
  icon: string;
  description: string;
  count: number;
}

const DocumentsTab = () => {
  const { currentHospital } = useHospital();
  const [documents, setDocuments] = useState<DocumentItem[]>([]);
  const [categories, setCategories] = useState<DocumentCategory[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [filterCategory, setFilterCategory] = useState<string | null>(null);
  const [newDocumentDialogOpen, setNewDocumentDialogOpen] = useState(false);
  const [editDocumentDialogOpen, setEditDocumentDialogOpen] = useState(false);
  const [documentToEdit, setDocumentToEdit] = useState<DocumentItem | null>(null);
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isUploading, setIsUploading] = useState(false);
  const [formData, setFormData] = useState({
    title: '',
    fileUrl: '',
    fileSize: 0,
    fileType: '',
    status: 'active',
    department: '',
    category_id: ''
  });
  
  // Fetch documents and categories
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        console.log('Fetching document categories...');
        // Fetch document categories
        const categoriesResponse = await fetch('/api/documentation/categories');
        console.log('Categories response status:', categoriesResponse.status);
        
        if (!categoriesResponse.ok) {
          const errorText = await categoriesResponse.text();
          console.error('Categories error response:', errorText);
          throw new Error(`Failed to fetch categories: ${categoriesResponse.statusText}`);
        }
        
        const categoriesData = await categoriesResponse.json();
        console.log('Categories data:', categoriesData);
        setCategories(categoriesData);
        
        console.log('Fetching documents...');
        // Fetch documents
        const documentsUrl = filterCategory 
          ? `/api/documentation/documents?categoryId=${filterCategory}` 
          : '/api/documentation/documents';
        
        const documentsResponse = await fetch(documentsUrl);
        console.log('Documents response status:', documentsResponse.status);
        
        if (!documentsResponse.ok) {
          const errorText = await documentsResponse.text();
          console.error('Documents error response:', errorText);
          throw new Error(`Failed to fetch documents: ${documentsResponse.statusText}`);
        }
        
        const documentsData = await documentsResponse.json();
        console.log('Documents data:', documentsData);
        setDocuments(documentsData);
        
        setError(null);
      } catch (err) {
        console.error('Error fetching data:', err);
        setError(err instanceof Error ? err.message : 'Failed to load documents');
        // Show empty arrays instead of leaving the previous state
        setCategories([]);
        setDocuments([]);
      } finally {
        setLoading(false);
      }
    };
    
    fetchData();
  }, [filterCategory]);
  
  // Handle form input changes
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };
  
  // Handle select changes
  const handleSelectChange = (name: string, value: string) => {
    console.log(`Setting ${name} to ${value}`);
    setFormData(prev => ({ ...prev, [name]: value }));
  };
  
  // Reset form data
  const resetForm = () => {
    setFormData({
      title: '',
      fileUrl: '',
      fileSize: 0,
      fileType: '',
      status: 'active',
      department: '',
      category_id: ''
    });
    setSelectedFile(null);
    setUploadProgress(0);
  };
  
  // Open edit document dialog
  const openEditDialog = (document: DocumentItem) => {
    setDocumentToEdit(document);
    setFormData({
      title: document.title,
      fileUrl: document.fileUrl || '',
      fileSize: document.fileSize || 0,
      fileType: document.fileType || '',
      status: document.status,
      department: document.department || '',
      category_id: document.category_id
    });
    if (document.fileName) {
      // No need to set selectedFile here, just display the filename
    }
    setEditDocumentDialogOpen(true);
  };
  
  // Handle file selection
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setSelectedFile(file);
      setFormData(prev => ({
        ...prev,
        fileType: file.type,
        fileSize: file.size
      }));
      console.log("File selected:", file.name);
    }
  };

  // Upload the selected file
  const uploadFile = async () => {
    if (!selectedFile) {
      console.log("No file selected for upload");
      return null;
    }
    
    try {
      setIsUploading(true);
      setUploadProgress(0);
      
      const formData = new FormData();
      formData.append('file', selectedFile);
      console.log("Uploading file:", selectedFile.name);
      
      const response = await fetch('/api/documentation/upload', {
        method: 'POST',
        body: formData,
      });
      
      if (!response.ok) {
        throw new Error('Failed to upload file');
      }
      
      setUploadProgress(100);
      const data = await response.json();
      setIsUploading(false);
      console.log("File uploaded successfully:", data);
      
      return data;
    } catch (error) {
      console.error('Error uploading file:', error);
      setIsUploading(false);
      toast.error('Failed to upload file');
      return null;
    }
  };

  // Create new document
  const createDocument = async () => {
    try {
      console.log('Creating document with data:', formData);
      
      let fileData = null;
      
      // Upload file if selected
      if (selectedFile) {
        console.log('Uploading file for new document');
        fileData = await uploadFile();
        if (!fileData) {
          toast.error('Failed to upload file');
          return;
        }
      } else {
        console.log('No file selected for new document');
      }
      
      // Prepare document data with file information if available
      const documentData = {
        ...formData,
        ...(fileData && {
          fileName: fileData.fileName || selectedFile?.name,
          fileUrl: fileData.fileUrl,
          fileSize: fileData.fileSize || selectedFile?.size,
          fileType: fileData.fileType || selectedFile?.type
        })
      };
      
      console.log('Sending document data to API:', documentData);
      
      const response = await fetch('/api/documentation/documents', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(documentData),
      });
      
      if (!response.ok) {
        throw new Error('Failed to create document');
      }
      
      const newDocument = await response.json();
      console.log('Document created successfully:', newDocument);
      
      // Update the documents list
      setDocuments(prev => [newDocument, ...prev]);
      
      // Update the category count
      setCategories(prev => 
        prev.map(category => 
          category.id === newDocument.category_id 
            ? { ...category, count: category.count + 1 } 
            : category
        )
      );
      
      toast.success('Document created successfully');
      resetForm();
      setNewDocumentDialogOpen(false);
    } catch (error) {
      console.error('Error creating document:', error);
      toast.error('Failed to create document');
    }
  };
  
  // Update document
  const updateDocument = async () => {
    try {
      if (!documentToEdit) {
        toast.error('No document selected for editing');
        return;
      }
      
      let fileData = null;
      
      // Upload file if selected
      if (selectedFile) {
        console.log('Uploading new file for document update');
        fileData = await uploadFile();
        if (!fileData) {
          toast.error('Failed to upload file');
          return;
        }
      } else {
        console.log('No new file selected for document update');
      }
      
      // Prepare document data with file information if available
      const documentData = {
        id: documentToEdit.id,
        ...formData,
        // If we have a new file, use its data
        ...(fileData && {
          fileName: fileData.fileName || selectedFile?.name,
          fileUrl: fileData.fileUrl,
          fileSize: fileData.fileSize || selectedFile?.size,
          fileType: fileData.fileType || selectedFile?.type
        }),
        // If no new file and document already has a file, preserve existing file data
        ...(!fileData && documentToEdit.fileName && {
          fileName: documentToEdit.fileName,
          fileUrl: documentToEdit.fileUrl,
          fileSize: documentToEdit.fileSize,
          fileType: documentToEdit.fileType
        })
      };
      
      console.log('Sending updated document data to API:', documentData);
      
      const response = await fetch('/api/documentation/documents', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(documentData),
      });
      
      if (!response.ok) {
        throw new Error('Failed to update document');
      }
      
      const updatedDocument = await response.json();
      console.log('Document updated successfully:', updatedDocument);
      
      // Update the documents list
      setDocuments(prev => 
        prev.map(doc => 
          doc.id === updatedDocument.id ? updatedDocument : doc
        )
      );
      
      // If category changed, update category counts
      if (documentToEdit.category_id !== updatedDocument.category_id) {
        setCategories(prev => 
          prev.map(category => {
            if (category.id === documentToEdit.category_id) {
              return { ...category, count: category.count - 1 };
            }
            if (category.id === updatedDocument.category_id) {
              return { ...category, count: category.count + 1 };
            }
            return category;
          })
        );
      }
      
      toast.success('Document updated successfully');
      setDocumentToEdit(null);
      resetForm();
      setEditDocumentDialogOpen(false);
    } catch (error) {
      console.error('Error updating document:', error);
      toast.error('Failed to update document');
    }
  };
  
  // Delete document
  const deleteDocument = async (id: string, categoryId: string) => {
    if (!confirm('Are you sure you want to delete this document?')) {
      return;
    }
    
    try {
      const response = await fetch(`/api/documentation/documents?id=${id}`, {
        method: 'DELETE',
      });
      
      if (!response.ok) {
        throw new Error('Failed to delete document');
      }
      
      // Remove document from list
      setDocuments(prev => prev.filter(doc => doc.id !== id));
      
      // Update category count
      setCategories(prev => 
        prev.map(category => 
          category.id === categoryId 
            ? { ...category, count: category.count - 1 } 
            : category
        )
      );
      
      toast.success('Document deleted successfully');
    } catch (error) {
      console.error('Error deleting document:', error);
      toast.error('Failed to delete document');
    }
  };
  
  // Download document
  const downloadDocument = (doc: DocumentItem) => {
    try {
      if (!doc.fileUrl) {
        toast.error('No file available for download');
        return;
      }
      
      // Create an anchor element and trigger download
      const anchor = document.createElement('a');
      anchor.href = doc.fileUrl;
      anchor.download = doc.fileName || 'document';
      anchor.target = '_blank';
      document.body.appendChild(anchor);
      anchor.click();
      document.body.removeChild(anchor);
    } catch (error) {
      console.error('Error downloading document:', error);
      toast.error('Failed to download document');
    }
  };
  
  // Format date for display
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'short', 
      day: 'numeric' 
    });
  };

  if (loading) {
    return <DocumentsTabSkeleton />;
  }

  if (error) {
    return (
      <Card>
        <CardContent className="flex flex-col items-center justify-center py-10 text-center">
          <Info className="h-10 w-10 text-red-300 mb-3" />
          <h3 className="font-medium mb-1">Error loading documents</h3>
          <p className="text-sm text-muted-foreground max-w-md">
            {error}
          </p>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-4">
      <Card>
        <CardHeader className="flex flex-row items-center justify-between">
          <div>
            <CardTitle>Hospital Documents - {currentHospital?.name}</CardTitle>
          <CardDescription>
              All documents {filterCategory ? `in category: ${categories.find(c => c.id === filterCategory)?.name}` : ''}
          </CardDescription>
          </div>
          <Button onClick={() => setNewDocumentDialogOpen(true)}>
            <Plus className="h-4 w-4 mr-2" />
            Add Document
          </Button>
        </CardHeader>
        
        <CardContent>
          {filterCategory && (
            <div className="mb-4">
              <button 
                onClick={() => setFilterCategory(null)}
                className="text-blue-600 text-sm hover:underline flex items-center gap-1"
              >
                ← Back to all documents
              </button>
            </div>
          )}
          
          <div className="space-y-4">
            {documents.length > 0 ? (
              documents.map(doc => (
                <div 
                  key={doc.id} 
                  className="flex items-center justify-between p-3 border-b border-gray-100 last:border-0 hover:bg-gray-50"
                >
                  <div className="flex items-center gap-3">
                    <FileText className="h-5 w-5 text-gray-400" />
                    <div>
                      <div className="font-medium">{doc.title}</div>
                      <div className="text-sm text-gray-500">
                        {doc.category.name} • Last updated on {formatDate(doc.updatedAt)} by {doc.updatedBy}
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center gap-3">
                    <Badge className={
                      doc.status === 'active' 
                      ? 'bg-green-100 text-green-800 hover:bg-green-100' 
                      : doc.status === 'review'
                        ? 'bg-amber-100 text-amber-800 hover:bg-amber-100'
                        : 'bg-gray-100 text-gray-800 hover:bg-gray-100'
                    }>
                      {doc.status === 'active' ? 'Active' : doc.status === 'review' ? 'Review Needed' : 'Archived'}
                    </Badge>
                    <button 
                      className="text-blue-600 hover:text-blue-800"
                      title="Edit document"
                      onClick={() => openEditDialog(doc)}
                    >
                      <Pencil className="h-4 w-4" />
                    </button>
                    <button 
                      className="text-blue-600 hover:text-blue-800"
                      title="Download document"
                      onClick={() => downloadDocument(doc)}
                    >
                      <Download className="h-4 w-4" />
                    </button>
                    <button 
                      className="text-red-600 hover:text-red-800"
                      title="Delete document"
                      onClick={() => deleteDocument(doc.id, doc.category_id)}
                    >
                      <Trash2 className="h-4 w-4" />
                    </button>
                  </div>
                </div>
              ))
            ) : (
              <div className="text-center py-8 text-gray-500">
                <div className="flex flex-col items-center justify-center">
                  <Info className="h-10 w-10 text-gray-300 mb-2" />
                  <h3 className="font-medium mb-1">No documents found</h3>
                  <p className="text-sm text-gray-500">
                    {filterCategory 
                      ? `No documents in this category.` 
                      : 'No documents are available. Click "Add Document" to create one.'}
                  </p>
                </div>
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {!filterCategory && (
        <Card>
          <CardHeader>
            <CardTitle>Document Categories</CardTitle>
            <CardDescription>
              Browse documents by category
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
              {categories.map(category => (
                <Card 
                  key={category.id} 
                  className="overflow-hidden cursor-pointer hover:shadow-md transition-shadow"
                  onClick={() => setFilterCategory(category.id)}
                >
                  <CardHeader className="pb-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <FileText className="h-5 w-5 text-blue-600" />
                        <CardTitle className="text-base font-medium">{category.name}</CardTitle>
                      </div>
                      <span className="text-sm font-medium bg-blue-100 text-blue-700 px-2 py-1 rounded-full">
                        {category.count}
                      </span>
                    </div>
                  </CardHeader>
                  <CardContent className="pt-0 pb-3">
                    <p className="text-xs text-gray-500">
                      {category.description}
                    </p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </CardContent>
        </Card>
      )}
      
      {/* New Document Dialog */}
      <Dialog open={newDocumentDialogOpen} onOpenChange={setNewDocumentDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Add New Document</DialogTitle>
            <DialogDescription>
              Create a new document for your hospital.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="title">Document Title</Label>
              <Input 
                id="title" 
                name="title" 
                placeholder="Enter document title" 
                value={formData.title}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="file">Upload Document</Label>
              <div className="grid w-full max-w-sm items-center gap-1.5">
                <Input
                  id="file"
                  type="file"
                  onChange={handleFileChange}
                  className="cursor-pointer border-2 border-dashed border-gray-300 p-4"
                />
                {isUploading && (
                  <div className="w-full h-2 bg-gray-200 rounded-full mt-2">
                    <div 
                      className="h-full bg-blue-600 rounded-full" 
                      style={{ width: `${uploadProgress}%` }}
                    />
                  </div>
                )}
                {selectedFile && (
                  <p className="text-sm text-green-600 font-medium mt-1">
                    File selected: {selectedFile.name} ({Math.round(selectedFile.size / 1024)} KB)
                  </p>
                )}
                {!selectedFile && (
                  <p className="text-xs text-muted-foreground mt-1">
                    Select a file to upload (PDF, Word, Excel, etc.)
                  </p>
                )}
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="department">Department</Label>
              <Input 
                id="department" 
                name="department" 
                placeholder="Clinical Operations" 
                value={formData.department}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="category">Category</Label>
              <Select 
                value={formData.category_id} 
                onValueChange={(value) => handleSelectChange('category_id', value)}
              >
                <SelectTrigger id="category">
                  <SelectValue placeholder="Select a category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground mt-1">
                {formData.category_id ? `Selected category ID: ${formData.category_id}` : 'No category selected'}
              </p>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="status">Status</Label>
              <Select 
                value={formData.status} 
                onValueChange={(value) => handleSelectChange('status', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="review">Review Needed</SelectItem>
                  <SelectItem value="archived">Archived</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => {
              resetForm();
              setNewDocumentDialogOpen(false);
            }}>
              Cancel
            </Button>
            <Button onClick={createDocument}>
              Create Document
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
      
      {/* Edit Document Dialog */}
      <Dialog open={editDocumentDialogOpen} onOpenChange={setEditDocumentDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Document</DialogTitle>
            <DialogDescription>
              Update the document information.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="edit-title">Document Title</Label>
              <Input 
                id="edit-title" 
                name="title" 
                placeholder="Enter document title" 
                value={formData.title}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-file">Upload Document</Label>
              <div className="grid w-full max-w-sm items-center gap-1.5">
                <Input
                  id="edit-file"
                  type="file"
                  onChange={handleFileChange}
                  className="cursor-pointer border-2 border-dashed border-gray-300 p-4"
                />
                {isUploading && (
                  <div className="w-full h-2 bg-gray-200 rounded-full mt-2">
                    <div 
                      className="h-full bg-blue-600 rounded-full" 
                      style={{ width: `${uploadProgress}%` }}
                    />
                  </div>
                )}
                {selectedFile && (
                  <p className="text-sm text-green-600 font-medium mt-1">
                    File selected: {selectedFile.name} ({Math.round(selectedFile.size / 1024)} KB)
                  </p>
                )}
                {!selectedFile && documentToEdit?.fileName && (
                  <p className="text-sm text-blue-600 mt-1">
                    Current file: {documentToEdit.fileName}
                  </p>
                )}
                {!selectedFile && !documentToEdit?.fileName && (
                  <p className="text-xs text-muted-foreground mt-1">
                    Select a file to upload (PDF, Word, Excel, etc.)
                  </p>
                )}
              </div>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-department">Department</Label>
              <Input 
                id="edit-department" 
                name="department" 
                placeholder="Clinical Operations" 
                value={formData.department}
                onChange={handleInputChange}
              />
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-category">Category</Label>
              <Select 
                value={formData.category_id} 
                onValueChange={(value) => handleSelectChange('category_id', value)}
              >
                <SelectTrigger id="edit-category">
                  <SelectValue placeholder="Select a category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground mt-1">
                {formData.category_id ? `Selected category ID: ${formData.category_id}` : 'No category selected'}
              </p>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="edit-status">Status</Label>
              <Select 
                value={formData.status} 
                onValueChange={(value) => handleSelectChange('status', value)}
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select a status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="review">Review Needed</SelectItem>
                  <SelectItem value="archived">Archived</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <DialogFooter>
            <Button variant="outline" onClick={() => setEditDocumentDialogOpen(false)}>
              Cancel
            </Button>
            <Button onClick={updateDocument}>
              Update Document
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
};

const DocumentsTabSkeleton = () => {
  return (
    <div className="space-y-4">
      <Card>
        <CardHeader>
          <Skeleton className="h-7 w-64" />
          <Skeleton className="h-4 w-48 mt-2" />
        </CardHeader>
        
        <CardContent>
          <div className="space-y-4">
            {[1, 2, 3, 4, 5].map(i => (
              <div key={i} className="flex items-center justify-between p-3 border-b border-gray-100 last:border-0">
                <div className="flex items-center gap-3">
                  <Skeleton className="h-5 w-5 rounded-full" />
                  <div>
                    <Skeleton className="h-5 w-52 mb-1" />
                    <Skeleton className="h-4 w-40" />
                  </div>
                </div>
                <div className="flex items-center gap-3">
                  <Skeleton className="h-6 w-16 rounded-full" />
                  <Skeleton className="h-4 w-4 rounded-full" />
                  <Skeleton className="h-4 w-4 rounded-full" />
                  <Skeleton className="h-4 w-4 rounded-full" />
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <Skeleton className="h-7 w-48" />
          <Skeleton className="h-4 w-56 mt-2" />
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
            {[1, 2, 3, 4, 5, 6].map(i => (
              <Card key={i} className="overflow-hidden">
                <CardHeader className="pb-2">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Skeleton className="h-5 w-5 rounded-full" />
                      <Skeleton className="h-5 w-24" />
                    </div>
                    <Skeleton className="h-6 w-8 rounded-full" />
                  </div>
                </CardHeader>
                <CardContent className="pt-0 pb-3">
                  <Skeleton className="h-4 w-full" />
                </CardContent>
              </Card>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default DocumentsTab; 