import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Separator } from "@/components/ui/separator"
import { Badge } from "@/components/ui/badge"
import { format } from "date-fns"
import { AuditFinding } from "@/hooks/useAuditFindings"
import { Skeleton } from "@/components/ui/skeleton"
import { AlertCircle, CheckCircle2 } from "lucide-react"
import Link from "next/link"
import { cn } from "@/lib/utils"
import { useFindingOptions } from "@/hooks/useFindingOptions"

interface DynamicAuditFindingsProps {
  findings: AuditFinding[];
  loading?: boolean;
  className?: string;
}

export const DynamicAuditFindings = ({ findings, loading = false, className }: DynamicAuditFindingsProps) => {
  const { getSeverityColorClass, getStatusColorClass } = useFindingOptions();

  const getSeverityBadge = (severity: string) => {
    // Handle special cases for existing badge variants
    if (severity === 'critical') {
      return <Badge variant="destructive">Critical</Badge>;
    } else if (severity === 'low') {
      return <Badge variant="outline" className="text-blue-600">Low</Badge>;
    } else {
      const colorClass = getSeverityColorClass(severity);
      return <Badge className={colorClass}>{severity.charAt(0).toUpperCase() + severity.slice(1)}</Badge>;
    }
  };

  const getStatusBadge = (status: string) => {
    const colorClass = getStatusColorClass(status);
    const label = status === 'in_progress' ? 'In Progress' : status.charAt(0).toUpperCase() + status.slice(1);
    return <Badge variant="outline" className={colorClass}>{label}</Badge>;
  };

  if (loading) {
    return (
      <Card className={cn("h-full", className)}>
        <CardHeader className="pb-2">
          <CardTitle className="text-base">Recent Audit Findings</CardTitle>
          <CardDescription>Latest findings across departments</CardDescription>
        </CardHeader>
        <CardContent className="pt-0">
          <div className="space-y-4">
            {[1, 2, 3].map(i => (
              <div key={i}>
                {i > 1 && <Separator className="my-4" />}
                <div>
                  <Skeleton className="h-4 w-3/4 mb-2" />
                  <div className="flex items-center justify-between">
                    <Skeleton className="h-3 w-1/3" />
                    <Skeleton className="h-5 w-16 rounded-full" />
                  </div>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className={cn("h-full", className)}>
      <CardHeader className="pb-2">
        <CardTitle className="text-base">Recent Audit Findings</CardTitle>
        <CardDescription>Latest findings across departments</CardDescription>
      </CardHeader>
      <CardContent className="pt-0">
        {findings.length === 0 ? (
          <div className="flex flex-col items-center justify-center py-6 text-center text-gray-500">
            <AlertCircle className="h-10 w-10 mb-2 text-gray-400" />
            <p>No findings available</p>
          </div>
        ) : (
          <div className="space-y-4">
            {findings.map((finding, index) => (
              <div key={finding.id}>
                {index > 0 && <Separator className="my-4" />}
                <div>
                  <div className="flex items-center justify-between mb-1">
                    <h4 className="font-medium text-sm">{finding.title}</h4>
                    {getSeverityBadge(finding.severity)}
                  </div>
                  <div className="flex items-center justify-between text-sm">
                    <span className="text-gray-500">
                      {finding.department} • {format(new Date(finding.date_reported), 'MMM dd, yyyy')}
                    </span>
                    {getStatusBadge(finding.status)}
                  </div>
                </div>
              </div>
            ))}
            <div className="pt-4 mt-2 border-t text-center">
              <Link 
                href="/findings" 
                className="inline-flex items-center text-xs text-blue-600 hover:text-blue-800"
              >
                <CheckCircle2 className="h-3 w-3 mr-1" />
                View all findings
              </Link>
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}; 