'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Progress } from '@/components/ui/progress'
import { Badge } from '@/components/ui/badge'
import { useApiWithHospital } from '@/hooks/useApiWithHospital'
import { useHospital } from '@/lib/context/HospitalContext'
import { useDashboardData } from '@/hooks/useDashboardData'

interface ComplianceItem {
  id: number
  name: string
  category: string
  compliance: number
  required: number
}

export function ComplianceStats() {
  const { metrics, loading, error } = useDashboardData()
  const { currentHospital } = useHospital()
  
  const getStatusColor = (value: number, required: number) => {
    if (value >= required) return 'bg-green-500' // Equal to or greater than required
    if (value >= required * 0.8) return 'bg-yellow-500' // Within 80% of required threshold (close but not quite there)
    return 'bg-red-500' // Much less than required
  }

  const getStatusText = (value: number, required: number) => {
    if (value >= required) return 'Compliant'
    if (value >= required * 0.8) return 'At Risk'
    return 'Non-Compliant'
  }
  
  const getProgressColorClass = (value: number, required: number) => {
    if (value >= required) return '[&>div]:bg-green-500' // Equal to or greater than required
    if (value >= required * 0.8) return '[&>div]:bg-yellow-500' // Within 80% of required threshold
    return '[&>div]:bg-red-500' // Much less than required
  }

  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Compliance Overview</CardTitle>
          <CardDescription>Loading compliance data...</CardDescription>
        </CardHeader>
      </Card>
    )
  }

  if (error) {
    return (
      <Card>
        <CardHeader>
          <CardTitle>Compliance Overview</CardTitle>
          <CardDescription className="text-red-500">Error loading data</CardDescription>
        </CardHeader>
      </Card>
    )
  }

  // Get standards data from metrics
  const standardsData = metrics.standardsData || []

  return (
    <Card>
      <CardHeader>
        <CardTitle>Compliance Overview</CardTitle>
        <CardDescription>
          {currentHospital?.name || 'Hospital'} compliance status across standards
        </CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {standardsData.map((item: ComplianceItem) => (
            <div key={item.id} className="grid gap-2">
              <div className="flex items-center justify-between">
                <div className="font-medium">{item.name}</div>
                <Badge
                  className={getStatusColor(item.compliance, item.required)}
                  variant="outline"
                >
                  {getStatusText(item.compliance, item.required)}
                </Badge>
              </div>
              <div className="flex items-center gap-2">
                <Progress
                  value={item.compliance}
                  max={100}
                  className={`h-2 ${getProgressColorClass(item.compliance, item.required)}`}
                />
                <span className="w-12 text-sm text-gray-500">
                  {item.compliance}%
                </span>
              </div>
              <div className="text-xs text-gray-500">
                Required: {item.required}% | Category: {item.category}
              </div>
            </div>
          ))}

          {standardsData.length === 0 && (
            <div className="text-center text-gray-500 py-4">
              No compliance standards available for this hospital
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  )
} 