import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { cn } from "@/lib/utils";
import { BarChart as BarChartIcon, FileText } from "lucide-react";
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, Legend } from "recharts";
import { useEffect, useState } from "react";

interface ComplianceChartProps {
  data?: any[];
  className?: string;
}

const CustomTooltip = ({ active, payload, label }: any) => {
  if (active && payload && payload.length) {
    return (
      <div className="custom-tooltip bg-white p-3 border border-gray-200 shadow-md rounded-md">
        <p className="font-medium text-sm">{`${label}`}</p>
        <p className="text-blue-600 text-sm">{`Compliance: ${payload[0].value}%`}</p>
        <p className="text-amber-600 text-sm">{`Required: ${payload[1].value}%`}</p>
      </div>
    );
  }
  return null;
};

export const ComplianceChart = ({ data, className }: ComplianceChartProps) => {
  const [complianceData, setComplianceData] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  
  useEffect(() => {
    const fetchData = async () => {
      try {
        if (data) {
          setComplianceData(data);
          setLoading(false);
          return;
        }
        
        const response = await fetch('/api/compliance/standards');
        if (!response.ok) {
          throw new Error('Failed to fetch compliance data');
        }
        
        const fetchedData = await response.json();
        setComplianceData(fetchedData);
      } catch (error) {
        console.error('Error fetching compliance data:', error);
      } finally {
        setLoading(false);
      }
    };
    
    fetchData();
  }, [data]);
  
  if (loading) {
    return (
      <Card className="h-full">
        <CardHeader>
          <CardTitle>Compliance Metrics</CardTitle>
          <CardDescription>Current compliance level vs. required threshold</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {[1, 2, 3, 4, 5].map((_, index) => (
              <div key={index} className="space-y-2">
                <div className="flex items-center justify-between animate-pulse">
                  <div className="w-1/3 h-4 bg-gray-200 rounded"></div>
                  <div className="w-1/6 h-4 bg-gray-200 rounded"></div>
                </div>
                <div className="h-2 bg-gray-200 rounded-full w-full"></div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }
  
  // Display a subset of standards to keep the chart manageable
  const displayData = complianceData.slice(0, 8);
  
  return (
    <Card className={cn("h-full", className)}>
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <div className="flex items-center space-x-2">
          <BarChartIcon className="h-5 w-5 text-blue-600" />
          <CardTitle>Compliance by Regulatory Standard</CardTitle>
        </div>
        <button className="flex items-center text-blue-600 text-sm font-medium">
          <FileText className="h-4 w-4 mr-1" />
          Export
        </button>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="h-[300px] mt-4">
          <ResponsiveContainer width="100%" height="100%">
            <BarChart
              data={displayData}
              margin={{
                top: 5,
                right: 30,
                left: 20,
                bottom: 25,
              }}
              barGap={0}
            >
              <CartesianGrid strokeDasharray="3 3" vertical={false} />
              <XAxis 
                dataKey="name" 
                tick={{ fontSize: 12 }}
                axisLine={{ stroke: '#E5E7EB' }}
                tickLine={false}
                angle={-45}
                textAnchor="end"
                height={60}
              />
              <YAxis 
                tickFormatter={(value) => `${value}%`}
                axisLine={{ stroke: '#E5E7EB' }}
                tickLine={false}
                tick={{ fontSize: 12 }}
                domain={[0, 100]}
              />
              <Tooltip content={<CustomTooltip />} />
              <Legend 
                verticalAlign="top" 
                height={36} 
                iconType="circle"
                iconSize={8}
              />
              <Bar 
                name="Current Compliance" 
                dataKey="compliance" 
                fill="#3381FF" 
                radius={[4, 4, 0, 0]} 
              />
              <Bar 
                name="Required Threshold" 
                dataKey="required" 
                fill="#FFC107" 
                radius={[4, 4, 0, 0]} 
              />
            </BarChart>
          </ResponsiveContainer>
        </div>
      </CardContent>
    </Card>
  );
};

export default ComplianceChart; 