import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useStandardsData, addHospitalIdHeader } from "@/hooks/useComplianceData";
import { Skeleton } from "@/components/ui/skeleton";
import { format } from "date-fns";
import { Edit, Plus, Trash2, X } from "lucide-react";
import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { toast } from "sonner";
import { useSession } from "next-auth/react";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";

const StandardsTab = () => {
  const { standards, loading, error, refetch, deleteStandard, updateStandard, deleting, updating } = useStandardsData();
  const { data: session } = useSession();
  const [showForm, setShowForm] = useState(false);
  const [formData, setFormData] = useState({
    standard: "",
    category: "",
    compliance: "",
    required_threshold: "90",
  });
  const [submitting, setSubmitting] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [standardToDelete, setStandardToDelete] = useState<number | null>(null);
  const [isEditing, setIsEditing] = useState(false);
  const [editingStandardId, setEditingStandardId] = useState<number | null>(null);
  
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };
  
  const resetForm = () => {
    setFormData({
      standard: "",
      category: "",
      compliance: "",
      required_threshold: "90"
    });
    setIsEditing(false);
    setEditingStandardId(null);
    setShowForm(false);
  };
  
  const confirmDelete = (id: number) => {
    setStandardToDelete(id);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (standardToDelete === null) return;
    
    try {
      const success = await deleteStandard(standardToDelete);
      if (success) {
        toast.success("Standard deleted successfully");
      }
    } catch (error) {
      console.error("Error deleting standard:", error);
      toast.error("Failed to delete standard");
    } finally {
      setDeleteDialogOpen(false);
      setStandardToDelete(null);
    }
  };
  
  const handleEditStandard = (standard: any) => {
    setFormData({
      standard: standard.standard,
      category: standard.category,
      compliance: standard.compliance.toString(),
      required_threshold: standard.required_threshold.toString()
    });
    setIsEditing(true);
    setEditingStandardId(standard.id);
    setShowForm(true);
  };
  
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate inputs
    if (!formData.standard || !formData.category || !formData.compliance) {
      toast.error("All fields are required");
      return;
    }
    
    const compliance = parseFloat(formData.compliance);
    const requiredThreshold = parseFloat(formData.required_threshold);
    
    if (isNaN(compliance)) {
      toast.error("Compliance value must be a number");
      return;
    }
    
    if (compliance < 0 || compliance > 100) {
      toast.error("Compliance value must be between 0 and 100");
      return;
    }
    
    if (isNaN(requiredThreshold)) {
      toast.error("Required threshold must be a number");
      return;
    }
    
    if (requiredThreshold < 0 || requiredThreshold > 100) {
      toast.error("Required threshold must be between 0 and 100");
      return;
    }
    
    setSubmitting(true);
    
    try {
      const isSuperAdmin = session?.user?.role === 'super_admin';
      
      if (isEditing && editingStandardId) {
        // Update existing standard
        const updateData = {
          standard: formData.standard,
          category: formData.category,
          compliance: compliance,
          required_threshold: requiredThreshold
        };
        
        const success = await updateStandard(editingStandardId, updateData);
        
        if (success) {
          toast.success("Standard updated successfully");
          resetForm();
        }
      } else {
        // Create new standard
        const headers = addHospitalIdHeader({ 
          'Content-Type': 'application/json',
        }, Boolean(isSuperAdmin));
        
        const response = await fetch("/api/compliance/standards", {
          method: "POST",
          headers,
          body: JSON.stringify({
            standard: formData.standard,
            category: formData.category,
            compliance: compliance,
            required_threshold: requiredThreshold
          }),
        });
        
        if (!response.ok) {
          try {
            const errorData = await response.json();
            console.error("API Error:", errorData);
            throw new Error(errorData.error || errorData.message || "Failed to add standard");
          } catch (jsonError) {
            console.error("Error parsing error response:", jsonError);
            throw new Error("Failed to add standard: Server error");
          }
        }
        
        toast.success("Standard added successfully");
        resetForm();
        refetch();
      }
    } catch (err) {
      console.error(isEditing ? "Error updating standard:" : "Error adding standard:", err);
      toast.error((err as Error).message);
    } finally {
      setSubmitting(false);
    }
  };
  
  if (loading) {
    return <StandardsTabSkeleton />;
  }
  
  if (error) {
    return <div className="p-4 text-red-500">Error loading standards data: {error}</div>;
  }
  
  const getComplianceClass = (compliance: number, requiredThreshold: number) => {
    if (compliance >= requiredThreshold) return "bg-green-100 text-green-800";
    if (compliance >= requiredThreshold * 0.9) return "bg-amber-100 text-amber-800"; // Within 90% of threshold
    return "bg-red-100 text-red-800";
  };

  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <div>
          <CardTitle>Regulatory Standards</CardTitle>
          <CardDescription>Manage applicable standards and regulations</CardDescription>
        </div>
        <Button 
          variant="outline" 
          size="sm" 
          onClick={() => {
            if (showForm && isEditing) {
              resetForm();
            } else {
              setShowForm(!showForm);
            }
          }}
          className="flex items-center gap-1"
        >
          {showForm ? <X size={16} /> : <Plus size={16} />}
          {showForm ? "Cancel" : "Add Standard"}
        </Button>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {showForm && (
            <Card className="border-dashed border-2 border-blue-200 bg-blue-50 mb-4">
              <CardContent className="pt-4">
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div>
                      <label htmlFor="standard" className="block text-sm font-medium mb-1">Standard Name</label>
                      <Input
                        id="standard"
                        name="standard"
                        placeholder="e.g. HIPAA Privacy Rule"
                        value={formData.standard}
                        onChange={handleInputChange}
                        className="w-full"
                      />
                    </div>
                    <div>
                      <label htmlFor="category" className="block text-sm font-medium mb-1">Category</label>
                      <Input
                        id="category"
                        name="category"
                        placeholder="e.g. Data Privacy"
                        value={formData.category}
                        onChange={handleInputChange}
                        className="w-full"
                      />
                    </div>
                    <div>
                      <label htmlFor="compliance" className="block text-sm font-medium mb-1">Compliance Level (%)</label>
                      <Input
                        id="compliance"
                        name="compliance"
                        type="number"
                        min="0"
                        max="100"
                        placeholder="e.g. 85"
                        value={formData.compliance}
                        onChange={handleInputChange}
                        className="w-full"
                      />
                    </div>
                    <div>
                      <label htmlFor="required_threshold" className="block text-sm font-medium mb-1">Required Threshold (%)</label>
                      <Input
                        id="required_threshold"
                        name="required_threshold"
                        type="number"
                        min="0"
                        max="100"
                        placeholder="e.g. 90"
                        value={formData.required_threshold}
                        onChange={handleInputChange}
                        className="w-full"
                      />
                    </div>
                  </div>
                  <div className="flex justify-end">
                    <Button 
                      type="submit" 
                      disabled={submitting || updating}
                      className="bg-blue-600 hover:bg-blue-700"
                    >
                      {submitting || updating ? (isEditing ? "Updating..." : "Adding...") : (isEditing ? "Update Standard" : "Add Standard")}
                    </Button>
                  </div>
                </form>
              </CardContent>
            </Card>
          )}

          <div className="border rounded-md">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="text-left p-3 text-sm">Standard</th>
                  <th className="text-left p-3 text-sm">Category</th>
                  <th className="text-left p-3 text-sm">Compliance Level</th>
                  <th className="text-left p-3 text-sm">Required Threshold</th>
                  <th className="text-left p-3 text-sm">Last Assessed</th>
                  <th className="text-left p-3 text-sm">Actions</th>
                </tr>
              </thead>
              <tbody>
                {standards.map((item) => (
                  <tr key={item.id} className="border-t">
                    <td className="p-3 font-medium">{item.standard}</td>
                    <td className="p-3">{item.category}</td>
                    <td className="p-3">
                      <span className={`px-2 py-1 rounded-full text-xs ${getComplianceClass(item.compliance, item.required_threshold)}`}>
                        {item.compliance}% Compliant
                      </span>
                    </td>
                    <td className="p-3">
                      {item.required_threshold}%
                    </td>
                    <td className="p-3">{format(new Date(item.last_assessed), 'MMM dd, yyyy')}</td>
                    <td className="p-3 flex gap-2">
                      <button 
                        className="text-blue-600 text-sm hover:underline flex items-center gap-1"
                        onClick={() => handleEditStandard(item)}
                      >
                        <Edit size={14} />
                        Edit
                      </button>
                      <button 
                        className="text-red-500 text-sm hover:underline flex items-center gap-1" 
                        onClick={() => confirmDelete(item.id)}
                      >
                        <Trash2 size={14} />
                        Delete
                      </button>
                    </td>
                  </tr>
                ))}
                
                {standards.length === 0 && (
                  <tr>
                    <td colSpan={5} className="p-4 text-center text-gray-500">
                      No standards data available
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>
      </CardContent>

      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Standard</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete this standard? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel disabled={deleting}>Cancel</AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleDelete} 
              disabled={deleting}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleting ? "Deleting..." : "Delete"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </Card>
  );
};

const StandardsTabSkeleton = () => {
  return (
    <Card>
      <CardHeader>
        <CardTitle>Regulatory Standards</CardTitle>
        <CardDescription>Manage applicable standards and regulations</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div className="border rounded-md">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="text-left p-3 text-sm">Standard</th>
                  <th className="text-left p-3 text-sm">Category</th>
                  <th className="text-left p-3 text-sm">Compliance Level</th>
                  <th className="text-left p-3 text-sm">Required Threshold</th>
                  <th className="text-left p-3 text-sm">Last Assessed</th>
                  <th className="text-left p-3 text-sm">Actions</th>
                </tr>
              </thead>
              <tbody>
                {[1, 2, 3].map((i) => (
                  <tr key={i} className="border-t">
                    <td className="p-3"><Skeleton className="h-5 w-24" /></td>
                    <td className="p-3"><Skeleton className="h-5 w-20" /></td>
                    <td className="p-3"><Skeleton className="h-5 w-20" /></td>
                    <td className="p-3"><Skeleton className="h-5 w-20" /></td>
                    <td className="p-3"><Skeleton className="h-5 w-24" /></td>
                    <td className="p-3"><Skeleton className="h-5 w-16" /></td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default StandardsTab; 