import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Info, Plus, Trash2, X } from "lucide-react";
import { toast } from "sonner";
import { useComplianceData } from "@/hooks/useComplianceData";
import ComplianceChart from "@/components/dashboard/ComplianceChart";
import { Skeleton } from "@/components/ui/skeleton";
import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { AlertDialog, AlertDialogAction, AlertDialogCancel, AlertDialogContent, AlertDialogDescription, AlertDialogFooter, AlertDialogHeader, AlertDialogTitle } from "@/components/ui/alert-dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useSession } from "next-auth/react";

const ComplianceOverview = () => {
  const { data: session } = useSession();
  const { complianceData, loading, error, refetch, deleteComplianceItem, deleting } = useComplianceData();
  const [showForm, setShowForm] = useState(false);
  const [formData, setFormData] = useState({
    name: "",
    compliance: "",
    required: ""
  });
  const [submitting, setSubmitting] = useState(false);
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [itemToDelete, setItemToDelete] = useState<number | null>(null);
  const [hospitals, setHospitals] = useState<{id: number, name: string}[]>([]);
  const [selectedHospitalId, setSelectedHospitalId] = useState<string>("");
  const [loadingHospitals, setLoadingHospitals] = useState(false);
  
  const isSuperAdmin = session?.user?.role === 'super_admin';

  // Fetch available hospitals for super admin
  useEffect(() => {
    if (isSuperAdmin) {
      const fetchHospitals = async () => {
        setLoadingHospitals(true);
        try {
          const response = await fetch('/api/admin/hospitals');
          if (!response.ok) {
            throw new Error('Failed to fetch hospitals');
          }
          const data = await response.json();
          setHospitals(data);
          
          // Get hospital ID from cookie or localStorage
          const cookieHospitalId = document.cookie
            .split('; ')
            .find(row => row.startsWith('selectedHospitalId='))
            ?.split('=')[1];
            
          const savedHospitalId = cookieHospitalId || localStorage.getItem('selectedHospitalId');
          
          if (savedHospitalId && data.some((h: any) => h.id.toString() === savedHospitalId)) {
            setSelectedHospitalId(savedHospitalId);
          } else if (data.length > 0) {
            // Set first hospital as default
            setSelectedHospitalId(data[0].id.toString());
            // Save to cookie and localStorage
            document.cookie = `selectedHospitalId=${data[0].id}; path=/; max-age=2592000`;
            localStorage.setItem('selectedHospitalId', data[0].id.toString());
          }
        } catch (err) {
          console.error('Error fetching hospitals:', err);
          toast.error('Failed to load hospitals');
        } finally {
          setLoadingHospitals(false);
        }
      };
      
      fetchHospitals();
    }
  }, [isSuperAdmin]);
  
  // Handle hospital selection change
  const handleHospitalChange = (hospitalId: string) => {
    setSelectedHospitalId(hospitalId);
    // Save selection to cookie and localStorage
    document.cookie = `selectedHospitalId=${hospitalId}; path=/; max-age=2592000`;
    localStorage.setItem('selectedHospitalId', hospitalId);
    // Refetch data with new hospital
    setTimeout(() => {
      refetch();
    }, 100);
  };

  const handleComplianceAction = (standard: string) => {
    toast.success(`Remediation plan initiated for ${standard}`);
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const confirmDelete = (id: number) => {
    setItemToDelete(id);
    setDeleteDialogOpen(true);
  };

  const handleDelete = async () => {
    if (itemToDelete === null) return;
    
    try {
      const success = await deleteComplianceItem(itemToDelete);
      if (success) {
        toast.success("Compliance item deleted successfully");
      }
    } catch (error) {
      console.error("Error deleting compliance item:", error);
      toast.error("Failed to delete compliance item");
    } finally {
      setDeleteDialogOpen(false);
      setItemToDelete(null);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validate inputs
    if (!formData.name || !formData.compliance || !formData.required) {
      toast.error("All fields are required");
      return;
    }
    
    const compliance = parseFloat(formData.compliance);
    const required = parseFloat(formData.required);
    
    if (isNaN(compliance) || isNaN(required)) {
      toast.error("Compliance and required values must be numbers");
      return;
    }
    
    if (compliance < 0 || compliance > 100 || required < 0 || required > 100) {
      toast.error("Compliance and required values must be between 0 and 100");
      return;
    }
    
    setSubmitting(true);
    
    try {
      const response = await fetch("/api/compliance", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          name: formData.name,
          compliance: compliance,
          required: required,
        }),
      });
      
      if (!response.ok) {
        throw new Error("Failed to add compliance item");
      }
      
      toast.success("Compliance item added successfully");
      setFormData({ name: "", compliance: "", required: "" });
      setShowForm(false);
      refetch();
    } catch (err) {
      console.error("Error adding compliance item:", err);
      toast.error((err as Error).message);
    } finally {
      setSubmitting(false);
    }
  };

  if (loading) {
    return <ComplianceOverviewSkeleton />;
  }

  if (error) {
    return (
      <div className="space-y-4">
        {isSuperAdmin && (
          <Card className="bg-amber-50 border-amber-200">
            <CardContent className="pt-6">
              <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
                <div>
                  <h3 className="font-medium text-amber-800">Super Admin Hospital Selection</h3>
                  <p className="text-sm text-amber-700 mb-2">
                    Please select a hospital to view its compliance data
                  </p>
                </div>
                <div className="w-full md:w-64">
                  <Select 
                    value={selectedHospitalId} 
                    onValueChange={handleHospitalChange}
                    disabled={loadingHospitals}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select a hospital" />
                    </SelectTrigger>
                    <SelectContent>
                      {hospitals.map((hospital) => (
                        <SelectItem key={hospital.id} value={hospital.id.toString()}>
                          {hospital.name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardContent>
          </Card>
        )}
        <div className="p-4 text-red-500">Error loading compliance data: {error}</div>
      </div>
    );
  }

  return (
    <div className="space-y-4">
      {isSuperAdmin && (
        <Card className="bg-blue-50 border-blue-200">
          <CardContent className="pt-6">
            <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
              <div>
                <h3 className="font-medium text-blue-800">Super Admin Hospital Selection</h3>
                <p className="text-sm text-blue-700 mb-2">
                  Viewing data for selected hospital
                </p>
              </div>
              <div className="w-full md:w-64">
                <Select 
                  value={selectedHospitalId} 
                  onValueChange={handleHospitalChange}
                  disabled={loadingHospitals}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select a hospital" />
                  </SelectTrigger>
                  <SelectContent>
                    {hospitals.map((hospital) => (
                      <SelectItem key={hospital.id} value={hospital.id.toString()}>
                        {hospital.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="flex justify-between items-center">
        <h2 className="text-xl font-semibold">Compliance Standards</h2>
        <Button 
          variant="outline" 
          size="sm" 
          onClick={() => setShowForm(!showForm)}
          className="flex items-center gap-1"
        >
          {showForm ? <X size={16} /> : <Plus size={16} />}
          {showForm ? "Cancel" : "Add Standard"}
        </Button>
      </div>

      {showForm && (
        <Card className="border-dashed border-2 border-blue-200 bg-blue-50">
          <CardContent className="pt-4">
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label htmlFor="name" className="block text-sm font-medium mb-1">Standard Name</label>
                  <Input
                    id="name"
                    name="name"
                    placeholder="e.g. HIPAA, ISO 9001"
                    value={formData.name}
                    onChange={handleInputChange}
                    className="w-full"
                  />
                </div>
                <div>
                  <label htmlFor="compliance" className="block text-sm font-medium mb-1">Current Compliance (%)</label>
                  <Input
                    id="compliance"
                    name="compliance"
                    type="number"
                    min="0"
                    max="100"
                    placeholder="e.g. 85"
                    value={formData.compliance}
                    onChange={handleInputChange}
                    className="w-full"
                  />
                </div>
                <div>
                  <label htmlFor="required" className="block text-sm font-medium mb-1">Required Compliance (%)</label>
                  <Input
                    id="required"
                    name="required"
                    type="number"
                    min="0"
                    max="100"
                    placeholder="e.g. 90"
                    value={formData.required}
                    onChange={handleInputChange}
                    className="w-full"
                  />
                </div>
              </div>
              <div className="flex justify-end">
                <Button 
                  type="submit" 
                  disabled={submitting}
                  className="bg-blue-600 hover:bg-blue-700"
                >
                  {submitting ? "Adding..." : "Add Standard"}
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4 mt-4">
        {complianceData.map((item) => (
          <Card key={item.id} className="overflow-hidden">
            <div className="flex">
              <div className={`w-1 h-full ${item.compliance >= item.required ? 'bg-green-500' : 'bg-red-500'}`}></div>
              <div className="flex-1">
                <CardHeader className="pb-2">
                  <div className="flex items-start justify-between">
                    <div>
                      <CardTitle className="text-base font-medium">{item.name}</CardTitle>
                      <CardDescription className="text-xs mt-1">Required: {item.required}%</CardDescription>
                    </div>
                    <div className="flex items-center gap-2">
                      <div className="cursor-help" title={`${item.name} requires ${item.required}% compliance level`}>
                        <Info className="h-4 w-4 text-gray-400" />
                      </div>
                      <button 
                        className="text-gray-400 hover:text-red-500 rounded-full p-1 hover:bg-gray-100"
                        onClick={() => confirmDelete(item.id)}
                        title="Delete standard"
                      >
                        <Trash2 size={14} />
                      </button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="pt-0">
                  <div className="flex items-center justify-between">
                    <span className={`px-2 py-1 rounded text-xs font-medium ${item.compliance >= item.required ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                      {item.compliance}% Compliant
                    </span>
                    <button 
                      className="text-xs text-blue-600 hover:text-blue-800"
                      onClick={() => handleComplianceAction(item.name)}
                    >
                      Remediate
                    </button>
                  </div>
                </CardContent>
              </div>
            </div>
          </Card>
        ))}
      </div>

      <Card className="col-span-3">
        <CardHeader>
          <CardTitle>Compliance Trends</CardTitle>
          <CardDescription>Current compliance metrics compared to required thresholds</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="h-[400px]">
            <ComplianceChart data={complianceData} />
          </div>
        </CardContent>
      </Card>

      <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Delete Compliance Standard</AlertDialogTitle>
            <AlertDialogDescription>
              Are you sure you want to delete this compliance standard? This action cannot be undone.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel disabled={deleting}>Cancel</AlertDialogCancel>
            <AlertDialogAction 
              onClick={handleDelete} 
              disabled={deleting}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              {deleting ? "Deleting..." : "Delete"}
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
};

const ComplianceOverviewSkeleton = () => {
  return (
    <div className="space-y-4">
      <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4 mt-4">
        {[1, 2, 3, 4, 5, 6].map((i) => (
          <Card key={i} className="overflow-hidden">
            <div className="flex">
              <div className="w-1 h-full bg-gray-200"></div>
              <div className="flex-1">
                <CardHeader className="pb-2">
                  <div className="flex items-start justify-between">
                    <div className="w-full">
                      <Skeleton className="h-5 w-3/4 mb-2" />
                      <Skeleton className="h-3 w-1/3" />
                    </div>
                  </div>
                </CardHeader>
                <CardContent className="pt-0">
                  <div className="flex items-center justify-between">
                    <Skeleton className="h-5 w-1/3" />
                    <Skeleton className="h-5 w-1/4" />
                  </div>
                </CardContent>
              </div>
            </div>
          </Card>
        ))}
      </div>

      <Card className="col-span-3">
        <CardHeader>
          <CardTitle>Compliance Trends</CardTitle>
          <CardDescription>Current compliance metrics compared to required thresholds</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="h-[400px]">
            <Skeleton className="h-full w-full" />
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default ComplianceOverview; 