import { NextResponse } from "next/server";
import { auth } from '@/lib/auth';
import prisma from "@/lib/prisma";

// GET: Fetch all resource allocations for the user's hospital
export async function GET() {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Fetch resource allocations for the hospital
    const resourceAllocations = await prisma.resourceAllocation.findMany({
      where: {
        hospital_id: user.hospital_id,
      },
      orderBy: {
        startDate: "asc",
      },
    });
    
    return NextResponse.json(resourceAllocations);
  } catch (error) {
    console.error("Error fetching resource allocations:", error);
    return NextResponse.json({ error: "Failed to fetch resource allocations" }, { status: 500 });
  }
}

// POST: Create a new resource allocation
export async function POST(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    const data = await request.json();
    
    // Validate required fields
    const requiredFields = ["resourceName", "role", "allocatedTo", "startDate", "endDate", "utilizationPercentage", "department"];
    for (const field of requiredFields) {
      if (!data[field]) {
        return NextResponse.json({ error: `Missing required field: ${field}` }, { status: 400 });
      }
    }
    
    // Create the resource allocation with the hospital ID
    const resourceAllocation = await prisma.resourceAllocation.create({
      data: {
        resourceName: data.resourceName,
        role: data.role,
        allocatedTo: data.allocatedTo,
        startDate: new Date(data.startDate),
        endDate: new Date(data.endDate),
        utilizationPercentage: parseInt(data.utilizationPercentage),
        department: data.department,
        hospital_id: user.hospital_id,
      },
    });
    
    return NextResponse.json(resourceAllocation, { status: 201 });
  } catch (error) {
    console.error("Error creating resource allocation:", error);
    return NextResponse.json({ error: "Failed to create resource allocation" }, { status: 500 });
  }
}

// PATCH: Update an existing resource allocation
export async function PATCH(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const data = await request.json();
    
    if (!data.id) {
      return NextResponse.json({ error: "Missing resource allocation ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Verify the resource allocation belongs to the user's hospital
    const existingAllocation = await prisma.resourceAllocation.findUnique({
      where: {
        id: data.id,
      },
    });
    
    if (!existingAllocation) {
      return NextResponse.json({ error: "Resource allocation not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingAllocation.hospital_id !== user.hospital_id) {
      return NextResponse.json({ error: "Unauthorized to modify this resource allocation" }, { status: 403 });
    }
    
    // Prepare update data
    const updateData: any = { 
      resourceName: data.resourceName,
      role: data.role,
      allocatedTo: data.allocatedTo,
      department: data.department,
    };
    
    // Format dates if present
    if (data.startDate) {
      updateData.startDate = new Date(data.startDate);
    }
    
    if (data.endDate) {
      updateData.endDate = new Date(data.endDate);
    }
    
    // Update utilization percentage if present
    if (data.utilizationPercentage !== undefined) {
      updateData.utilizationPercentage = parseInt(data.utilizationPercentage);
    }
    
    // Update the resource allocation
    const updatedAllocation = await prisma.resourceAllocation.update({
      where: {
        id: data.id,
      },
      data: updateData,
    });
    
    return NextResponse.json(updatedAllocation);
  } catch (error) {
    console.error("Error updating resource allocation:", error);
    return NextResponse.json({ error: "Failed to update resource allocation" }, { status: 500 });
  }
}

// DELETE: Delete a resource allocation
export async function DELETE(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const { searchParams } = new URL(request.url);
    const id = searchParams.get("id");
    
    if (!id) {
      return NextResponse.json({ error: "Missing resource allocation ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Verify the resource allocation belongs to the user's hospital
    const existingAllocation = await prisma.resourceAllocation.findUnique({
      where: {
        id,
      },
    });
    
    if (!existingAllocation) {
      return NextResponse.json({ error: "Resource allocation not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingAllocation.hospital_id !== user.hospital_id) {
      return NextResponse.json({ error: "Unauthorized to delete this resource allocation" }, { status: 403 });
    }
    
    // Delete the resource allocation
    await prisma.resourceAllocation.delete({
      where: {
        id,
      },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error("Error deleting resource allocation:", error);
    return NextResponse.json({ error: "Failed to delete resource allocation" }, { status: 500 });
  }
} 