import { NextResponse } from "next/server";
import { auth } from '@/lib/auth';
import prisma from "@/lib/prisma";

// GET: Fetch all regulatory deadlines for the user's hospital
export async function GET() {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Fetch regulatory deadlines for the hospital
    const regulatoryDeadlines = await prisma.regulatoryDeadline.findMany({
      where: {
        hospital_id: user.hospital_id,
      },
      orderBy: {
        dueDate: "asc",
      },
    });
    
    return NextResponse.json(regulatoryDeadlines);
  } catch (error) {
    console.error("Error fetching regulatory deadlines:", error);
    return NextResponse.json({ error: "Failed to fetch regulatory deadlines" }, { status: 500 });
  }
}

// POST: Create a new regulatory deadline
export async function POST(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    const data = await request.json();
    
    // Validate required fields
    const requiredFields = ["title", "dueDate", "authority", "criticalLevel", "status", "assignedTo", "description"];
    for (const field of requiredFields) {
      if (!data[field]) {
        return NextResponse.json({ error: `Missing required field: ${field}` }, { status: 400 });
      }
    }
    
    // Create the regulatory deadline with the hospital ID
    const regulatoryDeadline = await prisma.regulatoryDeadline.create({
      data: {
        title: data.title,
        dueDate: new Date(data.dueDate),
        authority: data.authority,
        criticalLevel: data.criticalLevel,
        status: data.status,
        assignedTo: data.assignedTo,
        description: data.description,
        hospital_id: user.hospital_id,
      },
    });
    
    return NextResponse.json(regulatoryDeadline, { status: 201 });
  } catch (error) {
    console.error("Error creating regulatory deadline:", error);
    return NextResponse.json({ error: "Failed to create regulatory deadline" }, { status: 500 });
  }
}

// PATCH: Update an existing regulatory deadline
export async function PATCH(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const data = await request.json();
    
    if (!data.id) {
      return NextResponse.json({ error: "Missing deadline ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Verify the regulatory deadline belongs to the user's hospital
    const existingDeadline = await prisma.regulatoryDeadline.findUnique({
      where: {
        id: data.id,
      },
    });
    
    if (!existingDeadline) {
      return NextResponse.json({ error: "Regulatory deadline not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingDeadline.hospital_id !== user.hospital_id) {
      return NextResponse.json({ error: "Unauthorized to modify this deadline" }, { status: 403 });
    }
    
    // Prepare update data
    const updateData: any = { 
      title: data.title,
      authority: data.authority,
      criticalLevel: data.criticalLevel,
      status: data.status,
      assignedTo: data.assignedTo,
      description: data.description,
    };
    
    // Format date if present
    if (data.dueDate) {
      updateData.dueDate = new Date(data.dueDate);
    }
    
    // Update the regulatory deadline
    const updatedDeadline = await prisma.regulatoryDeadline.update({
      where: {
        id: data.id,
      },
      data: updateData,
    });
    
    return NextResponse.json(updatedDeadline);
  } catch (error) {
    console.error("Error updating regulatory deadline:", error);
    return NextResponse.json({ error: "Failed to update regulatory deadline" }, { status: 500 });
  }
}

// DELETE: Delete a regulatory deadline
export async function DELETE(request: Request) {
  try {
    const session = await auth();
    
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    
    const { searchParams } = new URL(request.url);
    const id = searchParams.get("id");
    
    if (!id) {
      return NextResponse.json({ error: "Missing deadline ID" }, { status: 400 });
    }
    
    // Get the hospital ID from the user session
    const user = await prisma.user.findUnique({
      where: {
        email: session.user.email as string,
      },
      select: {
        hospital_id: true,
      },
    });
    
    if (!user || !user.hospital_id) {
      return NextResponse.json({ error: "No hospital associated with user" }, { status: 404 });
    }
    
    // Verify the regulatory deadline belongs to the user's hospital
    const existingDeadline = await prisma.regulatoryDeadline.findUnique({
      where: {
        id,
      },
    });
    
    if (!existingDeadline) {
      return NextResponse.json({ error: "Regulatory deadline not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingDeadline.hospital_id !== user.hospital_id) {
      return NextResponse.json({ error: "Unauthorized to delete this deadline" }, { status: 403 });
    }
    
    // Delete the regulatory deadline
    await prisma.regulatoryDeadline.delete({
      where: {
        id,
      },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error("Error deleting regulatory deadline:", error);
    return NextResponse.json({ error: "Failed to delete regulatory deadline" }, { status: 500 });
  }
} 