import { NextResponse } from "next/server";
import prisma from "@/lib/prisma";
import { Prisma } from "@prisma/client";
import { getApiContext, handleApiError } from '@/lib/api-helpers';

// Define a type for the AuditSchedule from raw queries
interface RawAuditSchedule {
  id: string;
  title: string;
  date: Date;
  department: string;
  auditor: string;
  status: string;
  hospital_id: number;
  createdAt: Date;
  updatedAt: Date;
}

// GET: Fetch all audit schedules for the user's hospital
export async function GET() {
  console.log("GET /api/schedule/audits/route.ts - Start");
  try {
    // Get context including hospital ID
    const { hospitalId, user } = await getApiContext({} as any);
    
    console.log("User data:", { 
      email: user.email, 
      role: user.role, 
      hospitalId: hospitalId 
    });
    
    if (!hospitalId) {
      // If the user is a superadmin, provide a more helpful message
      if (user.role === 'super_admin') {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'As a superadmin, you need to select a hospital first to view its audit schedules'
        }, { status: 400 });
      } else {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'A hospital must be selected to view audit schedules' 
        }, { status: 400 });
      }
    }
    
    console.log("Fetching audit schedules for hospital:", hospitalId);
    
    // Use prisma.$queryRaw to work around type issues
    const auditSchedules = await prisma.$queryRaw<RawAuditSchedule[]>`
      SELECT * FROM audit_schedules 
      WHERE hospital_id = ${hospitalId}
      ORDER BY date ASC
    `;
    
    console.log(`Found ${auditSchedules.length} audit schedules`);
    return NextResponse.json(auditSchedules);
    
  } catch (error: any) {
    console.error("Error fetching audit schedules:", error);
    return handleApiError(error);
  }
}

// POST: Create a new audit schedule
export async function POST(request: Request) {
  try {
    const { hospitalId, user } = await getApiContext(request);
    
    console.log("POST - User data:", { 
      email: user.email, 
      role: user.role, 
      hospitalId: hospitalId 
    });
    
    if (!hospitalId) {
      // If the user is a superadmin, provide a more helpful message
      if (user.role === 'super_admin') {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'As a superadmin, you need to select a hospital first to create audit schedules'
        }, { status: 400 });
      } else {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'A hospital must be selected to create audit schedules' 
        }, { status: 400 });
      }
    }
    
    const data = await request.json();
    
    // Validate required fields
    const requiredFields = ["title", "date", "department", "auditor", "status"];
    for (const field of requiredFields) {
      if (!data[field]) {
        return NextResponse.json({ error: `Missing required field: ${field}` }, { status: 400 });
      }
    }
    
    // Create the audit schedule with the hospital ID
    const date = new Date(data.date);
    const uuid = crypto.randomUUID();
    
    await prisma.$executeRaw`
      INSERT INTO audit_schedules (id, title, date, department, auditor, status, hospital_id, createdAt, updatedAt)
      VALUES (
        ${uuid},
        ${data.title},
        ${date},
        ${data.department},
        ${data.auditor},
        ${data.status},
        ${hospitalId},
        NOW(),
        NOW()
      )
    `;
    
    // Fetch the created audit for response
    const createdAudit = await prisma.$queryRaw<RawAuditSchedule[]>`
      SELECT * FROM audit_schedules WHERE id = ${uuid}
    `;
    
    return NextResponse.json(createdAudit[0], { status: 201 });
  } catch (error: any) {
    console.error("Error creating audit schedule:", error);
    return handleApiError(error);
  }
}

// PATCH: Update an existing audit schedule
export async function PATCH(request: Request) {
  try {
    const { hospitalId, user } = await getApiContext(request);
    
    if (!hospitalId) {
      // If the user is a superadmin, provide a more helpful message
      if (user.role === 'super_admin') {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'As a superadmin, you need to select a hospital first to update audit schedules'
        }, { status: 400 });
      } else {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'A hospital must be selected to update audit schedules' 
        }, { status: 400 });
      }
    }
    
    const data = await request.json();
    
    if (!data.id) {
      return NextResponse.json({ error: "Missing schedule ID" }, { status: 400 });
    }
    
    // Verify the audit schedule belongs to the user's hospital
    const existingSchedule = await prisma.$queryRaw<RawAuditSchedule[]>`
      SELECT * FROM audit_schedules WHERE id = ${data.id}
    `;
    
    if (!existingSchedule || existingSchedule.length === 0) {
      return NextResponse.json({ error: "Audit schedule not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingSchedule[0].hospital_id !== hospitalId) {
      return NextResponse.json({ error: "Unauthorized to modify this schedule" }, { status: 403 });
    }
    
    // Update the audit schedule
    const date = new Date(data.date);
    
    await prisma.$executeRaw`
      UPDATE audit_schedules SET
        title = ${data.title},
        date = ${date},
        department = ${data.department},
        auditor = ${data.auditor},
        status = ${data.status},
        updatedAt = NOW()
      WHERE id = ${data.id}
    `;
    
    // Fetch the updated audit for response
    const updatedAudit = await prisma.$queryRaw<RawAuditSchedule[]>`
      SELECT * FROM audit_schedules WHERE id = ${data.id}
    `;
    
    return NextResponse.json(updatedAudit[0]);
  } catch (error: any) {
    console.error("Error updating audit schedule:", error);
    return handleApiError(error);
  }
}

// DELETE: Delete an audit schedule
export async function DELETE(request: Request) {
  try {
    const { hospitalId, user } = await getApiContext(request);
    
    if (!hospitalId) {
      // If the user is a superadmin, provide a more helpful message
      if (user.role === 'super_admin') {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'As a superadmin, you need to select a hospital first to delete audit schedules'
        }, { status: 400 });
      } else {
        return NextResponse.json({ 
          error: 'Hospital selection required',
          message: 'A hospital must be selected to delete audit schedules' 
        }, { status: 400 });
      }
    }
    
    const url = new URL(request.url);
    const id = url.searchParams.get('id');
    
    if (!id) {
      return NextResponse.json({ error: "Missing schedule ID" }, { status: 400 });
    }
    
    // Verify the audit schedule exists and belongs to the user's hospital
    const existingSchedule = await prisma.$queryRaw<RawAuditSchedule[]>`
      SELECT * FROM audit_schedules WHERE id = ${id}
    `;
    
    if (!existingSchedule || existingSchedule.length === 0) {
      return NextResponse.json({ error: "Audit schedule not found" }, { status: 404 });
    }
    
    // Check hospital ownership
    if (existingSchedule[0].hospital_id !== hospitalId) {
      return NextResponse.json({ error: "Unauthorized to delete this schedule" }, { status: 403 });
    }
    
    // Delete the audit schedule
    await prisma.$executeRaw`
      DELETE FROM audit_schedules WHERE id = ${id}
    `;
    
    return NextResponse.json({ success: true });
  } catch (error: any) {
    console.error("Error deleting audit schedule:", error);
    return handleApiError(error);
  }
} 