import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext } from '@/lib/api-helpers';

// GET /api/regulations/standards - Get all regulatory standards
export async function GET(req: NextRequest) {
  try {
    // Use our helper to get context including hospital ID
    const { hospitalId } = await getApiContext(req);

    // Fetch regulatory standards - simple query
    const standards = await prisma.regulatoryStandard.findMany({
      where: hospitalId ? { hospital_id: hospitalId } : {},
      orderBy: { name: 'asc' },
      select: {
        id: true,
        name: true,
        category: true,
        relevance: true,
        lastUpdated: true,
        description: true,
        hospital_id: true,
        createdAt: true,
        updatedAt: true
      }
    });

    return NextResponse.json(standards);
    
  } catch (error) {
    console.error('Error in regulations standards API:', error);
    return NextResponse.json(
      { error: 'Failed to fetch regulatory standards' },
      { status: 500 }
    );
  }
}

// POST /api/regulations/standards - Create a new regulatory standard
export async function POST(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    const { name, category, relevance, lastUpdated, description } = await request.json();
    
    if (!name || !category || !relevance) {
      return NextResponse.json(
        { error: 'Name, category, and relevance are required' },
        { status: 400 }
      );
    }
    
    let hospitalId = null;
    
    if (session) {
      // Get user's hospital_id from session
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user && user.hospital_id) {
        hospitalId = user.hospital_id;
      }
    } else if (isDevelopment) {
      // In development, use the first hospital
      const firstHospital = await prisma.hospital.findFirst();
      if (firstHospital) {
        hospitalId = firstHospital.id;
      }
    }
    
    const newStandard = await prisma.regulatoryStandard.create({
      data: {
        name,
        category,
        relevance,
        lastUpdated: lastUpdated ? new Date(lastUpdated) : new Date(),
        description: description || '',
        hospital_id: hospitalId,
      },
    });
    
    return NextResponse.json(newStandard, { status: 201 });
  } catch (error) {
    console.error('Failed to create regulatory standard:', error);
    return NextResponse.json(
      { error: 'Failed to create regulatory standard' },
      { status: 500 }
    );
  }
}

// PUT /api/regulations/standards - Update an existing regulatory standard
export async function PUT(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    const { id, name, category, relevance, lastUpdated, description } = await request.json();
    
    if (!id || !name || !category || !relevance) {
      return NextResponse.json(
        { error: 'ID, name, category, and relevance are required' },
        { status: 400 }
      );
    }
    
    // Check if standard exists
    const existingStandard = await prisma.regulatoryStandard.findUnique({
      where: { id },
    });
    
    if (!existingStandard) {
      return NextResponse.json(
        { error: 'Regulatory standard not found' },
        { status: 404 }
      );
    }
    
    // Get user's hospital and role
    let hospitalId = null;
    let isAdmin = false;
    
    if (session) {
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user) {
        hospitalId = user.hospital_id;
        isAdmin = user.role === 'super_admin';
      }
    }
    
    // Check permission: only admin or users from the same hospital can update
    if (!isDevelopment && 
        existingStandard.hospital_id !== null && 
        existingStandard.hospital_id !== hospitalId && 
        !isAdmin) {
      return NextResponse.json(
        { error: 'You do not have permission to update this standard' },
        { status: 403 }
      );
    }
    
    // Update the standard
    const updatedStandard = await prisma.regulatoryStandard.update({
      where: { id },
      data: {
        name,
        category,
        relevance,
        lastUpdated: lastUpdated ? new Date(lastUpdated) : existingStandard.lastUpdated,
        description: description || existingStandard.description,
      },
    });
    
    return NextResponse.json(updatedStandard);
  } catch (error) {
    console.error('Failed to update regulatory standard:', error);
    return NextResponse.json(
      { error: 'Failed to update regulatory standard' },
      { status: 500 }
    );
  }
}

// DELETE /api/regulations/standards - Delete a regulatory standard
export async function DELETE(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    // Get URL parameters
    const url = new URL(request.url);
    const id = url.searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        { error: 'ID is required' },
        { status: 400 }
      );
    }
    
    // Check if standard exists
    const existingStandard = await prisma.regulatoryStandard.findUnique({
      where: { id },
    });
    
    if (!existingStandard) {
      return NextResponse.json(
        { error: 'Regulatory standard not found' },
        { status: 404 }
      );
    }
    
    // Get user's hospital and role
    let hospitalId = null;
    let isAdmin = false;
    
    if (session) {
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user) {
        hospitalId = user.hospital_id;
        isAdmin = user.role === 'super_admin';
      }
    }
    
    // Check permission: only admin or users from the same hospital can delete
    if (!isDevelopment && 
        existingStandard.hospital_id !== null && 
        existingStandard.hospital_id !== hospitalId && 
        !isAdmin) {
      return NextResponse.json(
        { error: 'You do not have permission to delete this standard' },
        { status: 403 }
      );
    }
    
    // Delete the standard
    await prisma.regulatoryStandard.delete({
      where: { id },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Failed to delete regulatory standard:', error);
    return NextResponse.json(
      { error: 'Failed to delete regulatory standard' },
      { status: 500 }
    );
  }
} 