import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { getApiContext } from '@/lib/api-helpers';
import { auth } from '@/lib/auth';

// GET /api/regulations/news - Get all regulatory news
export async function GET(req: NextRequest) {
  try {
    // Use our helper to get context including hospital ID
    const { hospitalId } = await getApiContext(req);

    // Fetch regulatory news - simple query
    const news = await prisma.regulatoryNews.findMany({
      where: hospitalId ? { hospital_id: hospitalId } : {},
      orderBy: { date: 'desc' },
      take: 50, // Limit results for better performance
      select: {
        id: true,
        title: true,
        source: true,
        date: true,
        category: true,
        content: true,
        url: true,
        isNew: true,
        hospital_id: true,
        createdAt: true,
        updatedAt: true
      }
    });

    return NextResponse.json(news);
    
  } catch (error) {
    console.error('Error in regulations news API:', error);
    return NextResponse.json(
      { error: 'Failed to fetch regulatory news' },
      { status: 500 }
    );
  }
}

// POST /api/regulations/news - Create a new regulatory news item
export async function POST(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    const { title, source, date, category, content, url, isNew } = await request.json();
    
    if (!title || !source || !category) {
      return NextResponse.json(
        { error: 'Title, source, and category are required' },
        { status: 400 }
      );
    }
    
    let hospitalId = null;
    
    if (session) {
      // Get user's hospital_id from session
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user && user.hospital_id) {
        hospitalId = user.hospital_id;
      }
    } else if (isDevelopment) {
      // In development, use the first hospital
      const firstHospital = await prisma.hospital.findFirst();
      if (firstHospital) {
        hospitalId = firstHospital.id;
      }
    }
    
    const newsItem = await prisma.regulatoryNews.create({
      data: {
        title,
        source,
        date: date ? new Date(date) : new Date(),
        category,
        content: content || null,
        url: url || null,
        isNew: isNew !== undefined ? isNew : true,
        hospital_id: hospitalId,
      },
    });
    
    return NextResponse.json(newsItem, { status: 201 });
  } catch (error) {
    console.error('Failed to create regulatory news:', error);
    return NextResponse.json(
      { error: 'Failed to create regulatory news' },
      { status: 500 }
    );
  }
}

// PUT /api/regulations/news - Update an existing regulatory news item
export async function PUT(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    const { id, title, source, date, category, content, url, isNew } = await request.json();
    
    if (!id || !title || !source || !category) {
      return NextResponse.json(
        { error: 'ID, title, source, and category are required' },
        { status: 400 }
      );
    }
    
    // Check if news item exists
    const existingNews = await prisma.regulatoryNews.findUnique({
      where: { id },
    });
    
    if (!existingNews) {
      return NextResponse.json(
        { error: 'Regulatory news item not found' },
        { status: 404 }
      );
    }
    
    // Get user's hospital and role
    let hospitalId = null;
    let isAdmin = false;
    
    if (session) {
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user) {
        hospitalId = user.hospital_id;
        isAdmin = user.role === 'super_admin';
      }
    }
    
    // Check permission: only admin or users from the same hospital can update
    if (!isDevelopment && 
        existingNews.hospital_id !== null && 
        existingNews.hospital_id !== hospitalId && 
        !isAdmin) {
      return NextResponse.json(
        { error: 'You do not have permission to update this news item' },
        { status: 403 }
      );
    }
    
    // Update the news item
    const updatedNews = await prisma.regulatoryNews.update({
      where: { id },
      data: {
        title,
        source,
        date: date ? new Date(date) : existingNews.date,
        category,
        content: content !== undefined ? content : existingNews.content,
        url: url !== undefined ? url : existingNews.url,
        isNew: isNew !== undefined ? isNew : existingNews.isNew,
      },
    });
    
    return NextResponse.json(updatedNews);
  } catch (error) {
    console.error('Failed to update regulatory news:', error);
    return NextResponse.json(
      { error: 'Failed to update regulatory news' },
      { status: 500 }
    );
  }
}

// DELETE /api/regulations/news - Delete a regulatory news item
export async function DELETE(request: Request) {
  try {
    const session = await auth();
    const isDevelopment = process.env.NODE_ENV === 'development';
    
    // Only enforce authentication in production
    if (!session && !isDevelopment) {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      );
    }
    
    // Get URL parameters
    const url = new URL(request.url);
    const id = url.searchParams.get('id');
    
    if (!id) {
      return NextResponse.json(
        { error: 'ID is required' },
        { status: 400 }
      );
    }
    
    // Check if news item exists
    const existingNews = await prisma.regulatoryNews.findUnique({
      where: { id },
    });
    
    if (!existingNews) {
      return NextResponse.json(
        { error: 'Regulatory news item not found' },
        { status: 404 }
      );
    }
    
    // Get user's hospital and role
    let hospitalId = null;
    let isAdmin = false;
    
    if (session) {
      const user = await prisma.user.findUnique({
        where: {
          email: session.user?.email as string,
        },
        select: {
          hospital_id: true,
          role: true,
        },
      });
      
      if (user) {
        hospitalId = user.hospital_id;
        isAdmin = user.role === 'super_admin';
      }
    }
    
    // Check permission: only admin or users from the same hospital can delete
    if (!isDevelopment && 
        existingNews.hospital_id !== null && 
        existingNews.hospital_id !== hospitalId && 
        !isAdmin) {
      return NextResponse.json(
        { error: 'You do not have permission to delete this news item' },
        { status: 403 }
      );
    }
    
    // Delete the news item
    await prisma.regulatoryNews.delete({
      where: { id },
    });
    
    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Failed to delete regulatory news:', error);
    return NextResponse.json(
      { error: 'Failed to delete regulatory news' },
      { status: 500 }
    );
  }
} 