import { NextResponse } from 'next/server';
import { getApiContext, handleApiError } from '@/lib/api-helpers';
import prisma from '@/lib/prisma';

// GET - fetch all departments for a hospital
export async function GET(req: Request) {
  try {
    console.log('Hospital Departments API - Starting GET request');
    
    // Get user context (hospital ID)
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to fetch departments' 
      }, { status: 400 });
    }
    
    console.log(`Hospital Departments API - Fetching departments for hospital: ${hospitalId}`);
    
    // Fetch departments from database
    const departments = await prisma.department.findMany({
      where: {
        hospital_id: hospitalId
      },
      orderBy: {
        name: 'asc'
      }
    });
    
    console.log(`Hospital Departments API - Found ${departments.length} departments`);
    
    // If no departments exist, create default ones
    if (departments.length === 0) {
      console.log('No departments found, creating defaults...');
      const defaultDepartments = await createDefaultDepartments(hospitalId);
      return NextResponse.json({ departments: defaultDepartments });
    }
    
    return NextResponse.json({ departments });
    
  } catch (error) {
    console.error("Hospital Departments API - Error:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// Helper function to create default departments
async function createDefaultDepartments(hospitalId: number) {
  const defaultDepartments = [
    { name: 'Nursing', description: 'Nursing and patient care staff' },
    { name: 'Pharmacy', description: 'Medication management and pharmacy services' },
    { name: 'Laboratory', description: 'Medical testing and laboratory services' },
    { name: 'Radiology', description: 'Imaging and diagnostic services' },
    { name: 'Emergency', description: 'Emergency medicine and urgent care' },
    { name: 'Administration', description: 'Hospital administration and management' }
  ];
  
  const createdDepartments = [];
  
  for (const dept of defaultDepartments) {
    try {
      const newDept = await prisma.department.create({
        data: {
          name: dept.name,
          description: dept.description,
          hospital_id: hospitalId
        }
      });
      createdDepartments.push(newDept);
    } catch (error) {
      console.error(`Error creating default department ${dept.name}:`, error);
    }
  }
  
  return createdDepartments;
}

// POST - create a new department
export async function POST(req: Request) {
  try {
    console.log('Hospital Departments API - Starting POST request');
    
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    console.log('Hospital Departments API - User context:', { hospitalId, isSuperAdmin, userEmail: user?.email });
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      console.log('Hospital Departments API - Missing hospital ID');
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create departments' 
      }, { status: 400 });
    }
    
    // Parse request body
    const data = await req.json();
    console.log('Hospital Departments API - Request data:', data);
    
    // Validate required fields
    if (!data.name || !data.name.trim()) {
      console.log('Hospital Departments API - Missing required fields');
      return NextResponse.json({ 
        error: 'Missing required fields',
        message: 'Department name is required' 
      }, { status: 400 });
    }
    
    // Check if department with same name already exists
    const existingDepartment = await prisma.department.findFirst({
      where: {
        hospital_id: hospitalId,
        name: data.name.trim()
      }
    });
    
    if (existingDepartment) {
      console.log('Hospital Departments API - Department with same name already exists');
      return NextResponse.json({ 
        error: 'Department exists',
        message: 'A department with this name already exists' 
      }, { status: 409 });
    }
    
    // Create new department
    const newDepartment = await prisma.department.create({
      data: {
        name: data.name.trim(),
        description: data.description?.trim() || null,
        hospital_id: hospitalId
      }
    });
    
    console.log('Hospital Departments API - Department created:', newDepartment);
    
    return NextResponse.json({
      success: true,
      id: newDepartment.id,
      message: 'Department created successfully',
      department: newDepartment
    }, { status: 201 });
  } catch (error) {
    console.error("Error creating department:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// PATCH - update an existing department
export async function PATCH(req: Request) {
  try {
    console.log('Hospital Departments API - Starting PATCH request');
    
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to update departments' 
      }, { status: 400 });
    }
    
    // Parse request body
    const data = await req.json();
    console.log('Hospital Departments API - Request data:', data);
    
    // Validate record ID
    if (!data.id) {
      console.log('Hospital Departments API - Missing department ID');
      return NextResponse.json({ 
        error: 'Missing department ID',
        message: 'Department ID is required for updates' 
      }, { status: 400 });
    }
    
    // Check if department exists and belongs to this hospital
    const existingDepartment = await prisma.department.findUnique({
      where: {
        id: data.id
      }
    });
    
    if (!existingDepartment) {
      return NextResponse.json({ 
        error: 'Department not found',
        message: 'The specified department does not exist' 
      }, { status: 404 });
    }
    
    if (existingDepartment.hospital_id !== hospitalId) {
      return NextResponse.json({ 
        error: 'Unauthorized',
        message: 'You do not have permission to update this department' 
      }, { status: 403 });
    }
    
    // Check if new name conflicts with existing department
    if (data.name && data.name !== existingDepartment.name) {
      const nameConflict = await prisma.department.findFirst({
        where: {
          hospital_id: hospitalId,
          name: data.name.trim(),
          NOT: {
            id: data.id
          }
        }
      });
      
      if (nameConflict) {
        return NextResponse.json({ 
          error: 'Department exists',
          message: 'A department with this name already exists' 
        }, { status: 409 });
      }
    }
    
    // Update the department
    const updatedDepartment = await prisma.department.update({
      where: {
        id: data.id
      },
      data: {
        name: data.name?.trim() || existingDepartment.name,
        description: data.description?.trim() ?? existingDepartment.description
      }
    });
    
    console.log('Hospital Departments API - Department updated successfully:', updatedDepartment);
    
    return NextResponse.json({
      success: true,
      id: updatedDepartment.id,
      message: 'Department updated successfully',
      department: updatedDepartment
    });
  } catch (error) {
    console.error("Error updating department:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
}

// DELETE - delete a department
export async function DELETE(req: Request) {
  try {
    console.log('Hospital Departments API - Starting DELETE request');
    
    // Get user context
    const { hospitalId, isSuperAdmin, user } = await getApiContext(req);
    
    // Ensure we have a hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to delete departments' 
      }, { status: 400 });
    }
    
    // Get department ID from query parameter
    const url = new URL(req.url);
    const departmentId = url.searchParams.get('id');
    console.log(`Hospital Departments API - Department ID from URL: ${departmentId}`);
    
    if (!departmentId) {
      console.log('Hospital Departments API - Missing department ID');
      return NextResponse.json({ 
        error: 'Missing department ID',
        message: 'Department ID is required for deletion' 
      }, { status: 400 });
    }
    
    // Check if department exists and belongs to this hospital
    const existingDepartment = await prisma.department.findUnique({
      where: {
        id: departmentId
      }
    });
    
    if (!existingDepartment) {
      return NextResponse.json({ 
        error: 'Department not found',
        message: 'The specified department does not exist' 
      }, { status: 404 });
    }
    
    if (existingDepartment.hospital_id !== hospitalId) {
      return NextResponse.json({ 
        error: 'Unauthorized',
        message: 'You do not have permission to delete this department' 
      }, { status: 403 });
    }
    
    // Delete the department
    await prisma.department.delete({
      where: {
        id: departmentId
      }
    });
    
    console.log(`Hospital Departments API - Department deleted successfully`);
    
    return NextResponse.json({
      success: true,
      message: 'Department deleted successfully'
    });
  } catch (error) {
    console.error("Error deleting department:", error instanceof Error ? error.message : String(error));
    return handleApiError(error);
  }
} 