import { NextRequest, NextResponse } from 'next/server';
import { writeFile } from 'fs/promises';
import path from 'path';
import { v4 as uuidv4 } from 'uuid';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

// POST /api/documentation/upload - Upload a file for a document
export async function POST(request: NextRequest) {
  try {
    const { hospitalId } = await getApiContext(request);
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'A hospital must be selected to upload files' 
      }, { status: 400 });
    }
    
    // Parse the FormData
    const formData = await request.formData();
    const file = formData.get('file') as File;
    
    if (!file) {
      return NextResponse.json({ error: 'No file provided' }, { status: 400 });
    }
    
    // Get file details
    const bytes = await file.arrayBuffer();
    const buffer = Buffer.from(bytes);
    const fileSize = buffer.length;
    const fileName = file.name;
    const fileType = file.type;
    
    // Generate a unique filename to prevent conflicts
    const uniqueFilename = `${uuidv4()}-${fileName}`;
    
    // Create the directory path for the hospital
    const hospitalDir = path.join(process.cwd(), 'public', 'documents', hospitalId.toString());
    
    // Ensure the directory exists
    try {
      await writeFile(path.join(hospitalDir, uniqueFilename), buffer);
    } catch (error) {
      console.error('Error writing file:', error);
      // Create directory if it doesn't exist
      const { mkdir } = require('fs/promises');
      await mkdir(hospitalDir, { recursive: true });
      await writeFile(path.join(hospitalDir, uniqueFilename), buffer);
    }
    
    // Create a URL for the file
    const fileUrl = `/documents/${hospitalId}/${uniqueFilename}`;
    
    return NextResponse.json({
      success: true,
      fileName,
      fileUrl,
      fileSize,
      fileType
    });
    
  } catch (error) {
    console.error('Error uploading file:', error);
    return handleApiError(error);
  }
} 