import { NextRequest, NextResponse } from 'next/server';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

// GET /api/documentation/categories - Get all document categories
export async function GET(request: NextRequest) {
  try {
    // Get context including hospital ID
    const { hospitalId } = await getApiContext(request);
    
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'A hospital must be selected to view categories' 
      }, { status: 400 });
    }

    // Fetch categories from database for the specific hospital
    const categories = await prisma.documentCategory.findMany({
      where: { hospital_id: hospitalId },
      orderBy: { name: 'asc' },
      select: {
        id: true,
        name: true,
        icon: true,
        description: true,
        _count: {
          select: {
            documents: true
          }
        }
      }
    });

    // Transform the data to include count
    const categoriesWithCount = categories.map(category => ({
      id: category.id,
      name: category.name,
      icon: category.icon,
      description: category.description,
      count: category._count.documents
    }));

    return NextResponse.json(categoriesWithCount);
  } catch (error) {
    return handleApiError(error);
  }
}

// POST /api/documentation/categories - Create a new document category
export async function POST(request: Request) {
  try {
    const { user, hospitalId } = await getApiContext(request);
    
    const { name, icon, description } = await request.json();
    
    if (!name || !icon) {
      return NextResponse.json(
        { error: 'Name and icon are required' },
        { status: 400 }
      );
    }
    
    if (!hospitalId) {
      return NextResponse.json(
        { error: 'Hospital selection required' },
        { status: 400 }
      );
    }
    
    const newCategory = await prisma.documentCategory.create({
      data: {
        name,
        icon,
        description: description || '',
        hospital_id: hospitalId,
      },
    });
    
    // Include document count for consistency with GET response
    const categoryWithCount = {
      ...newCategory,
      count: 0
    };
    
    return NextResponse.json(categoryWithCount, { status: 201 });
  } catch (error) {
    return handleApiError(error);
  }
} 