import { NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

export async function GET(req: Request) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(req);
    
    // For debugging
    console.log(`Standards API - Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Please select a hospital to view standards data. Use the hospital selector at the top of the page.' 
      }, { status: 400 });
    }
    
    // For regular users without hospital ID, return helpful error
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'Your account is not associated with any hospital. Please contact an administrator.' 
      }, { status: 400 });
    }
    
    // Get standards compliance data for the selected/assigned hospital
    const standards = await prisma.standardCompliance.findMany({
      where: { hospital_id: hospitalId },
      orderBy: { standard: 'asc' }
    });
    
    return NextResponse.json(standards);
  } catch (error) {
    return handleApiError(error);
  }
}

export async function POST(req: Request) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(req);
    
    // For debugging
    console.log(`Creating standard - Hospital ID: ${hospitalId}, Super Admin: ${isSuperAdmin}`);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Please select a hospital to add standards data. Use the hospital selector at the top of the page.' 
      }, { status: 400 });
    }
    
    // Ensure we have a valid hospital ID
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create a standard. Please ensure your account is associated with a hospital.' 
      }, { status: 400 });
    }
    
    const data = await req.json();
    
    // Log the received data for debugging
    console.log('Received data:', JSON.stringify(data));
    
    // Validate the data
    if (!data.standard || !data.category || typeof data.compliance !== 'number') {
      console.log('Validation failed:', {
        standard: data.standard,
        category: data.category,
        compliance: data.compliance,
        complianceType: typeof data.compliance
      });
      
      return NextResponse.json({ 
        error: 'Invalid data. Must include standard, category, and compliance fields.' 
      }, { status: 400 });
    }
    
    // Validate required_threshold if provided
    let requiredThreshold = 90; // Default value
    if (data.required_threshold !== undefined) {
      if (typeof data.required_threshold !== 'number' || data.required_threshold < 0 || data.required_threshold > 100) {
        console.log('Invalid required_threshold:', data.required_threshold, typeof data.required_threshold);
        
        return NextResponse.json({ 
          error: 'Invalid required threshold. Must be a number between 0 and 100.' 
        }, { status: 400 });
      }
      requiredThreshold = data.required_threshold;
    }
    
    // Log the final data before creating
    console.log('Creating standard with data:', {
      standard: data.standard,
      category: data.category,
      compliance: data.compliance,
      required_threshold: requiredThreshold,
      hospital_id: hospitalId
    });
    
    // Create the standard compliance item for the selected/assigned hospital
    const newStandard = await prisma.standardCompliance.create({
      data: {
        standard: data.standard,
        category: data.category,
        compliance: data.compliance,
        required_threshold: requiredThreshold,
        last_assessed: new Date(),
        hospital_id: hospitalId
      }
    });
    
    return NextResponse.json(newStandard, { status: 201 });
  } catch (error) {
    console.error('Error creating standard:', error);
    
    // Create a safe error response
    return NextResponse.json({ 
      error: error instanceof Error ? error.message : 'An unexpected error occurred',
      details: error instanceof Error ? error.stack : String(error)
    }, { status: 500 });
  }
}

 