import { NextRequest, NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';
import { getApiContext, handleApiError } from '@/lib/api-helpers';

export async function GET(request: NextRequest) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(request);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Please select a hospital to view compliance data. Use the hospital selector at the top of the page.' 
      }, { status: 400 });
    }
    
    // For regular users without hospital ID, return helpful error
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'Your account is not associated with any hospital. Please contact an administrator.' 
      }, { status: 400 });
    }
    
    // Query compliance data for the selected/assigned hospital
    const complianceData = await prisma.complianceData.findMany({
      where: { hospital_id: hospitalId },
      orderBy: { created_at: 'desc' }
    });
    
    return NextResponse.json(complianceData);
  } catch (error) {
    return handleApiError(error);
  }
}

export async function POST(req: Request) {
  try {
    // Use our new helper to get context including hospital ID
    const { user, isSuperAdmin, hospitalId } = await getApiContext(req);
    
    // If super admin with no hospital selected, return an error
    if (isSuperAdmin && !hospitalId) {
      return NextResponse.json({ 
        error: 'Hospital selection required',
        message: 'Super admin must select a hospital to add compliance data' 
      }, { status: 400 });
    }
    
    // For any user without hospital ID, return helpful error
    if (!hospitalId) {
      return NextResponse.json({ 
        error: 'Missing hospital ID',
        message: 'A hospital ID is required to create compliance data. Please ensure your account is associated with a hospital.' 
      }, { status: 400 });
    }
    
    const data = await req.json();
    
    // Validate the data
    if (!data.name || typeof data.compliance !== 'number' || typeof data.required !== 'number') {
      return NextResponse.json({ 
        error: 'Invalid data. Must include name, compliance, and required fields.' 
      }, { status: 400 });
    }
    
    // Insert using raw SQL to avoid model name issues
    const insertQuery = `
      INSERT INTO compliance_data (name, compliance, required, hospital_id, created_at, updated_at)
      VALUES (?, ?, ?, ?, NOW(), NOW())
    `;
    
    await prisma.$executeRawUnsafe(
      insertQuery, 
      data.name, 
      data.compliance, 
      data.required, 
      hospitalId
    );
    
    return NextResponse.json({ success: true });
  } catch (error) {
    return handleApiError(error);
  }
} 