import { NextResponse } from 'next/server';
import { auth } from '@/lib/auth';
import prisma from '@/lib/prisma';

export async function GET() {
  try {
    const session = await auth();
    
    if (!session) {
      return NextResponse.json({ 
        isSuperAdmin: false, 
        reason: 'Not authenticated',
        authUrl: process.env.NEXTAUTH_URL
      });
    }
    
    const userEmail = session.user?.email;
    console.log('User email from session:', userEmail);
    
    if (!userEmail) {
      return NextResponse.json({ 
        isSuperAdmin: false, 
        reason: 'No email in session',
        sessionData: session
      });
    }
    
    // Get user details and check role
    try {
      const user = await prisma.user.findUnique({
        where: { email: userEmail },
        select: {
          id: true,
          username: true,
          email: true,
          role: true
        }
      });
      
      if (!user) {
        return NextResponse.json({ 
          isSuperAdmin: false, 
          reason: 'User not found in database',
          email: userEmail
        });
      }
      
      const isSuperAdmin = user.role === 'super_admin';
      
      return NextResponse.json({
        isSuperAdmin,
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          role: user.role
        },
        sessionEmail: userEmail,
        nextAuthUrl: process.env.NEXTAUTH_URL
      });
    } catch (dbError) {
      // Fix the console.error null payload issue
      if (dbError) {
        console.error('Database error checking super_admin:', 
          dbError instanceof Error ? dbError.message : String(dbError)
        );
      }
      
      return NextResponse.json({ 
        isSuperAdmin: false, 
        reason: 'Database error',
        error: dbError instanceof Error ? dbError.message : String(dbError),
        email: userEmail
      }, { status: 500 });
    }
  } catch (error) {
    // Fix the console.error null payload issue
    if (error) {
      console.error('Super admin check error:', 
        error instanceof Error ? error.message : String(error)
      );
    }
    
    return NextResponse.json({ 
      isSuperAdmin: false, 
      reason: 'Server error',
      error: error instanceof Error ? error.message : String(error)
    }, { status: 500 });
  }
} 