'use client'

import { useEffect, useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Users, FileCheck, BarChart, BadgeCheck, Info } from "lucide-react";
import { useTrainingData } from "@/hooks/useTrainingData";
import { TrainingEventList } from "@/components/training/TrainingEventList";
import { StaffComplianceTable } from "@/components/training/StaffComplianceTable";
import { TrainingMetrics } from "@/components/training/TrainingMetrics";

// This page structure is imported from the global-health-guardian-dash-main_4 project
export default function TrainingPage() {
  const [activeTab, setActiveTab] = useState("training");
  const {
    trainingEvents,
    trainingModules,
    staffCompliance,
    loading,
    error,
    createTrainingEvent,
    updateTrainingEvent,
    deleteTrainingEvent,
    getTrainingMetrics,
    createStaffCompliance,
    updateStaffCompliance,
    deleteStaffCompliance
  } = useTrainingData();
  
  useEffect(() => {
    document.title = "Staff & Training | Health Audit System";
  }, []);
  
  const metrics = getTrainingMetrics();
  
  return (
    <div className="container mx-auto py-6 max-w-7xl">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Staff & Training</h1>
          <p className="text-muted-foreground">Manage staff compliance and training</p>
        </div>
      </div>

      {/* Display metrics dashboard at the top */}
      <TrainingMetrics metrics={metrics} />

      <Tabs 
        defaultValue="training" 
        value={activeTab} 
        onValueChange={setActiveTab} 
        className="w-full"
      >
        <TabsList className="mb-4">
          <TabsTrigger value="training" className="text-base">Training Events</TabsTrigger>
          <TabsTrigger value="staff" className="text-base">Staff Compliance</TabsTrigger>
          <TabsTrigger value="modules" className="text-base">Training Modules</TabsTrigger>
          <TabsTrigger value="reports" className="text-base">Training Reports</TabsTrigger>
        </TabsList>
        
        <TabsContent value="training" className="space-y-4">
          <TrainingEventList 
            events={trainingEvents}
            loading={loading.events}
            error={error.events}
            onAdd={createTrainingEvent}
            onEdit={updateTrainingEvent}
            onDelete={deleteTrainingEvent}
          />
        </TabsContent>
        
        <TabsContent value="staff">
          <StaffComplianceTable
            overallRate={staffCompliance.overallRate}
            departments={staffCompliance.departments}
            loading={loading.compliance}
            onAdd={createStaffCompliance}
            onEdit={(data) => updateStaffCompliance({ id: data.id!, name: data.name, staffCount: data.staffCount, complianceRate: data.complianceRate })}
            onDelete={deleteStaffCompliance}
          />
        </TabsContent>
        
        <TabsContent value="modules">
          <TrainingModulesList modules={trainingModules} loading={loading.modules} />
        </TabsContent>
        
        <TabsContent value="reports">
          <ReportsTab />
        </TabsContent>
      </Tabs>
    </div>
  );
}

// Training modules display
function TrainingModulesList({ modules, loading }: { modules: any[], loading: boolean }) {
  if (loading) {
  return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-gray-900"></div>
        </div>
    );
}

  return (
        <div className="space-y-4">
      <h2 className="text-xl font-semibold">Training Modules</h2>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {modules.map(module => (
          <TrainingModule 
            key={module.id}
            title={module.title}
            completionRate={module.completionRate}
            dueDate={module.dueDate}
            department={module.department}
            priority={module.priority}
            />
        ))}
                            </div>
                          </div>
  );
}

// Training module card
function TrainingModule({ 
  title, 
  completionRate, 
  dueDate, 
  department, 
  priority 
}: { 
  title: string;
  completionRate: number;
  dueDate: string;
  department: string;
  priority: 'high' | 'medium' | 'low';
}) {
  const priorityColors = {
    high: 'bg-red-50 text-red-700 border-red-200',
    medium: 'bg-amber-50 text-amber-700 border-amber-200',
    low: 'bg-green-50 text-green-700 border-green-200'
  };

  return (
    <Card>
      <CardHeader className="pb-2">
        <div className="flex justify-between items-start">
          <CardTitle className="text-base font-medium">{title}</CardTitle>
          <div className={`px-2 py-1 rounded text-xs font-medium ${priorityColors[priority]}`}>
            {priority.charAt(0).toUpperCase() + priority.slice(1)} Priority
          </div>
        </div>
        <CardDescription>{department}</CardDescription>
            </CardHeader>
      <CardContent>
        <div className="space-y-4">
          <div>
            <div className="flex justify-between mb-1">
              <span className="text-sm">Completion Rate</span>
              <span className="text-sm font-medium">{completionRate}%</span>
                </div>
            <div className="h-2 bg-gray-100 rounded-full overflow-hidden">
                    <div 
                className={`h-full ${getCompletionColor(completionRate)}`} 
                style={{ width: `${completionRate}%` }}
                    ></div>
            </div>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Due Date:</span>
            <span className="font-medium">{formatDate(dueDate)}</span>
                </div>
              </div>
            </CardContent>
          </Card>
  );
}

// Reports tab
function ReportsTab() {
  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between space-y-0">
        <div>
          <CardTitle>Training Reports</CardTitle>
          <CardDescription>Generate and view training reports</CardDescription>
        </div>
        <BarChart className="h-6 w-6 text-primary" />
            </CardHeader>
            <CardContent>
        <div className="text-center p-8">
          <Info className="h-12 w-12 mx-auto text-muted-foreground mb-3" />
          <h3 className="text-lg font-medium mb-1">Coming Soon</h3>
          <p className="text-muted-foreground max-w-md mx-auto">
            Training reports will be available in the next update.
            This feature will allow you to generate custom reports on staff training and compliance.
          </p>
              </div>
            </CardContent>
          </Card>
  );
}

// Helper functions
function getCompletionColor(rate: number): string {
  if (rate < 60) return 'bg-red-500';
  if (rate < 80) return 'bg-amber-500';
  return 'bg-green-500';
}

function formatDate(dateString: string): string {
  try {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  } catch (e) {
    return dateString;
  }
} 