'use client'

import { useEffect, useState, useMemo } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { BarChart, FileDown, PieChart, Calendar, ArrowDown, ArrowUp, Loader2 } from "lucide-react";
import { useReportsData } from "@/hooks/useReportsData";
import { Separator } from "@/components/ui/separator";
import { Button } from "@/components/ui/button";
import { toast } from "sonner";
import { Badge } from "@/components/ui/badge";
import { useApiWithHospital } from "@/hooks/useApiWithHospital";
import type { Report as ApiReport } from "@/hooks/useReportsData";

// Add CSS for the modal dialog
import './reports.css';

// Use ApiReport to avoid conflicts with the built-in Report type
type Report = ApiReport;

// Define form data interface for type safety
interface ReportFormData {
  title: string;
  type: string;
  timePeriod: string;
  departments: string[];
  includeData: {
    compliance_metrics: boolean;
    audit_findings: boolean;
    training_completion: boolean;
    risk_assessment: boolean;
  };
}

export default function ReportsPage() {
  const { recentReports, reportTypes, summaryData, scheduledReports, loading, error, refreshData, deleteReport } = useReportsData();
  const { fetchWithHospital } = useApiWithHospital();
  const [generatingReport, setGeneratingReport] = useState(false);
  const [viewingReport, setViewingReport] = useState<Report | null>(null);
  const [reportContent, setReportContent] = useState<string>("");
  const [loadingReport, setLoadingReport] = useState(false);
  
  // State for filtering reports
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedType, setSelectedType] = useState("all");
  const [selectedTimeframe, setSelectedTimeframe] = useState("all");
  
  // State for delete confirmation
  const [confirmDelete, setConfirmDelete] = useState(false);
  const [reportToDelete, setReportToDelete] = useState<Report | null>(null);
  const [deleting, setDeleting] = useState(false);
  
  // Form state
  const [formData, setFormData] = useState<ReportFormData>({
    title: "",
    type: reportTypes[0]?.value || "compliance_summary",
    timePeriod: "last_30_days",
    departments: ["All Departments"],
    includeData: {
      compliance_metrics: true,
      audit_findings: true,
      training_completion: true,
      risk_assessment: true,
    }
  });

  useEffect(() => {
    document.title = "Reports | Health Guardian";
  }, []);

  useEffect(() => {
    console.log("initializing formData");
    // Update form data when report types change (first load)
    if (reportTypes.length > 0 && !formData.type) {
      setFormData(prev => ({
        ...prev,
        type: reportTypes[0].value
      }));
    }
  }, [reportTypes]);

  const handleInputChange = (field: string, value: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleIncludeDataChange = (field: string, checked: boolean) => {
    setFormData(prev => ({
      ...prev,
      includeData: {
        ...prev.includeData,
        [field]: checked
      }
    }));
  };

  const handleDepartmentChange = (dept: string, checked: boolean) => {
    setFormData(prev => {
      // If "All Departments" is checked, clear other selections
      if (dept === "All Departments" && checked) {
        return {
          ...prev,
          departments: ["All Departments"]
        };
      }
      
      // If any other department is checked, remove "All Departments"
      let newDepts = [...prev.departments];
      if (checked) {
        // Add department if checked
        if (!newDepts.includes(dept)) {
          newDepts.push(dept);
        }
        // Remove "All Departments" if it's in the list
        newDepts = newDepts.filter(d => d !== "All Departments");
      } else {
        // Remove department if unchecked
        newDepts = newDepts.filter(d => d !== dept);
        // If no departments left, select "All Departments"
        if (newDepts.length === 0) {
          newDepts = ["All Departments"];
        }
      }
      
      return {
        ...prev,
        departments: newDepts
      };
    });
  };

  const handleGenerateReport = async () => {
    if (!formData.title) {
      toast.error("Please enter a report title");
      return;
    }

    setGeneratingReport(true);
    toast.info("Report generation started", {
      description: "Your report is being generated...",
    });

    try {
      console.log("Sending report generation request with data:", {
        title: formData.title,
        type: formData.type,
        timePeriod: formData.timePeriod,
        departments: formData.departments,
        includeData: formData.includeData
      });

      // Use postWithHospital instead of fetchWithHospital for better handling
      const data = await fetchWithHospital('/api/compliance/reports', {
        method: 'POST',
        body: JSON.stringify({
          title: formData.title,
          type: formData.type,
          action: 'generate',
          timePeriod: formData.timePeriod,
          departments: formData.departments,
          includeData: formData.includeData
        }),
      });

      console.log("Successfully received response data:", data);
      
      toast.success("Report generated successfully", {
        description: "Your report is now available in the archive",
      });
      
      // Reset form data
      setFormData({
        ...formData,
        title: ""
      });
      
      // Refresh reports data to show new report
      refreshData();
      
    } catch (error) {
      console.error("Error generating report:", error);
      toast.error("Failed to generate report", {
        description: error instanceof Error ? error.message : "An unexpected error occurred",
      });
    } finally {
      setGeneratingReport(false);
    }
  };

  const handleViewReport = async (report: Report) => {
    console.log("Viewing report:", report);
    setViewingReport(report);
    setLoadingReport(true);
    
    // For pharmacy reports, add a default path if file_url is not available
    let fileUrl = report.file_url;
    if (!fileUrl && (report.title.toLowerCase().includes('pharmacy') || report.type.toLowerCase().includes('pharmacy'))) {
      // Fallback to a sample file for pharmacy reports
      fileUrl = 'sample-reports/pharmacy_compliance.txt';
    }
    
    if (fileUrl) {
      // Check if this is a dynamically generated report (looking for specific patterns)
      const isDynamicReport = fileUrl.match(/reports\/\d+\/[\w-]+\.pdf$/i);
      
      // If it's a dynamic report with .pdf extension, try the .txt version instead
      if (isDynamicReport) {
        const txtFileUrl = fileUrl.replace(/\.pdf$/i, '.txt');
        console.log("Converting PDF path to TXT path:", txtFileUrl);
        fileUrl = txtFileUrl;
      }
      
      try {
        // Use the API endpoint to get the file with proper headers
        // Make sure the path doesn't start with a slash for our API
        const filePath = fileUrl.startsWith('/') ? fileUrl.substring(1) : fileUrl;
        const apiUrl = `/api/reports/download?file=${encodeURIComponent(filePath)}`;
        console.log("Fetching report from URL:", apiUrl);
        
        const response = await fetch(apiUrl);
        if (!response.ok) {
          if (response.status === 404) {
            // For 404 errors, provide a more specific message
            const fallbackReport = await fetchFallbackReport(report);
            if (fallbackReport) {
              // Successfully found a fallback report
              setReportContent(fallbackReport);
              showReportModal();
              setLoadingReport(false);
              return;
            } else {
              throw new Error("Report file not found. The system may still be generating this report.");
            }
          } else {
            throw new Error(`Failed to fetch report: ${response.statusText}`);
          }
        }
        
        // Get the content as text
        const content = await response.text();
        setReportContent(content);
        
        // Show the report modal
        showReportModal();
      } catch (error) {
        console.error("Error loading report:", error);
        toast.error("Failed to load report", {
          description: error instanceof Error ? error.message : "An unexpected error occurred",
        });
      } finally {
        setLoadingReport(false);
      }
    } else {
      // If no file URL, show a toast with the report details
      toast.info(`Viewing ${report.title}`, {
        description: `Type: ${report.type.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}, Created: ${report.date}`,
      });
      setLoadingReport(false);
    }
  };

  // Helper function to open the modal
  const showReportModal = () => {
    const reportModal = document.getElementById('report-modal') as HTMLDialogElement;
    if (reportModal) {
      reportModal.showModal();
    }
  };
  
  // Helper function to fetch a fallback report when the specific one isn't found
  const fetchFallbackReport = async (report: Report): Promise<string | null> => {
    try {
      // Try to find a similar report based on the type
      const basePath = report.file_url?.split('/').slice(0, -1).join('/');
      
      if (!basePath) return null;
      
      // Try to fetch a list of existing reports from the same directory
      const response = await fetchWithHospital('/api/compliance/reports?detailed=true') as any;
      
      if (!response || !response.reports || !Array.isArray(response.reports)) {
        return null;
      }
      
      // Find reports of the same type
      const similarReports = response.reports.filter((r: any) => 
        r.type === report.type && 
        r.id !== report.id && 
        r.file_url
      );
      
      if (similarReports.length === 0) {
        // Try one of our sample reports
        const sampleReports = [
          'sample-reports/pharmacy_compliance.txt',
          'sample-reports/medication_safety.txt'
        ];
        
        for (const samplePath of sampleReports) {
          try {
            const sampleResponse = await fetch(`/api/reports/download?file=${encodeURIComponent(samplePath)}`);
            if (sampleResponse.ok) {
              return await sampleResponse.text();
            }
          } catch (e) {
            console.error("Failed to fetch sample report:", e);
          }
        }
        
        return null;
      }
      
      // Sort by creation date, newest first
      similarReports.sort((a: any, b: any) => 
        new Date(b.created_at).getTime() - new Date(a.created_at).getTime()
      );
      
      // Try to fetch the most recent similar report
      const mostRecentReport = similarReports[0];
      if (mostRecentReport.file_url) {
        const filePath = mostRecentReport.file_url.startsWith('/') 
          ? mostRecentReport.file_url.substring(1) 
          : mostRecentReport.file_url;
          
        const fallbackResponse = await fetch(`/api/reports/download?file=${encodeURIComponent(filePath)}`);
        if (fallbackResponse.ok) {
          // Found a fallback report!
          toast.info("Using similar report", {
            description: "The requested report is not available. Displaying a similar report instead."
          });
          return await fallbackResponse.text();
        }
      }
      
      return null;
    } catch (error) {
      console.error("Error fetching fallback report:", error);
      return null;
    }
  };

  const handlePrintReport = () => {
    // Create a new window with just the report content
    const printWindow = window.open('', '_blank');
    if (!printWindow) {
      toast.error("Could not open print window. Please check your popup blocker settings.");
      return;
    }
    
    // Create HTML content for the print window
    const reportTitle = viewingReport?.title || 'Report';
    const reportDate = viewingReport?.date || new Date().toLocaleDateString();
    const reportType = viewingReport?.type || '';
    
    printWindow.document.write(`
      <!DOCTYPE html>
      <html>
        <head>
          <title>${reportTitle}</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              line-height: 1.6;
              margin: 30px;
            }
            .report-header {
              text-align: center;
              margin-bottom: 30px;
              border-bottom: 1px solid #ddd;
              padding-bottom: 20px;
            }
            .report-title {
              font-size: 24px;
              font-weight: bold;
              margin-bottom: 10px;
            }
            .report-meta {
              color: #666;
              font-size: 14px;
              margin-bottom: 5px;
            }
            .report-content {
              white-space: pre-wrap;
              font-family: monospace;
              background-color: #f9f9f9;
              padding: 20px;
              border: 1px solid #ddd;
              border-radius: 5px;
            }
            @media print {
              body {
                margin: 0.5in;
              }
            }
          </style>
        </head>
        <body>
          <div class="report-header">
            <div class="report-title">${reportTitle}</div>
            <div class="report-meta">Generated: ${reportDate}</div>
            <div class="report-meta">Type: ${reportType.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}</div>
          </div>
          <div class="report-content">${reportContent}</div>
        </body>
      </html>
    `);
    
    // Wait for content to load then print
    printWindow.document.close();
    printWindow.addEventListener('load', () => {
      printWindow.focus();
      printWindow.print();
      // Close the window after printing (some browsers might block this)
      // printWindow.close();
    });
  };

  const closeReportModal = () => {
    const reportModal = document.getElementById('report-modal') as HTMLDialogElement;
    if (reportModal) {
      reportModal.close();
    }
    setViewingReport(null);
    setReportContent("");
  };

  // Get metrics with default values if summaryData is null
  const hipaaScore = summaryData?.complianceMetrics?.hipaa ?? 80;
  const isoScore = summaryData?.complianceMetrics?.iso ?? 75;
  const jciScore = summaryData?.complianceMetrics?.jci ?? 70;

  // Filter reports based on current filters
  const filteredReports = useMemo(() => {
    return recentReports.filter(report => {
      // Filter by search term
      const matchesSearch = searchTerm === "" || 
        report.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        report.author?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        report.type.toLowerCase().includes(searchTerm.toLowerCase());
      
      // Filter by type
      const matchesType = selectedType === "all" || report.type === selectedType;
      
      // Filter by timeframe
      let matchesTimeframe = true;
      if (selectedTimeframe !== "all" && report.date) {
        const reportDate = new Date(report.date);
        const now = new Date();
        
        switch (selectedTimeframe) {
          case "last_30_days":
            const thirtyDaysAgo = new Date();
            thirtyDaysAgo.setDate(now.getDate() - 30);
            matchesTimeframe = reportDate >= thirtyDaysAgo;
            break;
          case "last_quarter":
            const threeMonthsAgo = new Date();
            threeMonthsAgo.setMonth(now.getMonth() - 3);
            matchesTimeframe = reportDate >= threeMonthsAgo;
            break;
          case "last_year":
            const oneYearAgo = new Date();
            oneYearAgo.setFullYear(now.getFullYear() - 1);
            matchesTimeframe = reportDate >= oneYearAgo;
            break;
        }
      }
      
      return matchesSearch && matchesType && matchesTimeframe;
    });
  }, [recentReports, searchTerm, selectedType, selectedTimeframe]);

  // Specific function to filter and display only pharmacy reports
  const showPharmacyReports = () => {
    setSearchTerm("pharmacy");
    // Switch to the archive tab
    const archiveTab = document.querySelector('[value="archive"]') as HTMLElement;
    if (archiveTab) {
      archiveTab.click();
    }
  };

  // Handle delete button click
  const handleDeleteClick = (report: Report) => {
    setReportToDelete(report);
    setConfirmDelete(true);
  };

  // Handle delete confirmation
  const handleConfirmDelete = async () => {
    if (!reportToDelete) return;
    
    setDeleting(true);
    try {
      const success = await deleteReport(reportToDelete.id);
      if (success) {
        toast.success("Report deleted successfully");
        setConfirmDelete(false);
        setReportToDelete(null);
      } else {
        toast.error("Failed to delete report");
      }
    } catch (error) {
      console.error("Error deleting report:", error);
      toast.error("An error occurred while deleting the report");
    } finally {
      setDeleting(false);
    }
  };

  // Cancel delete
  const handleCancelDelete = () => {
    setConfirmDelete(false);
    setReportToDelete(null);
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Compliance Reports</h1>
          <p className="text-muted-foreground">Generate and analyze compliance reports</p>
        </div>
      </div>

      <Tabs defaultValue="dashboard" className="w-full">
        <TabsList>
          <TabsTrigger value="dashboard">Report Dashboard</TabsTrigger>
          <TabsTrigger value="generate">Generate Report</TabsTrigger>
          <TabsTrigger value="scheduled">Scheduled Reports</TabsTrigger>
          <TabsTrigger value="archive">Report Archive</TabsTrigger>
        </TabsList>
        
        <TabsContent value="dashboard" className="space-y-4 mt-4">
          {loading ? (
            <div className="flex justify-center items-center h-64">
              <Loader2 className="h-8 w-8 text-blue-600 animate-spin" />
              <span className="ml-2 text-lg text-gray-600">Loading report data...</span>
            </div>
          ) : error ? (
            <Card>
              <CardContent className="pt-6">
                <div className="text-center text-red-500">
                  <p className="text-lg">{error}</p>
                  <Button 
                    variant="outline" 
                    className="mt-4"
                    onClick={() => window.location.reload()}
                  >
                    Retry
                  </Button>
                </div>
              </CardContent>
            </Card>
          ) : (
            <>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between space-y-0">
                    <div>
                      <CardTitle>Compliance Summary</CardTitle>
                      <CardDescription>Overall compliance status</CardDescription>
                    </div>
                    <PieChart className="h-5 w-5 text-blue-600" />
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">HIPAA Compliance</span>
                        <span className={`text-sm ${hipaaScore >= 90 ? 'text-green-600' : 'text-amber-600'}`}>
                          {hipaaScore}%
                        </span>
                      </div>
                      <div className="h-2 bg-gray-200 rounded-full">
                        <div 
                          className={`h-full rounded-full ${hipaaScore >= 90 ? 'bg-green-500' : 'bg-amber-500'}`} 
                          style={{ width: `${hipaaScore}%` }}
                        ></div>
                      </div>
                      
                      <div className="flex items-center justify-between mt-2">
                        <span className="text-sm font-medium">ISO 9001 Standards</span>
                        <span className={`text-sm ${isoScore >= 90 ? 'text-green-600' : isoScore >= 80 ? 'text-amber-600' : 'text-red-600'}`}>
                          {isoScore}%
                        </span>
                      </div>
                      <div className="h-2 bg-gray-200 rounded-full">
                        <div 
                          className={`h-full rounded-full ${
                            isoScore >= 90 ? 'bg-green-500' : 
                            isoScore >= 80 ? 'bg-amber-500' : 'bg-red-500'
                          }`} 
                          style={{ width: `${isoScore}%` }}
                        ></div>
                      </div>
                      
                      <div className="flex items-center justify-between mt-2">
                        <span className="text-sm font-medium">JCI Requirements</span>
                        <span className={`text-sm ${jciScore >= 90 ? 'text-green-600' : jciScore >= 80 ? 'text-amber-600' : 'text-red-600'}`}>
                          {jciScore}%
                        </span>
                      </div>
                      <div className="h-2 bg-gray-200 rounded-full">
                        <div 
                          className={`h-full rounded-full ${
                            jciScore >= 90 ? 'bg-green-500' : 
                            jciScore >= 80 ? 'bg-amber-500' : 'bg-red-500'
                          }`}
                          style={{ width: `${jciScore}%` }}
                        ></div>
                      </div>
                    </div>
                  </CardContent>
                </Card>
                
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between space-y-0">
                    <div>
                      <CardTitle>Risk Assessment</CardTitle>
                      <CardDescription>Identified compliance risks</CardDescription>
                    </div>
                    <BarChart className="h-5 w-5 text-blue-600" />
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      {summaryData?.riskAssessment?.map((risk, index) => (
                        <div key={index} className="flex items-center justify-between">
                          <div>
                            <div className="font-medium text-sm">{risk.area}</div>
                            <div className="text-xs text-gray-500">{risk.priority}</div>
                          </div>
                          <span className={`text-xs px-2 py-1 rounded-full ${
                            risk.risk === 'high' ? 'bg-red-100 text-red-800' :
                            risk.risk === 'medium' ? 'bg-amber-100 text-amber-800' :
                            'bg-blue-100 text-blue-800'
                          }`}>
                            {risk.risk === 'high' ? 'High Risk' :
                             risk.risk === 'medium' ? 'Medium Risk' :
                             'Low Risk'}
                          </span>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
                
                <Card>
                  <CardHeader className="flex flex-row items-center justify-between space-y-0">
                    <div>
                      <CardTitle>Report Metrics</CardTitle>
                      <CardDescription>Report generation statistics</CardDescription>
                    </div>
                    <Calendar className="h-5 w-5 text-blue-600" />
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      <div>
                        <div className="text-2xl font-bold">{summaryData?.totalReports || 0}</div>
                        <div className="text-xs text-gray-500">Total reports generated</div>
                      </div>
                      
                      <Separator />
                      
                      <div className="space-y-2">
                        {reportTypes.slice(0, 3).map((type) => (
                          <div key={type.value} className="flex justify-between items-center">
                            <span className="text-sm">{type.label}</span>
                            <span className="text-sm font-medium">{type.count}</span>
                          </div>
                        ))}
                        
                        {reportTypes.length > 3 && (
                          <Button variant="ghost" size="sm" className="w-full text-xs mt-2">
                            View all report types
                          </Button>
                        )}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </div>
              
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0">
                  <div>
                    <CardTitle>Recent Reports</CardTitle>
                    <CardDescription>Reports generated in the last 30 days</CardDescription>
                  </div>
                  <FileDown className="h-5 w-5 text-blue-600" />
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {recentReports.length === 0 ? (
                      <div className="text-center py-8 text-gray-500">
                        No reports have been generated recently
                      </div>
                    ) : (
                      recentReports.map((report) => (
                        <div key={report.id} className="flex items-center justify-between py-2 border-b border-gray-100 last:border-0">
                          <div>
                            <div className="font-medium">{report.title}</div>
                            <div className="text-sm text-gray-500">{report.author} • {report.date}</div>
                          </div>
                          <div className="flex items-center gap-2">
                            <Badge variant="outline">{report.type.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}</Badge>
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="text-blue-600 hover:text-blue-800"
                              onClick={() => handleViewReport(report)}
                            >
                              View
                            </Button>
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                </CardContent>
              </Card>
              
              {/* Add a quick link to pharmacy reports in the dashboard */}
              <Button 
                variant="outline" 
                onClick={showPharmacyReports}
                className="mt-4"
              >
                View Pharmacy Reports
              </Button>
            </>
          )}
        </TabsContent>
        
        <TabsContent value="generate">
          <Card>
            <CardHeader>
              <CardTitle>Generate New Report</CardTitle>
              <CardDescription>Create custom compliance reports</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium">Report Title</label>
                  <input 
                    type="text" 
                    value={formData.title}
                    onChange={(e) => handleInputChange('title', e.target.value)}
                    placeholder="Enter report title" 
                    className="w-full rounded-md border border-gray-200 p-2"
                  />
                </div>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <label className="text-sm font-medium">Report Type</label>
                    <select 
                      className="w-full rounded-md border border-gray-200 p-2"
                      value={formData.type}
                      onChange={(e) => handleInputChange('type', e.target.value)}
                    >
                      {reportTypes.map((type) => (
                        <option key={type.value} value={type.value}>
                          {type.label} ({type.count})
                        </option>
                      ))}
                    </select>
                  </div>
                  
                  <div className="space-y-2">
                    <label className="text-sm font-medium">Time Period</label>
                    <select 
                      className="w-full rounded-md border border-gray-200 p-2"
                      value={formData.timePeriod}
                      onChange={(e) => handleInputChange('timePeriod', e.target.value)}
                    >
                      <option value="last_30_days">Last 30 days</option>
                      <option value="last_quarter">Last Quarter</option>
                      <option value="year_to_date">Year to Date</option>
                      <option value="custom_range">Custom Range</option>
                    </select>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <label className="text-sm font-medium">Department(s)</label>
                  <div className="flex flex-wrap gap-2">
                    {['All Departments', 'Clinical Operations', 'Pharmacy', 'Laboratory', 'Radiology', 'Administration'].map((dept) => (
                      <div key={dept} className="flex items-center">
                        <input 
                          type="checkbox" 
                          id={dept} 
                          className="mr-1" 
                          checked={formData.departments.includes(dept)}
                          onChange={(e) => handleDepartmentChange(dept, e.target.checked)}
                        />
                        <label htmlFor={dept} className="text-sm">{dept}</label>
                      </div>
                    ))}
                  </div>
                </div>
                
                <div className="space-y-2">
                  <label className="text-sm font-medium">Include Data</label>
                  <div className="grid grid-cols-2 gap-2">
                    <div className="flex items-center">
                      <input 
                        type="checkbox" 
                        id="compliance-metrics" 
                        checked={formData.includeData.compliance_metrics}
                        onChange={(e) => handleIncludeDataChange('compliance_metrics', e.target.checked)}
                        className="mr-1" 
                      />
                      <label htmlFor="compliance-metrics" className="text-sm">Compliance Metrics</label>
                    </div>
                    <div className="flex items-center">
                      <input 
                        type="checkbox" 
                        id="audit-findings" 
                        checked={formData.includeData.audit_findings}
                        onChange={(e) => handleIncludeDataChange('audit_findings', e.target.checked)}
                        className="mr-1" 
                      />
                      <label htmlFor="audit-findings" className="text-sm">Audit Findings</label>
                    </div>
                    <div className="flex items-center">
                      <input 
                        type="checkbox" 
                        id="training-completion" 
                        checked={formData.includeData.training_completion}
                        onChange={(e) => handleIncludeDataChange('training_completion', e.target.checked)}
                        className="mr-1" 
                      />
                      <label htmlFor="training-completion" className="text-sm">Training Completion</label>
                    </div>
                    <div className="flex items-center">
                      <input 
                        type="checkbox" 
                        id="risk-assessment" 
                        checked={formData.includeData.risk_assessment}
                        onChange={(e) => handleIncludeDataChange('risk_assessment', e.target.checked)}
                        className="mr-1" 
                      />
                      <label htmlFor="risk-assessment" className="text-sm">Risk Assessment</label>
                    </div>
                  </div>
                </div>
                
                <Button 
                  onClick={handleGenerateReport}
                  disabled={generatingReport || !formData.title} 
                  className="w-full md:w-auto"
                >
                  {generatingReport ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Generating...
                    </>
                  ) : (
                    'Generate Report'
                  )}
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="scheduled">
          <Card>
            <CardHeader>
              <CardTitle>Scheduled Reports</CardTitle>
              <CardDescription>Automatically generated reports</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {loading ? (
                  <div className="flex justify-center items-center h-32">
                    <Loader2 className="h-6 w-6 text-blue-600 animate-spin" />
                    <span className="ml-2 text-gray-600">Loading scheduled reports...</span>
                  </div>
                ) : scheduledReports.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">
                    No scheduled reports found
                  </div>
                ) : (
                  scheduledReports.map((report) => (
                    <div key={report.id} className="border rounded-md">
                      <div className="p-4 border-b">
                        <div className="flex justify-between items-start">
                          <div>
                            <h3 className="font-medium">{report.title}</h3>
                            <p className="text-sm text-gray-500">Generated {report.frequency}</p>
                          </div>
                          <Badge variant="outline" className="bg-blue-50">{report.status === 'active' ? 'Active' : 'Inactive'}</Badge>
                        </div>
                        <div className="mt-4 flex justify-between items-center text-sm">
                          <span>Next run: <span className="font-medium">{report.nextRun}</span></span>
                          <span>Format: <span className="font-medium">{report.format}</span></span>
                          <span>Recipients: <span className="font-medium">{report.recipients}</span></span>
                        </div>
                      </div>
                      <div className="p-4 flex justify-end space-x-2">
                        <Button variant="outline" size="sm">Edit</Button>
                        <Button variant="outline" size="sm">Run Now</Button>
                      </div>
                    </div>
                  ))
                )}
                
                <Button variant="outline" className="w-full">+ Create New Scheduled Report</Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="archive">
          <Card>
            <CardHeader>
              <CardTitle>Report Archive</CardTitle>
              <CardDescription>Historical reports and analysis</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex justify-between items-center">
                  <div className="relative w-64">
                    <input 
                      type="text" 
                      placeholder="Search reports..." 
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="w-full pl-8 pr-4 py-2 rounded-md border-gray-200 focus:border-blue-500 focus:ring focus:ring-blue-200 focus:ring-opacity-50"
                    />
                    <span className="absolute left-2 top-2.5">🔍</span>
                  </div>
                  
                  <div className="flex items-center space-x-2">
                    <select 
                      className="rounded-md border border-gray-200 p-2"
                      value={selectedType}
                      onChange={(e) => setSelectedType(e.target.value)}
                    >
                      <option value="all">All Types</option>
                      {reportTypes.map(type => (
                        <option key={type.value} value={type.value}>{type.label}</option>
                      ))}
                    </select>
                    
                    <select 
                      className="rounded-md border border-gray-200 p-2"
                      value={selectedTimeframe}
                      onChange={(e) => setSelectedTimeframe(e.target.value)}
                    >
                      <option value="all">All Time</option>
                      <option value="last_30_days">Last 30 Days</option>
                      <option value="last_quarter">Last Quarter</option>
                      <option value="last_year">Last Year</option>
                    </select>
                  </div>
                </div>
                
                <div className="border rounded-md overflow-hidden">
                  <table className="min-w-full">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Report Name</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Author</th>
                        <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {filteredReports.length === 0 ? (
                        <tr>
                          <td colSpan={5} className="px-6 py-4 text-center text-sm text-gray-500">
                            No reports found matching your criteria
                          </td>
                        </tr>
                      ) : (
                        filteredReports.map((report) => (
                        <tr key={report.id} className="hover:bg-gray-50">
                          <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">{report.title}</td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            {report.type.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{report.date}</td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{report.author}</td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            <div className="flex space-x-2">
                                <Button 
                                  variant="ghost" 
                                  size="sm"
                                  onClick={() => handleViewReport(report)}
                                >
                                  View
                                </Button>
                                <Button 
                                  variant="ghost" 
                                  size="sm"
                                  className="text-red-600 hover:text-red-800 hover:bg-red-50"
                                  onClick={() => handleDeleteClick(report)}
                                >
                                  Delete
                                </Button>
                            </div>
                          </td>
                        </tr>
                        ))
                      )}
                    </tbody>
                  </table>
                </div>
                
                <div className="flex justify-between items-center">
                  <div className="text-sm text-gray-500">
                    Showing {filteredReports.length} of {summaryData?.totalReports || 0} reports
                  </div>
                  
                  <div className="flex space-x-2">
                    <Button variant="outline" size="sm" disabled={true}>
                      <ArrowUp className="h-4 w-4 mr-1" /> Previous
                    </Button>
                    <Button variant="outline" size="sm">
                      Next <ArrowDown className="h-4 w-4 ml-1" />
                    </Button>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Report Viewing Modal */}
      <dialog id="report-modal" className="modal p-0 rounded-lg shadow-lg max-w-3xl w-full">
        <div className="modal-box bg-white p-6">
          <div className="flex justify-between items-center mb-4 border-b pb-3">
            <h3 className="text-lg font-bold">{viewingReport?.title}</h3>
            <Button 
              variant="ghost" 
              size="sm" 
              className="text-gray-500"
              onClick={closeReportModal}
            >
              ✕
            </Button>
          </div>
          
          {loadingReport ? (
            <div className="flex justify-center items-center h-64">
              <Loader2 className="h-8 w-8 text-blue-600 animate-spin" />
              <span className="ml-2 text-gray-600">Loading report...</span>
            </div>
          ) : (
            <>
              <div className="report-metadata mb-4">
                <div className="text-sm text-gray-500">
                  <span className="font-medium">Type:</span> {viewingReport?.type.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}
                </div>
                <div className="text-sm text-gray-500">
                  <span className="font-medium">Date:</span> {viewingReport?.date}
                </div>
                <div className="text-sm text-gray-500">
                  <span className="font-medium">Author:</span> {viewingReport?.author}
                </div>
              </div>
              
              <div className="report-content p-4 bg-gray-50 border rounded-md mb-4 overflow-auto max-h-[60vh]">
                <pre className="whitespace-pre-wrap font-mono text-sm">{reportContent}</pre>
              </div>
              
              <div className="flex justify-end">
                <Button 
                  variant="outline" 
                  className="mr-2"
                  onClick={closeReportModal}
                >
                  Close
                </Button>
                <Button 
                  onClick={handlePrintReport}
                  className="bg-blue-600 text-white hover:bg-blue-700"
                >
                  Print Report
                </Button>
              </div>
            </>
          )}
        </div>
      </dialog>

      {/* Delete Confirmation Dialog */}
      {confirmDelete && (
        <div className="fixed inset-0 bg-black bg-opacity-30 flex items-center justify-center z-50">
          <div className="bg-white p-6 rounded-lg shadow-lg max-w-md w-full">
            <h3 className="text-lg font-semibold mb-4">Delete Report</h3>
            <p className="mb-6">
              Are you sure you want to delete the report "{reportToDelete?.title}"? This action cannot be undone.
            </p>
            <div className="flex justify-end space-x-2">
              <Button
                variant="outline"
                onClick={handleCancelDelete}
                disabled={deleting}
              >
                Cancel
              </Button>
              <Button
                variant="destructive"
                onClick={handleConfirmDelete}
                disabled={deleting}
              >
                {deleting ? (
                  <>
                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                    Deleting...
                  </>
                ) : (
                  "Delete"
                )}
              </Button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
} 