'use client'

import { useEffect, useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useRegulatoryNews } from "@/hooks/useRegulatoryNews";
import { useRegulatoryStandards } from "@/hooks/useRegulatoryStandards";
import { Book, Calendar, FileText, BarChart, Loader2, Plus, Edit, Trash } from "lucide-react";
import RegulatoryImpactAnalysis from "@/components/regulations/RegulatoryImpactAnalysis";
import RegulatoryCalendar from "@/components/regulations/RegulatoryCalendar";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { toast } from "sonner";

export default function RegulationsPage() {
  // States for dialogs
  const [standardDialogOpen, setStandardDialogOpen] = useState(false);
  const [newsDialogOpen, setNewsDialogOpen] = useState(false);
  const [isEditing, setIsEditing] = useState(false);
  
  // States for form data
  const [standardForm, setStandardForm] = useState({
    id: '',
    name: '',
    category: '',
    relevance: 'medium',
    lastUpdated: new Date().toISOString().split('T')[0],
    description: ''
  });
  
  const [newsForm, setNewsForm] = useState({
    id: '',
    title: '',
    source: '',
    date: new Date().toISOString().split('T')[0],
    category: '',
    content: '',
    url: '',
    isNew: true
  });
  
  // Fetch data from hooks
  const { 
    newsItems, 
    loading: newsLoading, 
    error: newsError,
    createNews,
    updateNews,
    deleteNews
  } = useRegulatoryNews();
  
  const {
    standards,
    loading: standardsLoading,
    error: standardsError,
    createStandard,
    updateStandard,
    deleteStandard,
    formatDate
  } = useRegulatoryStandards();
  
  useEffect(() => {
    document.title = "Regulations | Health Guardian";
  }, []);
  
  // Handle standard form submission
  const handleStandardSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!standardForm.name || !standardForm.category || !standardForm.relevance) {
      toast.error('Please fill in all required fields');
      return;
    }
    
    let success;
    
    if (isEditing) {
      success = await updateStandard(standardForm);
    } else {
      // @ts-ignore - id will be generated by the server
      success = await createStandard({
        name: standardForm.name,
        category: standardForm.category,
        relevance: standardForm.relevance,
        lastUpdated: standardForm.lastUpdated,
        description: standardForm.description
      });
    }
    
    if (success) {
      toast.success(isEditing ? 'Standard updated successfully' : 'Standard created successfully');
      setStandardDialogOpen(false);
      resetStandardForm();
    } else {
      toast.error(isEditing ? 'Failed to update standard' : 'Failed to create standard');
    }
  };
  
  // Handle news form submission
  const handleNewsSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!newsForm.title || !newsForm.source || !newsForm.category) {
      toast.error('Please fill in all required fields');
      return;
    }
    
    let success;
    
    if (isEditing) {
      success = await updateNews(newsForm);
    } else {
      // @ts-ignore - id will be generated by the server
      success = await createNews({
        title: newsForm.title,
        source: newsForm.source,
        date: newsForm.date,
        category: newsForm.category,
        content: newsForm.content,
        url: newsForm.url,
        isNew: newsForm.isNew
      });
    }
    
    if (success) {
      toast.success(isEditing ? 'News item updated successfully' : 'News item created successfully');
      setNewsDialogOpen(false);
      resetNewsForm();
    } else {
      toast.error(isEditing ? 'Failed to update news item' : 'Failed to create news item');
    }
  };
  
  // Handle standard deletion
  const handleDeleteStandard = async (id: string) => {
    if (confirm('Are you sure you want to delete this standard?')) {
      const success = await deleteStandard(id);
      
      if (success) {
        toast.success('Standard deleted successfully');
      } else {
        toast.error('Failed to delete standard');
      }
    }
  };
  
  // Handle news deletion
  const handleDeleteNews = async (id: string) => {
    if (confirm('Are you sure you want to delete this news item?')) {
      const success = await deleteNews(id);
      
      if (success) {
        toast.success('News item deleted successfully');
      } else {
        toast.error('Failed to delete news item');
      }
    }
  };
  
  // Open standard dialog for editing
  const openStandardEditDialog = (standard: any) => {
    setStandardForm({
      id: standard.id,
      name: standard.name,
      category: standard.category,
      relevance: standard.relevance,
      lastUpdated: standard.lastUpdated,
      description: standard.description || ''
    });
    
    setIsEditing(true);
    setStandardDialogOpen(true);
  };
  
  // Open news dialog for editing
  const openNewsEditDialog = (news: any) => {
    setNewsForm({
      id: news.id,
      title: news.title,
      source: news.source,
      date: typeof news.date === 'string' && news.date.includes('/') 
        ? new Date(news.date).toISOString().split('T')[0] // Convert display date format to ISO
        : news.date,
      category: news.category,
      content: news.content || '',
      url: news.url || '',
      isNew: news.isNew || false
    });
    
    setIsEditing(true);
    setNewsDialogOpen(true);
  };
  
  // Reset form states
  const resetStandardForm = () => {
    setStandardForm({
      id: '',
      name: '',
      category: '',
      relevance: 'medium',
      lastUpdated: new Date().toISOString().split('T')[0],
      description: ''
    });
    setIsEditing(false);
  };
  
  const resetNewsForm = () => {
    setNewsForm({
      id: '',
      title: '',
      source: '',
      date: new Date().toISOString().split('T')[0],
      category: '',
      content: '',
      url: '',
      isNew: true
    });
    setIsEditing(false);
  };
  
  // Handle dialog close
  const handleStandardDialogClose = () => {
    setStandardDialogOpen(false);
    resetStandardForm();
  };
  
  const handleNewsDialogClose = () => {
    setNewsDialogOpen(false);
    resetNewsForm();
  };

  // Loading state
  if (standardsLoading && newsLoading) {
    return (
      <div className="flex items-center justify-center h-[calc(100vh-200px)]">
        <div className="flex flex-col items-center">
          <Loader2 className="h-8 w-8 text-blue-500 animate-spin mb-2" />
          <p className="text-sm text-gray-500">Loading regulations data...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Regulations</h1>
          <p className="text-muted-foreground">Monitor and manage regulatory requirements</p>
        </div>
      </div>

      <Tabs defaultValue="standards" className="w-full">
        <TabsList>
          <TabsTrigger value="standards">Standards</TabsTrigger>
          <TabsTrigger value="updates">Regulatory Updates</TabsTrigger>
          <TabsTrigger value="calendar">Regulatory Calendar</TabsTrigger>
          <TabsTrigger value="analysis">Impact Analysis</TabsTrigger>
        </TabsList>
        
        <TabsContent value="standards" className="space-y-4 mt-4">
          <div className="flex justify-between items-center mb-4">
            <div className="text-sm text-muted-foreground">
              {standards.length} standards
            </div>
            <Button onClick={() => {
              resetStandardForm();
              setStandardDialogOpen(true);
            }}>
              <Plus className="h-4 w-4 mr-2" />
              Add Standard
            </Button>
          </div>
          
          <div className="grid grid-cols-1 gap-4">
            {standardsError ? (
              <Card className="p-6 text-center">
                <p className="text-red-500">Error loading standards: {standardsError}</p>
              </Card>
            ) : (
              standards.map((standard) => (
                <Card key={standard.id} className="overflow-hidden">
                  <div className="flex">
                    <div className={`w-1 h-full ${
                      standard.relevance === 'high' ? 'bg-blue-500' :
                      standard.relevance === 'medium' ? 'bg-amber-500' : 'bg-gray-500'
                    }`}></div>
                    <div className="flex-1">
                      <CardHeader className="pb-2">
                        <div className="flex items-start justify-between">
                          <div>
                            <CardTitle className="text-base font-medium">{standard.name}</CardTitle>
                            <CardDescription className="text-xs mt-1">
                              {standard.category}
                            </CardDescription>
                          </div>
                          <Badge className={
                            standard.relevance === 'high' 
                              ? 'bg-blue-100 text-blue-800' 
                              : standard.relevance === 'medium'
                                ? 'bg-amber-100 text-amber-800'
                                : 'bg-gray-100 text-gray-800'
                          }>
                            {standard.relevance === 'high' ? 'High Relevance' : 
                             standard.relevance === 'medium' ? 'Medium Relevance' : 'Low Relevance'}
                          </Badge>
                        </div>
                      </CardHeader>
                      <CardContent className="pt-0">
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-500">
                            Last updated: {formatDate(standard.lastUpdated)}
                          </span>
                          <div className="flex gap-2">
                            <Button variant="ghost" size="icon" onClick={() => openStandardEditDialog(standard)}>
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button variant="ghost" size="icon" onClick={() => handleDeleteStandard(standard.id)}>
                              <Trash className="h-4 w-4 text-red-500" />
                            </Button>
                          </div>
                        </div>
                      </CardContent>
                    </div>
                  </div>
                </Card>
              ))
            )}
          </div>
        </TabsContent>
        
        <TabsContent value="updates">
          <div className="flex justify-between items-center mb-4">
            <div className="text-sm text-muted-foreground">
              {newsItems.length} updates
            </div>
            <Button onClick={() => {
              resetNewsForm();
              setNewsDialogOpen(true);
            }}>
              <Plus className="h-4 w-4 mr-2" />
              Add News
            </Button>
          </div>
          
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <div>
                <CardTitle>Regulatory Updates</CardTitle>
                <CardDescription>Latest regulatory news and changes</CardDescription>
              </div>
            </CardHeader>
            <CardContent>
              {newsError ? (
                <div className="p-6 text-center">
                  <p className="text-red-500">Error loading news: {newsError}</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {newsItems.map((item) => (
                    <div key={item.id} className="flex items-start gap-4 pb-4 border-b border-gray-200 last:border-0 last:pb-0">
                      <div className="flex-1">
                        <h4 className="font-medium text-sm">{item.title}</h4>
                        <div className="flex items-center gap-2 mt-1">
                          <span className="text-xs text-gray-500">{item.source}</span>
                          <span className="text-xs text-gray-500">•</span>
                          <span className="text-xs text-gray-500">{item.date}</span>
                          {item.isNew && (
                            <Badge className="bg-blue-100 text-blue-800">New</Badge>
                          )}
                        </div>
                      </div>
                      <div className="flex gap-2 items-center">
                        <Badge variant="outline" className="text-xs">
                          {item.category}
                        </Badge>
                        <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => openNewsEditDialog(item)}>
                          <Edit className="h-3.5 w-3.5" />
                        </Button>
                        <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => handleDeleteNews(item.id)}>
                          <Trash className="h-3.5 w-3.5 text-red-500" />
                        </Button>
                      </div>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="calendar">
          <RegulatoryCalendar />
        </TabsContent>
        
        <TabsContent value="analysis">
          <RegulatoryImpactAnalysis />
        </TabsContent>
      </Tabs>
      
      {/* Dialog for adding/editing standards */}
      <Dialog open={standardDialogOpen} onOpenChange={handleStandardDialogClose}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>{isEditing ? 'Edit Regulatory Standard' : 'Add Regulatory Standard'}</DialogTitle>
            <DialogDescription>
              {isEditing ? 'Update the regulatory standard details.' : 'Add a new regulatory standard to track.'}
            </DialogDescription>
          </DialogHeader>
          
          <form onSubmit={handleStandardSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid gap-2">
                <Label htmlFor="name">Standard Name</Label>
                <Input 
                  id="name" 
                  value={standardForm.name} 
                  onChange={(e) => setStandardForm({...standardForm, name: e.target.value})} 
                  placeholder="ISO 9001:2015"
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="category">Category</Label>
                <Input 
                  id="category" 
                  value={standardForm.category} 
                  onChange={(e) => setStandardForm({...standardForm, category: e.target.value})} 
                  placeholder="Quality Management"
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="relevance">Relevance</Label>
                <Select 
                  value={standardForm.relevance} 
                  onValueChange={(value) => setStandardForm({...standardForm, relevance: value})}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select relevance" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="high">High</SelectItem>
                    <SelectItem value="medium">Medium</SelectItem>
                    <SelectItem value="low">Low</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="lastUpdated">Last Updated</Label>
                <Input 
                  id="lastUpdated" 
                  type="date" 
                  value={standardForm.lastUpdated} 
                  onChange={(e) => setStandardForm({...standardForm, lastUpdated: e.target.value})} 
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="description">Description</Label>
                <Textarea 
                  id="description" 
                  value={standardForm.description} 
                  onChange={(e) => setStandardForm({...standardForm, description: e.target.value})} 
                  placeholder="Description of the standard..."
                  rows={3}
                />
              </div>
            </div>
            
            <DialogFooter>
              <Button type="button" variant="outline" onClick={handleStandardDialogClose}>
                Cancel
              </Button>
              <Button type="submit">
                {isEditing ? 'Update Standard' : 'Add Standard'}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
      
      {/* Dialog for adding/editing news */}
      <Dialog open={newsDialogOpen} onOpenChange={handleNewsDialogClose}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>{isEditing ? 'Edit Regulatory News' : 'Add Regulatory News'}</DialogTitle>
            <DialogDescription>
              {isEditing ? 'Update the regulatory news details.' : 'Add a new regulatory news item.'}
            </DialogDescription>
          </DialogHeader>
          
          <form onSubmit={handleNewsSubmit}>
            <div className="grid gap-4 py-4">
              <div className="grid gap-2">
                <Label htmlFor="title">Title</Label>
                <Input 
                  id="title" 
                  value={newsForm.title} 
                  onChange={(e) => setNewsForm({...newsForm, title: e.target.value})} 
                  placeholder="New regulatory update announced"
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="source">Source</Label>
                <Input 
                  id="source" 
                  value={newsForm.source} 
                  onChange={(e) => setNewsForm({...newsForm, source: e.target.value})} 
                  placeholder="Joint Commission International"
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="date">Date</Label>
                <Input 
                  id="date" 
                  type="date" 
                  value={newsForm.date} 
                  onChange={(e) => setNewsForm({...newsForm, date: e.target.value})} 
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="category">Category</Label>
                <Input 
                  id="category" 
                  value={newsForm.category} 
                  onChange={(e) => setNewsForm({...newsForm, category: e.target.value})} 
                  placeholder="Standards Update"
                  required
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="content">Content</Label>
                <Textarea 
                  id="content" 
                  value={newsForm.content} 
                  onChange={(e) => setNewsForm({...newsForm, content: e.target.value})} 
                  placeholder="Details about the regulatory news..."
                  rows={3}
                />
              </div>
              
              <div className="grid gap-2">
                <Label htmlFor="url">URL</Label>
                <Input 
                  id="url" 
                  value={newsForm.url} 
                  onChange={(e) => setNewsForm({...newsForm, url: e.target.value})} 
                  placeholder="https://example.com/news"
                />
              </div>
              
              <div className="flex items-center space-x-2">
                <input
                  type="checkbox"
                  id="isNew"
                  checked={newsForm.isNew}
                  onChange={(e) => setNewsForm({...newsForm, isNew: e.target.checked})}
                  className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                />
                <Label htmlFor="isNew">Mark as new</Label>
              </div>
            </div>
            
            <DialogFooter>
              <Button type="button" variant="outline" onClick={handleNewsDialogClose}>
                Cancel
              </Button>
              <Button type="submit">
                {isEditing ? 'Update News' : 'Add News'}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
} 