'use client'

import { useState, useEffect } from 'react'
import { toast } from 'sonner'
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from '@/components/ui/card'
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog'
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Textarea } from '@/components/ui/textarea'
import { getSession } from 'next-auth/react'

// Interface for Department
interface Department {
  id: string
  name: string
  description: string | null
  hospital_id: number | null
  createdAt: string
  updatedAt: string
}

export default function DepartmentsPage() {
  const [departments, setDepartments] = useState<Department[]>([])
  const [loading, setLoading] = useState(true)
  const [dialogOpen, setDialogOpen] = useState(false)
  const [editDialogOpen, setEditDialogOpen] = useState(false)
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false)
  const [selectedDepartment, setSelectedDepartment] = useState<Department | null>(null)
  const [formData, setFormData] = useState({
    name: '',
    description: ''
  })

  // Fetch departments on component mount
  useEffect(() => {
    document.title = "Departments | Health Guardian"
    fetchDepartments()
  }, [])

  // Function to fetch departments
  const fetchDepartments = async () => {
    setLoading(true)
    try {
      console.log('Fetching departments...')
      const response = await fetch('/api/hospital/departments', {
        credentials: 'include', // Include credentials for authentication
        headers: {
          'Content-Type': 'application/json'
        }
      })
      
      if (!response.ok) {
        const errorText = await response.text()
        console.error(`API error response (${response.status}):`, errorText)
        
        // Handle authentication errors specifically
        if (response.status === 401 || response.status === 403) {
          toast.error('Authentication failed. Please log in again.')
          // You might want to redirect to login page here
          return
        }
        
        throw new Error(`Failed to fetch departments: ${response.status} ${response.statusText}`)
      }
      
      const result = await response.json()
      
      // Check if we have departments in the response
      if (result.departments && Array.isArray(result.departments)) {
        console.log(`Fetched ${result.departments.length} departments`)
        setDepartments(result.departments)
      } else {
        console.error('Unexpected API response format:', result)
        toast.error('Received invalid data format from server')
        setDepartments([])
      }
    } catch (error) {
      console.error('Error fetching departments:', error)
      toast.error('Failed to load departments. Please try again later.')
      // Set empty departments instead of showing loading forever
      setDepartments([])
    } finally {
      setLoading(false)
    }
  }

  // Handle input change
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({ ...prev, [name]: value }))
  }

  // Handle create department
  const handleCreateDepartment = async () => {
    if (!formData.name.trim()) {
      toast.error('Department name is required')
      return
    }

    try {
      const response = await fetch('/api/hospital/departments', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify(formData)
      })

      if (!response.ok) {
        let errorMessage = 'Failed to create department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      toast.success('Department created successfully')
      setDialogOpen(false)
      setFormData({ name: '', description: '' })
      fetchDepartments()
    } catch (error) {
      console.error('Error creating department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to create department')
    }
  }

  // Handle edit department
  const handleEditDepartment = async () => {
    if (!selectedDepartment || !formData.name.trim()) {
      toast.error('Department name is required')
      return
    }

    try {
      const response = await fetch('/api/hospital/departments', {
        method: 'PATCH',
        headers: {
          'Content-Type': 'application/json'
        },
        credentials: 'include',
        body: JSON.stringify({
          id: selectedDepartment.id,
          name: formData.name,
          description: formData.description
        })
      })

      if (!response.ok) {
        let errorMessage = 'Failed to update department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      toast.success('Department updated successfully')
      setEditDialogOpen(false)
      setSelectedDepartment(null)
      setFormData({ name: '', description: '' })
      fetchDepartments()
    } catch (error) {
      console.error('Error updating department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to update department')
    }
  }

  // Handle delete department
  const handleDeleteDepartment = async () => {
    if (!selectedDepartment) return

    try {
      const response = await fetch(`/api/hospital/departments?id=${selectedDepartment.id}`, {
        method: 'DELETE',
        credentials: 'include',
        headers: {
          'Content-Type': 'application/json'
        }
      })

      if (!response.ok) {
        let errorMessage = 'Failed to delete department'
        
        // Safely try to parse error response
        try {
          const responseText = await response.text()
          if (responseText) {
            const errorData = JSON.parse(responseText)
            errorMessage = errorData.message || errorMessage
          }
        } catch (parseError) {
          console.error('Error parsing error response:', parseError)
        }
        
        throw new Error(errorMessage)
      }

      toast.success('Department deleted successfully')
      setDeleteDialogOpen(false)
      setSelectedDepartment(null)
      fetchDepartments()
    } catch (error) {
      console.error('Error deleting department:', error)
      toast.error(error instanceof Error ? error.message : 'Failed to delete department')
    }
  }

  // Open edit dialog for a department
  const openEditDialog = (department: Department) => {
    setSelectedDepartment(department)
    setFormData({
      name: department.name,
      description: department.description || ''
    })
    setEditDialogOpen(true)
  }

  // Open delete dialog for a department
  const openDeleteDialog = (department: Department) => {
    setSelectedDepartment(department)
    setDeleteDialogOpen(true)
  }

  return (
    <div className="space-y-6 p-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold">Hospital Departments</h1>
          <p className="text-muted-foreground">Manage your hospital's departments</p>
        </div>
        <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
          <DialogTrigger asChild>
            <Button>Add Department</Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Add New Department</DialogTitle>
              <DialogDescription>
                Create a new department for your hospital.
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4 py-4">
              <div className="space-y-2">
                <Label htmlFor="name">Department Name</Label>
                <Input
                  id="name"
                  name="name"
                  placeholder="Enter department name"
                  value={formData.name}
                  onChange={handleInputChange}
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="description">Description (Optional)</Label>
                <Textarea
                  id="description"
                  name="description"
                  placeholder="Enter department description"
                  value={formData.description}
                  onChange={handleInputChange}
                  rows={3}
                />
              </div>
            </div>
            <DialogFooter>
              <Button variant="outline" onClick={() => setDialogOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleCreateDepartment}>Create Department</Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Departments List</CardTitle>
          <CardDescription>View and manage all departments in your hospital</CardDescription>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-4">Loading departments...</div>
          ) : departments.length === 0 ? (
            <div className="text-center py-4 text-muted-foreground">
              No departments found. Create your first department by clicking the "Add Department" button.
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Name</TableHead>
                  <TableHead>Description</TableHead>
                  <TableHead>Created</TableHead>
                  <TableHead className="text-right">Actions</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {departments.map((department) => (
                  <TableRow key={department.id}>
                    <TableCell className="font-medium">{department.name}</TableCell>
                    <TableCell>{department.description || '-'}</TableCell>
                    <TableCell>{new Date(department.createdAt).toLocaleDateString()}</TableCell>
                    <TableCell className="text-right">
                      <div className="flex justify-end space-x-2">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => openEditDialog(department)}
                        >
                          Edit
                        </Button>
                        <Button
                          variant="destructive"
                          size="sm"
                          onClick={() => openDeleteDialog(department)}
                        >
                          Delete
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Edit Dialog */}
      <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Edit Department</DialogTitle>
            <DialogDescription>
              Update the department information.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="edit-name">Department Name</Label>
              <Input
                id="edit-name"
                name="name"
                placeholder="Enter department name"
                value={formData.name}
                onChange={handleInputChange}
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="edit-description">Description (Optional)</Label>
              <Textarea
                id="edit-description"
                name="description"
                placeholder="Enter department description"
                value={formData.description}
                onChange={handleInputChange}
                rows={3}
              />
            </div>
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setEditDialogOpen(false)}>
              Cancel
            </Button>
            <Button onClick={handleEditDepartment}>Update Department</Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <Dialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Confirm Deletion</DialogTitle>
            <DialogDescription>
              Are you sure you want to delete the department "{selectedDepartment?.name}"? This action cannot be undone.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button variant="outline" onClick={() => setDeleteDialogOpen(false)}>
              Cancel
            </Button>
            <Button variant="destructive" onClick={handleDeleteDepartment}>
              Delete Department
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  )
} 