'use client'

import React from 'react'
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { 
  Book, 
  FileText, 
  Bell, 
  Calendar, 
  BarChart, 
  ShieldAlert, 
  Sparkles,
  CheckCircle, 
  FileCheck, 
  ExternalLink 
} from 'lucide-react'
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { useRegulatoryNews } from '@/hooks/useRegulatoryNews'
import { useRegulatoryStandards } from '@/hooks/useRegulatoryStandards'
import Link from 'next/link'

export default function DocumentationPage() {
  const { standards, error: standardsError } = useRegulatoryStandards()
  const { newsItems, error: newsError } = useRegulatoryNews()
  
  const apiEndpoints = [
    {
      name: '/api/regulations/standards',
      description: 'Get, create, update, and delete regulatory standards',
      methods: ['GET', 'POST', 'PUT', 'DELETE'],
    },
    {
      name: '/api/regulations/news',
      description: 'Get, create, update, and delete regulatory news items',
      methods: ['GET', 'POST', 'PUT', 'DELETE'],
    }
  ]
  
  return (
    <div className="container mx-auto py-6 max-w-5xl">
      <div className="flex items-center mb-6">
        <Book className="h-8 w-8 mr-3 text-primary" />
        <h1 className="text-3xl font-bold text-gray-900">Regulations Documentation</h1>
      </div>
      
      {(standardsError || newsError) && (
        <Alert variant="destructive" className="mb-6">
          <ShieldAlert className="h-4 w-4" />
          <AlertTitle>API Connection Error</AlertTitle>
          <AlertDescription>
            {standardsError || newsError}. The regulatory data API endpoints may not be available.
            The application will use fallback data until the API is accessible.
          </AlertDescription>
        </Alert>
      )}
      
      <Tabs defaultValue="overview" className="w-full">
        <TabsList className="mb-6">
          <TabsTrigger value="overview" className="text-base">Overview</TabsTrigger>
          <TabsTrigger value="standards" className="text-base">Standards</TabsTrigger>
          <TabsTrigger value="news" className="text-base">News</TabsTrigger>
          <TabsTrigger value="api" className="text-base">API Reference</TabsTrigger>
        </TabsList>
        
        <TabsContent value="overview">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <Card>
              <CardHeader>
                <CardTitle>Regulations Module</CardTitle>
                <CardDescription>
                  Overview of the regulatory compliance system
                </CardDescription>
              </CardHeader>
              <CardContent>
                <p className="mb-4">
                  The Regulations module helps hospitals track, monitor, and ensure compliance with
                  various healthcare regulations and standards. This system provides tools to:
                </p>
                <ul className="space-y-2 mb-4">
                  <li className="flex items-start">
                    <CheckCircle className="h-5 w-5 text-green-500 mr-2 flex-shrink-0 mt-0.5" />
                    <span>Track healthcare standards and their compliance status</span>
                  </li>
                  <li className="flex items-start">
                    <CheckCircle className="h-5 w-5 text-green-500 mr-2 flex-shrink-0 mt-0.5" />
                    <span>Stay updated with regulatory news and announcements</span>
                  </li>
                  <li className="flex items-start">
                    <CheckCircle className="h-5 w-5 text-green-500 mr-2 flex-shrink-0 mt-0.5" />
                    <span>Plan and schedule compliance activities and deadlines</span>
                  </li>
                  <li className="flex items-start">
                    <CheckCircle className="h-5 w-5 text-green-500 mr-2 flex-shrink-0 mt-0.5" />
                    <span>Analyze the impact of regulatory changes on hospital operations</span>
                  </li>
                </ul>
                
                <div className="mt-6">
                  <Button asChild>
                    <Link href="/regulations">
                      Go to Regulations
                      <ExternalLink className="ml-2 h-4 w-4" />
                    </Link>
                  </Button>
                </div>
              </CardContent>
            </Card>
            
            <Card>
              <CardHeader>
                <CardTitle>Key Features</CardTitle>
                <CardDescription>
                  Main components of the regulations system
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex items-start">
                    <div className="bg-blue-100 p-2 rounded-md mr-3">
                      <FileText className="h-5 w-5 text-blue-600" />
                    </div>
                    <div>
                      <h3 className="font-medium">Standards Tracking</h3>
                      <p className="text-sm text-gray-500">Monitor compliance with key standards like HIPAA, JCI, and ISO</p>
                    </div>
                  </div>
                  
                  <div className="flex items-start">
                    <div className="bg-amber-100 p-2 rounded-md mr-3">
                      <Bell className="h-5 w-5 text-amber-600" />
                    </div>
                    <div>
                      <h3 className="font-medium">Regulatory Updates</h3>
                      <p className="text-sm text-gray-500">Stay informed about new regulations and changes</p>
                    </div>
                  </div>
                  
                  <div className="flex items-start">
                    <div className="bg-green-100 p-2 rounded-md mr-3">
                      <Calendar className="h-5 w-5 text-green-600" />
                    </div>
                    <div>
                      <h3 className="font-medium">Regulatory Calendar</h3>
                      <p className="text-sm text-gray-500">Plan and schedule compliance activities and deadlines</p>
                    </div>
                  </div>
                  
                  <div className="flex items-start">
                    <div className="bg-purple-100 p-2 rounded-md mr-3">
                      <BarChart className="h-5 w-5 text-purple-600" />
                    </div>
                    <div>
                      <h3 className="font-medium">Impact Analysis</h3>
                      <p className="text-sm text-gray-500">Assess how regulatory changes affect your hospital</p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>
        
        <TabsContent value="standards">
          <Card>
            <CardHeader>
              <CardTitle>Regulatory Standards</CardTitle>
              <CardDescription>
                Understanding the standards tracking system
              </CardDescription>
            </CardHeader>
            <CardContent>
              <p className="mb-4">
                The Regulatory Standards tab allows you to track and manage compliance with various healthcare
                standards and regulations. Each standard includes information about its relevance, 
                category, and last update date.
              </p>
              
              <h3 className="font-medium text-lg mt-6 mb-3">Standard Attributes</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                <div className="bg-gray-50 p-4 rounded-md">
                  <h4 className="font-medium text-sm mb-2">Relevance Levels</h4>
                  <div className="space-y-2">
                    <div className="flex items-center">
                      <Badge className="bg-blue-100 text-blue-800 mr-2">High</Badge>
                      <span className="text-sm">Critical standards requiring immediate attention</span>
                    </div>
                    <div className="flex items-center">
                      <Badge className="bg-amber-100 text-amber-800 mr-2">Medium</Badge>
                      <span className="text-sm">Important standards to monitor regularly</span>
                    </div>
                    <div className="flex items-center">
                      <Badge className="bg-gray-100 text-gray-800 mr-2">Low</Badge>
                      <span className="text-sm">Standards with minimal impact on operations</span>
                    </div>
                  </div>
                </div>
                
                <div className="bg-gray-50 p-4 rounded-md">
                  <h4 className="font-medium text-sm mb-2">Common Categories</h4>
                  <ul className="space-y-1 text-sm">
                    <li>Quality Management</li>
                    <li>Data Privacy</li>
                    <li>Data Protection</li>
                    <li>Healthcare Quality</li>
                    <li>Electronic Records</li>
                    <li>Workplace Safety</li>
                  </ul>
                </div>
              </div>
              
              <h3 className="font-medium text-lg mt-6 mb-3">Currently Tracked Standards</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {standards.slice(0, 4).map((standard) => (
                  <div key={standard.id} className="border rounded-md p-3">
                    <div className="flex justify-between items-start">
                      <h4 className="font-medium">{standard.name}</h4>
                      <Badge className={
                        standard.relevance === 'high' 
                          ? 'bg-blue-100 text-blue-800' 
                          : standard.relevance === 'medium'
                            ? 'bg-amber-100 text-amber-800'
                            : 'bg-gray-100 text-gray-800'
                      }>
                        {standard.relevance}
                      </Badge>
                    </div>
                    <p className="text-sm text-gray-500 mt-1">{standard.category}</p>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="news">
          <Card>
            <CardHeader>
              <CardTitle>Regulatory News</CardTitle>
              <CardDescription>
                Understanding the regulatory updates system
              </CardDescription>
            </CardHeader>
            <CardContent>
              <p className="mb-4">
                The Regulatory Updates tab keeps you informed about important changes, announcements, 
                and news related to healthcare regulations. This ensures you stay current with the 
                latest regulatory landscape.
              </p>
              
              <h3 className="font-medium text-lg mt-6 mb-3">News Attributes</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                <div className="bg-gray-50 p-4 rounded-md">
                  <h4 className="font-medium text-sm mb-2">News Categories</h4>
                  <ul className="space-y-1 text-sm">
                    <li>Standards Update</li>
                    <li>Regulatory Guidance</li>
                    <li>Enforcement</li>
                    <li>Privacy</li>
                    <li>Compliance Deadlines</li>
                  </ul>
                </div>
                
                <div className="bg-gray-50 p-4 rounded-md">
                  <h4 className="font-medium text-sm mb-2">News Sources</h4>
                  <ul className="space-y-1 text-sm">
                    <li>Joint Commission International</li>
                    <li>U.S. FDA</li>
                    <li>EU Commission</li>
                    <li>World Health Organization</li>
                    <li>Regional Health Authorities</li>
                  </ul>
                </div>
              </div>
              
              <h3 className="font-medium text-lg mt-6 mb-3">Recent Regulatory Updates</h3>
              <div className="space-y-4">
                {newsItems.slice(0, 3).map((item) => (
                  <div key={item.id} className="border rounded-md p-3">
                    <div className="flex justify-between items-start">
                      <h4 className="font-medium">{item.title}</h4>
                      {item.isNew && (
                        <Badge className="bg-blue-100 text-blue-800">New</Badge>
                      )}
                    </div>
                    <div className="flex items-center gap-2 mt-1">
                      <span className="text-xs text-gray-500">{item.source}</span>
                      <span className="text-xs text-gray-500">•</span>
                      <span className="text-xs text-gray-500">{item.date}</span>
                    </div>
                    <Badge variant="outline" className="mt-2">
                      {item.category}
                    </Badge>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="api">
          <Card>
            <CardHeader>
              <CardTitle>API Reference</CardTitle>
              <CardDescription>
                Documentation for regulatory API endpoints
              </CardDescription>
            </CardHeader>
            <CardContent>
              <p className="mb-4">
                The regulations system provides several API endpoints to interact with standards and news data.
                These endpoints support authentication and hospital-specific data access.
              </p>
              
              <div className="space-y-6">
                {apiEndpoints.map((endpoint) => (
                  <div key={endpoint.name} className="border rounded-md overflow-hidden">
                    <div className="bg-gray-50 p-3 border-b">
                      <h3 className="font-mono font-medium">{endpoint.name}</h3>
                    </div>
                    <div className="p-3">
                      <p className="text-sm mb-3">{endpoint.description}</p>
                      <div className="flex flex-wrap gap-2">
                        {endpoint.methods.map((method) => (
                          <Badge 
                            key={method} 
                            className={
                              method === 'GET' ? 'bg-green-100 text-green-800' :
                              method === 'POST' ? 'bg-blue-100 text-blue-800' :
                              method === 'PUT' ? 'bg-amber-100 text-amber-800' :
                              'bg-red-100 text-red-800'
                            }
                          >
                            {method}
                          </Badge>
                        ))}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
              
              <div className="mt-6 bg-blue-50 p-4 rounded-md">
                <div className="flex items-start">
                  <Sparkles className="h-5 w-5 text-blue-500 mr-2 mt-0.5" />
                  <div>
                    <h3 className="font-medium">Development Note</h3>
                    <p className="text-sm mt-1">
                      In development mode, the API endpoints will fall back to sample data if the actual 
                      endpoints are unavailable. This allows for continued testing and development without
                      requiring a fully operational backend.
                    </p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
} 