'use client'

import React, { useState, useEffect, Suspense } from 'react'
import { toast } from 'sonner'

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Progress } from '@/components/ui/progress'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Separator } from '@/components/ui/separator'
import { Skeleton } from '@/components/ui/skeleton'

// Icons
import {
  BadgeCheck,
  FileCheck,
  Calendar,
  UserCheck,
  AlertTriangle,
  BookOpen,
  CheckCircle2,
  Clock,
  ListChecks,
  ChevronRight,
  Globe,
  Info,
  AlertCircle
} from 'lucide-react'

import RecentStandards from '@/components/dashboard/RecentStandards'
import { DynamicAuditFindings } from '@/components/dashboard/DynamicAuditFindings'
import { useDashboardData, ActionItem as ActionItemType, ComplianceCard as ComplianceCardType, RegulatoryNews as RegulatoryNewsType } from '@/hooks/useDashboardData'
import { ComplianceStats } from '@/components/dashboard/ComplianceStats'

// Add type definitions
interface MetricsRowProps {
  loading: boolean;
  metrics: {
    auditCompletedCount: number;
    upcomingDeadlinesCount: number;
    staffCompliancePercentage: number;
    totalStandardsCount: number;
    complianceScore: number;
    previousComplianceScore: number;
    standardsData?: any[];
    trends?: {
      totalStandards: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      auditsCompleted: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      upcomingDeadlines: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      staffCompliance: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
    };
  };
  onStaffTrainingReminder: () => void;
  onDeadlineExtension: () => void;
  onScheduleAudit: () => void;
}

interface StatCardProps {
  title: string;
  value: number | string;
  description?: string;
  icon: React.ReactNode;
  trend: {
    value: number;
    isPositive?: boolean;
    isNeutral?: boolean;
    label: string;
  };
  tooltip: string;
  remediationAction?: {
    label: string;
    action: () => void;
  };
}

interface ComplianceScoreProps {
  score: number;
  previousScore: number;
}

interface FindingsProps {
  loading: boolean;
  findings: any[];
}

interface ComplianceCardProps {
  title: string;
  description: string;
  standard: string;
  status: string;
  dueDate: string;
  className?: string;
}

interface ActionItemsProps {
  items: ActionItemType[];
}

// Loading component for immediate page render
const DashboardSkeleton = () => (
    <div className="space-y-6">
    {/* Metrics Row Skeleton */}
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
      <div className="lg:col-span-1">
        <Card className="h-full">
          <CardHeader className="pb-2">
            <Skeleton className="h-5 w-48" />
          </CardHeader>
          <CardContent className="pt-0">
            <div className="flex items-center justify-center mb-4">
              <Skeleton className="w-32 h-32 rounded-full" />
            </div>
            <div className="text-center">
              <Skeleton className="h-6 w-24 mx-auto mb-2" />
              <Skeleton className="h-4 w-32 mx-auto" />
            </div>
          </CardContent>
        </Card>
      </div>
      
      <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
        {[1, 2, 3, 4].map(i => (
          <Card key={i} className="overflow-hidden h-full">
            <CardHeader className="pb-2 space-y-0">
              <Skeleton className="h-4 w-24" />
            </CardHeader>
            <CardContent className="pt-0">
              <Skeleton className="h-8 w-16 mb-2" />
              <Skeleton className="h-4 w-32" />
            </CardContent>
          </Card>
        ))}
      </div>
    </div>

    {/* Compliance Cards Skeleton */}
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-5">
      {[1, 2, 3].map(i => (
        <Card key={i} className="h-full">
          <CardHeader>
            <Skeleton className="h-5 w-48" />
            <Skeleton className="h-4 w-32" />
          </CardHeader>
          <CardContent>
            <Skeleton className="h-6 w-16 mb-4" />
            <Skeleton className="h-2 w-full" />
          </CardContent>
        </Card>
      ))}
    </div>
      
    {/* Chart and Findings Skeleton */}
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <Card className="h-full">
        <CardHeader>
          <Skeleton className="h-5 w-32" />
          <Skeleton className="h-4 w-48" />
        </CardHeader>
        <CardContent>
          <Skeleton className="h-64 w-full" />
        </CardContent>
      </Card>
      
      <Card className="h-full">
        <CardHeader>
          <Skeleton className="h-5 w-32" />
          <Skeleton className="h-4 w-48" />
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {[1, 2, 3].map(i => (
              <div key={i}>
                <Skeleton className="h-4 w-full mb-2" />
                <Skeleton className="h-3 w-3/4" />
                {i < 3 && <Separator className="mt-4" />}
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  </div>
)

// Lazy-loaded dashboard content
const DashboardContent = React.lazy(() => 
  import('./dashboard-content').then(module => ({ default: module.DashboardContent }))
)

export default function DashboardPage() {
  return (
    <Suspense fallback={<DashboardSkeleton />}>
      <DashboardContent />
    </Suspense>
  )
}

// Components for dashboard
const MetricsRow = ({
  loading,
  metrics,
  onStaffTrainingReminder,
  onDeadlineExtension,
  onScheduleAudit
}: MetricsRowProps) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
        <div className="lg:col-span-1">
          <Card className="h-full">
            <CardHeader className="pb-2">
              <CardTitle className="text-lg font-medium">Overall Compliance Score</CardTitle>
            </CardHeader>
            <CardContent className="pt-0">
              <div className="flex items-center justify-center mb-4">
                <Skeleton className="w-32 h-32 rounded-full" />
              </div>
              <div className="text-center">
                <Skeleton className="h-6 w-24 mx-auto mb-2" />
                <Skeleton className="h-4 w-32 mx-auto" />
              </div>
            </CardContent>
          </Card>
        </div>
        
        <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
          {[1, 2, 3, 4].map(i => (
            <Card key={i} className="overflow-hidden h-full">
              <CardHeader className="pb-2 space-y-0">
                <Skeleton className="h-4 w-24" />
              </CardHeader>
              <CardContent className="pt-0">
                <Skeleton className="h-8 w-16 mb-2" />
                <Skeleton className="h-4 w-32" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  const trends = metrics.trends || {
    totalStandards: { value: 5, isPositive: true, label: "from last month" },
    auditsCompleted: { value: 2, isPositive: true, label: "vs. last month" },
    upcomingDeadlines: { value: 2, isNeutral: true, label: "in next 14 days" },
    staffCompliance: { value: 3, isPositive: false, label: "below target" }
  };

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
      <div className="lg:col-span-1">
        <ComplianceScore 
          score={metrics.complianceScore} 
          previousScore={metrics.previousComplianceScore} 
        />
      </div>
      
      <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
        <StatCard 
          title="Total Standards"
          value={metrics.totalStandardsCount}
          icon={<BadgeCheck className="h-5 w-5 text-blue-600" />}
          trend={trends.totalStandards}
          tooltip="Total number of regulatory standards across all frameworks that apply to your hospital operations"
        />
        
        <StatCard 
          title="Audits Completed"
          value={metrics.auditCompletedCount}
          description="This Month"
          icon={<FileCheck className="h-5 w-5 text-green-600" />}
          trend={trends.auditsCompleted}
          tooltip="Internal and external audits completed to verify compliance with regulatory standards"
          remediationAction={{
            label: "Schedule ISO 9001 Audit",
            action: onScheduleAudit
          }}
        />
        
        <StatCard 
          title="Upcoming Deadlines"
          value={metrics.upcomingDeadlinesCount}
          icon={<Calendar className="h-5 w-5 text-amber-600" />}
          trend={trends.upcomingDeadlines}
          tooltip="Critical compliance deadlines requiring immediate attention to maintain regulatory status"
          remediationAction={{
            label: "Review & Prioritize",
            action: onDeadlineExtension
          }}
        />
        
        <StatCard 
          title="Staff Compliance"
          value={`${metrics.staffCompliancePercentage}%`}
          icon={<UserCheck className="h-5 w-5 text-blue-600" />}
          trend={trends.staffCompliance}
          tooltip="Percentage of staff who have completed all required compliance training modules"
          remediationAction={{
            label: "Send Training Reminders",
            action: onStaffTrainingReminder
          }}
        />
      </div>
    </div>
  );
};

const ComplianceScore = ({ score, previousScore }: ComplianceScoreProps) => {
  const getScoreColor = (value: number) => {
    if (value >= 90) return 'text-green-500';
    if (value >= 70) return 'text-amber-500';
    return 'text-red-500';
  };
  
  const scoreTrend = previousScore ? score - previousScore : 0;
  
  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg font-medium">Overall Compliance Score</CardTitle>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="flex items-center justify-center mb-4">
          <div className="w-32 h-32 flex items-center justify-center relative">
            <svg viewBox="0 0 100 100" className="w-full h-full">
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke="#E5E7EB"
                strokeWidth="10"
              />
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke={score >= 90 ? '#66BB6A' : score >= 70 ? '#FFC107' : '#F44336'}
                strokeWidth="10"
                strokeDasharray={`${2 * Math.PI * 45 * (score / 100)} ${2 * Math.PI * 45 * (1 - score / 100)}`}
                strokeDashoffset={2 * Math.PI * 45 * 0.25}
                transform="rotate(-90 50 50)"
              />
            </svg>
            <div className="absolute text-2xl font-bold">{score}%</div>
          </div>
        </div>
        
        <div className="text-center">
          <div className={`text-lg font-semibold mb-1 ${getScoreColor(score)}`}>
            {score >= 90 ? 'Excellent' : 
             score >= 80 ? 'Good' : 
             score >= 70 ? 'Average' : 
             score >= 60 ? 'Below Average' : 
             'Critical'}
          </div>
          
          {previousScore !== undefined && (
            <div className="flex items-center justify-center text-sm">
              <span className="text-gray-500 mr-1">Previous: {previousScore}%</span>
              <span className={
                scoreTrend > 0 ? "text-green-600 font-medium" : 
                scoreTrend < 0 ? "text-red-600 font-medium" : 
                "text-gray-500"
              }>
                ({scoreTrend > 0 ? '+' : ''}{scoreTrend}%)
              </span>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
};

const StatCard = ({
  title,
  value,
  description = "",
  icon,
  trend,
  tooltip,
  remediationAction
}: StatCardProps) => {
  return (
    <Card className="overflow-hidden h-full">
      <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
        <div className="flex items-center gap-2">
          <CardTitle className="text-sm font-medium text-gray-600">{title}</CardTitle>
          {tooltip && (
            <div className="group relative">
              <Info className="h-4 w-4 text-gray-400 hover:text-blue-500 cursor-help" />
              <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 -translate-y-2 hidden group-hover:block bg-white border border-gray-200 p-2 rounded shadow-lg z-10 w-48">
                <p className="text-xs text-gray-700">{tooltip}</p>
              </div>
            </div>
          )}
        </div>
        {icon && <div>{icon}</div>}
      </CardHeader>
      <CardContent className="pt-0">
        <div className="text-2xl font-bold">{value}</div>
        {description && (
          <p className="text-xs text-gray-500 mt-1">{description}</p>
        )}
        {trend && (
          <div className="flex items-center mt-2">
            <span className={
              trend.isPositive ? "text-xs font-medium text-green-600" : 
              trend.isNeutral ? "text-xs font-medium text-gray-500" : 
              "text-xs font-medium text-red-600"
            }>
              {trend.isPositive ? '+' : trend.isNeutral ? '' : '-'}{Math.abs(trend.value)}%
            </span>
            {trend.label && (
              <span className="text-xs text-gray-500 ml-1">
                {trend.label}
              </span>
            )}
          </div>
        )}
        
        {remediationAction && (
          <button 
            onClick={remediationAction.action} 
            className="mt-3 text-xs px-3 py-1.5 bg-blue-500 text-white rounded-md hover:bg-blue-600 transition-colors w-full"
          >
            {remediationAction.label}
          </button>
        )}
      </CardContent>
    </Card>
  );
};

const ComplianceCardsRow = ({ loading, complianceCards }: { loading: boolean; complianceCards: ComplianceCardType[] }) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-5">
        {[1, 2, 3].map(i => (
          <Card key={i} className="overflow-hidden h-full">
            <CardHeader className="pb-2">
              <Skeleton className="h-4 w-32 mb-2" />
              <Skeleton className="h-3 w-24" />
            </CardHeader>
            <CardContent className="pt-0">
              <Skeleton className="h-4 w-full mb-3" />
              <div className="flex items-center justify-between">
                <Skeleton className="h-6 w-20" />
                <Skeleton className="h-4 w-24" />
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }
  
  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-5">
      {complianceCards.map((card, index) => {
        // Determine border color based on status
        const borderColorClass = card.status === 'compliant' ? 'border-l-4 border-green-500' : 
                                 card.status === 'at-risk' ? 'border-l-4 border-amber-500' : 
                                 'border-l-4 border-red-500';
        
        return (
      <ComplianceCard
            key={index}
            title={card.title}
            description={card.description}
            standard={card.standard}
            status={card.status}
            dueDate={card.dueDate}
            className={borderColorClass}
          />
        );
      })}
    </div>
  );
};

const ComplianceCard = ({
  title,
  description,
  standard,
  status,
  dueDate,
  className = ""
}: ComplianceCardProps) => {
  const getStatusIcon = () => {
    switch (status) {
      case 'compliant':
        return <CheckCircle2 className="h-5 w-5 text-green-600" />;
      case 'partial':
        return <AlertTriangle className="h-5 w-5 text-amber-600" />;
      case 'non-compliant':
        return <AlertCircle className="h-5 w-5 text-red-600" />;
      default:
        return null;
    }
  };
  
  const getStatusText = () => {
    switch (status) {
      case 'compliant':
        return 'Compliant';
      case 'partial':
        return 'Partially Compliant';
      case 'non-compliant':
        return 'Non-Compliant';
      default:
        return '';
    }
  };
  
  const getStatusBadgeClass = () => {
    switch (status) {
      case 'compliant':
        return 'bg-green-100 text-green-800';
      case 'partial':
        return 'bg-amber-100 text-amber-800';
      case 'non-compliant':
        return 'bg-red-100 text-red-800';
      default:
        return '';
    }
  };

  return (
    <Card className={`overflow-hidden h-full ${className || ''}`}>
      <div className="flex">
        <div className={
          status === 'compliant' ? 'bg-green-500 w-1 h-full' : 
          status === 'partial' ? 'bg-amber-500 w-1 h-full' : 
          'bg-red-500 w-1 h-full'
        }></div>
        <div className="flex-1">
          <CardHeader className="pb-2">
            <div className="flex items-start justify-between">
              <div>
                <CardTitle className="text-base font-medium">{title}</CardTitle>
                <CardDescription className="text-xs mt-1">{standard}</CardDescription>
              </div>
              {getStatusIcon()}
            </div>
          </CardHeader>
          <CardContent className="pt-0">
            {description && (
              <p className="text-sm text-gray-600 mb-3">{description}</p>
            )}
            <div className="flex items-center justify-between">
              <span className={`text-xs px-2 py-1 rounded ${getStatusBadgeClass()}`}>{getStatusText()}</span>
              {dueDate && (
                <span className="text-xs text-gray-500">Due: {new Date(dueDate).toLocaleDateString()}</span>
              )}
            </div>
          </CardContent>
        </div>
      </div>
    </Card>
  );
};

const ComplianceChartAndFindings = ({ loading, findings }: FindingsProps) => {
  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <ComplianceStats />
      
      <DynamicAuditFindings 
        loading={loading} 
        findings={findings} 
      />
    </div>
  );
};

const ComplianceChart = () => {
  const complianceData = [
    { name: 'ISO 9001', compliance: 87, required: 90 },
    { name: 'HIPAA', compliance: 94, required: 90 },
    { name: 'JCI', compliance: 78, required: 90 },
    { name: 'GDPR', compliance: 82, required: 90 },
    { name: 'FDA 21 CFR', compliance: 76, required: 90 },
    { name: 'OSHA', compliance: 88, required: 90 },
    { name: 'ISO 27001', compliance: 81, required: 90 },
    { name: 'ISO 14001', compliance: 79, required: 85 }
  ];
  
  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle>Compliance Metrics</CardTitle>
        <CardDescription>Current compliance level vs. required threshold</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {complianceData.map((item, index) => (
            <div key={index} className="space-y-2">
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <span className={`h-2 w-2 rounded-full mr-2 ${
                    item.compliance >= item.required ? 'bg-green-500' : 'bg-red-500'
                  }`} />
                  <span className="text-sm font-medium">{item.name}</span>
                </div>
                <div className="flex items-center">
                  <span className="text-sm font-medium mr-2">{item.compliance}%</span>
                  <span className="text-xs text-gray-500">/ {item.required}%</span>
                </div>
              </div>
              <div className="h-2 bg-gray-200 rounded-full">
                <div 
                  className={`h-full rounded-full ${
                    item.compliance >= item.required ? 'bg-green-500' : 'bg-red-500'
                  }`} 
                  style={{ width: `${item.compliance}%` }}
                />
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};

const ActionItemsAndFeed = ({ loading, actionItems, regulatoryNews }: { loading: boolean; actionItems: ActionItemType[]; regulatoryNews: RegulatoryNewsType[] }) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
        <Card className="h-full">
          <CardHeader className="pb-2">
            <Skeleton className="h-5 w-24" />
            <Skeleton className="h-4 w-48 mt-1" />
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {[1, 2, 3].map(i => (
                <div key={i} className="p-3 rounded-md bg-gray-50">
                  <Skeleton className="h-4 w-full mb-2" />
                  <Skeleton className="h-3 w-3/4" />
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
        
        <Card className="h-full">
          <CardHeader className="pb-2">
            <Skeleton className="h-5 w-32" />
            <Skeleton className="h-4 w-48 mt-1" />
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {[1, 2, 3].map(i => (
                <div key={i}>
                  <Skeleton className="h-4 w-full mb-1" />
                  <Skeleton className="h-3 w-3/4" />
                  {i < 3 && <Separator className="mt-4" />}
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <ActionItems items={actionItems} />
      
      <Card className="h-full">
        <CardHeader className="pb-2">
          <CardTitle>Regulatory Updates</CardTitle>
          <CardDescription>Recent changes to healthcare standards</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {regulatoryNews.map((news, index) => (
              <React.Fragment key={news.id}>
            <div>
                  <div className="flex items-start justify-between">
                    <h4 className="text-sm font-medium">{news.title}</h4>
                    {news.severity === 'warning' && (
                      <AlertTriangle className="h-4 w-4 text-amber-500 flex-shrink-0 ml-2" />
                    )}
            </div>
                  <p className="text-xs text-gray-500 mt-1">{news.description}</p>
                  <p className="text-xs text-gray-400 mt-1">
                    {new Date(news.date).toLocaleDateString()} • {news.category}
              </p>
            </div>
                {index < regulatoryNews.length - 1 && <Separator />}
              </React.Fragment>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

const ActionItems = ({ items }: ActionItemsProps) => {
  const priorityColors = {
    high: 'text-red-500',
    medium: 'text-amber-500',
    low: 'text-blue-500'
  };
  
  const priorityBg = {
    high: 'bg-red-50',
    medium: 'bg-amber-50',
    low: 'bg-blue-50'
  };
  
  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle>Action Items</CardTitle>
        <CardDescription>Tasks requiring your attention</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-2">
          {items.map((item: ActionItemType, index: number) => (
            <div 
              key={item.id} 
              className={`p-3 rounded-md ${priorityBg[item.priority as keyof typeof priorityBg]} border border-${item.priority === 'high' ? 'red' : item.priority === 'medium' ? 'amber' : 'blue'}-100`}
            >
              <div className="flex items-start justify-between">
                <div>
                  <h4 className="text-sm font-medium">{item.title}</h4>
                  <div className="flex items-center text-xs text-gray-500 mt-1">
                    <span>Due: {new Date(item.dueDate).toLocaleDateString()}</span>
                    <span className="mx-2">•</span>
                    <span>Assignee: {item.assignee}</span>
                  </div>
                </div>
                <div className={`text-xs font-medium ${priorityColors[item.priority as keyof typeof priorityColors]}`}>
                  {item.priority.charAt(0).toUpperCase() + item.priority.slice(1)} Priority
                </div>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}; 