'use client'

import React, { useState, useEffect } from 'react'
import { toast } from 'sonner'

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Progress } from '@/components/ui/progress'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Separator } from '@/components/ui/separator'
import { Skeleton } from '@/components/ui/skeleton'

// Icons
import {
  BadgeCheck,
  FileCheck,
  Calendar,
  UserCheck,
  AlertTriangle,
  BookOpen,
  CheckCircle2,
  Clock,
  ListChecks,
  ChevronRight,
  Globe,
  Info,
  AlertCircle
} from 'lucide-react'

import RecentStandards from '@/components/dashboard/RecentStandards'
import { DynamicAuditFindings } from '@/components/dashboard/DynamicAuditFindings'
import { useDashboardData, ActionItem as ActionItemType, ComplianceCard as ComplianceCardType, RegulatoryNews as RegulatoryNewsType } from '@/hooks/useDashboardData'
import { ComplianceStats } from '@/components/dashboard/ComplianceStats'

// Add type definitions
interface MetricsRowProps {
  loading: boolean;
  metrics: {
    auditCompletedCount: number;
    upcomingDeadlinesCount: number;
    staffCompliancePercentage: number;
    totalStandardsCount: number;
    complianceScore: number;
    previousComplianceScore: number;
    standardsData?: any[];
    trends?: {
      totalStandards: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      auditsCompleted: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      upcomingDeadlines: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
      staffCompliance: { value: number; isPositive?: boolean; isNeutral?: boolean; label: string; };
    };
  };
  onStaffTrainingReminder: () => void;
  onDeadlineExtension: () => void;
  onScheduleAudit: () => void;
}

interface ComplianceCardsRowProps {
  loading: boolean;
  complianceCards: ComplianceCardType[];
}

interface FindingsProps {
  loading: boolean;
  findings: any[];
}

export const DashboardContent = () => {
  // Get dynamic dashboard data
  const { 
    metrics, 
    recentFindings, 
    complianceCards, 
    actionItems, 
    regulatoryNews, 
    loading, 
    error 
  } = useDashboardData();
  
  // Remediation action handlers
  const handleStaffTrainingReminder = () => {
    toast.success('Staff compliance training reminders sent to all departments');
    // In a real implementation, this would trigger an API call
  };

  const handleDeadlineExtension = () => {
    toast.success('Selected deadlines have been reviewed and prioritized');
    // In a real implementation, this would trigger an API call
  };

  const handleScheduleAudit = () => {
    toast.success('ISO 9001 compliance audit has been scheduled');
    // In a real implementation, this would trigger an API call
  };

  return (
    <div className="space-y-6">
      {/* Top row - Quick metrics */}
      <MetricsRow 
        loading={loading}
        metrics={metrics}
        onStaffTrainingReminder={handleStaffTrainingReminder}
        onDeadlineExtension={handleDeadlineExtension}
        onScheduleAudit={handleScheduleAudit}
      />
      
      {/* Second row - Compliance Cards */}
      <ComplianceCardsRow loading={loading} complianceCards={complianceCards} />
      
      {/* Third row - Chart and Findings */}
      <ComplianceChartAndFindings 
        loading={loading} 
        findings={recentFindings} 
      />
      
      {/* Fourth row - Action Items and News Feed */}
      <ActionItemsAndFeed loading={loading} actionItems={actionItems} regulatoryNews={regulatoryNews} />
    </div>
  )
}

// Components for dashboard
const MetricsRow = ({
  loading,
  metrics,
  onStaffTrainingReminder,
  onDeadlineExtension,
  onScheduleAudit
}: MetricsRowProps) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
        <div className="lg:col-span-1">
          <Card className="h-full">
            <CardHeader className="pb-2">
              <CardTitle className="text-lg font-medium">Overall Compliance Score</CardTitle>
            </CardHeader>
            <CardContent className="pt-0">
              <div className="flex items-center justify-center mb-4">
                <Skeleton className="w-32 h-32 rounded-full" />
              </div>
              <div className="text-center">
                <Skeleton className="h-6 w-24 mx-auto mb-2" />
                <Skeleton className="h-4 w-32 mx-auto" />
              </div>
            </CardContent>
          </Card>
        </div>
        
        <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
          {[1, 2, 3, 4].map(i => (
            <Card key={i} className="overflow-hidden h-full">
              <CardHeader className="pb-2 space-y-0">
                <Skeleton className="h-4 w-24" />
              </CardHeader>
              <CardContent className="pt-0">
                <Skeleton className="h-8 w-16 mb-2" />
                <Skeleton className="h-4 w-32" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  const trends = metrics.trends || {
    totalStandards: { value: 5, isPositive: true, label: "from last month" },
    auditsCompleted: { value: 2, isPositive: true, label: "vs. last month" },
    upcomingDeadlines: { value: 2, isNeutral: true, label: "in next 14 days" },
    staffCompliance: { value: 3, isPositive: false, label: "below target" }
  };

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
      <div className="lg:col-span-1">
        <ComplianceScore score={metrics.complianceScore} previousScore={metrics.previousComplianceScore} />
      </div>
      
      <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
        <StatCard 
          title="Total Standards"
          value={metrics.totalStandardsCount}
          icon={<BadgeCheck className="h-5 w-5 text-blue-600" />}
          trend={trends.totalStandards}
          tooltip="Total number of regulatory standards across all frameworks that apply to your hospital operations"
        />
        
        <StatCard 
          title="Audits Completed"
          value={metrics.auditCompletedCount}
          description="This Quarter"
          icon={<FileCheck className="h-5 w-5 text-green-600" />}
          trend={trends.auditsCompleted}
          tooltip="Internal and external audits completed to verify compliance with regulatory standards"
          remediationAction={{
            label: "Schedule ISO 9001 Audit",
            action: onScheduleAudit
          }}
        />
        
        <StatCard 
          title="Upcoming Deadlines"
          value={metrics.upcomingDeadlinesCount}
          icon={<Calendar className="h-5 w-5 text-amber-600" />}
          trend={trends.upcomingDeadlines}
          tooltip="Critical compliance deadlines requiring immediate attention to maintain regulatory status"
          remediationAction={{
            label: "Review & Prioritize",
            action: onDeadlineExtension
          }}
        />
        
        <StatCard 
          title="Staff Compliance"
          value={`${metrics.staffCompliancePercentage}%`}
          icon={<UserCheck className="h-5 w-5 text-blue-600" />}
          trend={trends.staffCompliance}
          tooltip="Percentage of staff who have completed all required compliance training modules"
          remediationAction={{
            label: "Send Training Reminders",
            action: onStaffTrainingReminder
          }}
        />
      </div>
    </div>
  );
};

interface ComplianceScoreProps {
  score: number;
  previousScore?: number;
}

const ComplianceScore: React.FC<ComplianceScoreProps> = ({ score, previousScore }) => {
  const getScoreColor = (value: number): string => {
    if (value >= 90) return 'text-green-500';
    if (value >= 70) return 'text-amber-500';
    return 'text-red-500';
  };
  
  const scoreTrend = previousScore ? score - previousScore : 0;
  
  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg font-medium">Overall Compliance Score</CardTitle>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="flex items-center justify-center mb-4">
          <div className="w-32 h-32 flex items-center justify-center relative">
            <svg viewBox="0 0 100 100" className="w-full h-full">
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke="#E5E7EB"
                strokeWidth="10"
              />
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke={score >= 90 ? '#66BB6A' : score >= 70 ? '#FFC107' : '#F44336'}
                strokeWidth="10"
                strokeDasharray={`${2 * Math.PI * 45 * (score / 100)} ${2 * Math.PI * 45 * (1 - score / 100)}`}
                strokeDashoffset={2 * Math.PI * 45 * 0.25}
                transform="rotate(-90 50 50)"
              />
            </svg>
            <div className="absolute text-2xl font-bold">{score}%</div>
          </div>
        </div>
        
        <div className="text-center">
          <div className={`text-lg font-semibold mb-1 ${getScoreColor(score)}`}>
            {score >= 90 ? 'Excellent' : 
             score >= 80 ? 'Good' : 
             score >= 70 ? 'Average' : 
             score >= 60 ? 'Below Average' : 
             'Critical'}
          </div>
          
          {previousScore !== undefined && (
            <div className="flex items-center justify-center text-sm">
              <span className="text-gray-500 mr-1">Previous: {previousScore}%</span>
              <span className={
                scoreTrend > 0 ? "text-green-600 font-medium" : 
                scoreTrend < 0 ? "text-red-600 font-medium" : 
                "text-gray-600"
              }>
                {scoreTrend > 0 ? `+${scoreTrend}%` : 
                 scoreTrend < 0 ? `${scoreTrend}%` : 
                 'No change'}
              </span>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
};

interface TrendInfo {
  value: number;
  isPositive?: boolean;
  isNeutral?: boolean;
  label?: string;
}

interface RemediationAction {
  label: string;
  action: () => void;
}

interface StatCardProps {
  title: string;
  value: string | number;
  description?: string;
  icon?: React.ReactNode;
  trend?: TrendInfo;
  tooltip?: string;
  remediationAction?: RemediationAction;
}

const StatCard: React.FC<StatCardProps> = ({
  title,
  value,
  description,
  icon,
  trend,
  tooltip,
  remediationAction
}) => {
  return (
    <Card className="overflow-hidden h-full">
      <CardHeader className="pb-2 space-y-0">
        <div className="flex items-center space-x-2">
          {icon}
          <CardTitle className="text-sm font-medium">{title}</CardTitle>
        </div>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="text-2xl font-bold">{value}</div>
        {description && (
          <p className="text-xs text-muted-foreground">{description}</p>
        )}
        {trend && (
          <div className="flex items-center text-xs mt-1">
            <span className={
              trend.isPositive ? "text-green-600 font-medium" :
              trend.isNeutral ? "text-blue-600" :
              "text-red-600 font-medium"
            }>
              {trend.isPositive ? "+" : ""}{trend.value} {trend.label}
            </span>
          </div>
        )}
        {remediationAction && (
          <Button 
            variant="outline" 
            size="sm" 
            className="mt-2 text-xs h-7"
            onClick={remediationAction.action}
          >
            {remediationAction.label}
          </Button>
        )}
      </CardContent>
    </Card>
  );
};

const ComplianceCardsRow = ({ loading, complianceCards }: ComplianceCardsRowProps) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-5">
        {[1, 2, 3].map(i => (
          <Card key={i} className="h-full">
            <CardHeader>
              <Skeleton className="h-5 w-48" />
              <Skeleton className="h-4 w-32" />
            </CardHeader>
            <CardContent>
              <Skeleton className="h-6 w-16 mb-4" />
              <Skeleton className="h-2 w-full" />
            </CardContent>
          </Card>
        ))}
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-5">
      {complianceCards.map((card, index) => (
        <ComplianceCard key={card.title} {...card} />
      ))}
    </div>
  );
};

interface ComplianceCardProps {
  title: string;
  description: string;
  standard: string;
  status: string;
  dueDate: string;
  className?: string;
}

const ComplianceCard: React.FC<ComplianceCardProps> = ({
  title,
  description,
  standard,
  status,
  dueDate,
  className
}) => {
  const getStatusIcon = () => {
    switch (status) {
      case 'compliant':
        return <CheckCircle2 className="h-4 w-4 text-green-600" />;
      case 'at-risk':
        return <Clock className="h-4 w-4 text-amber-600" />;
      case 'non-compliant':
        return <AlertTriangle className="h-4 w-4 text-red-600" />;
      default:
        return <Info className="h-4 w-4 text-blue-600" />;
    }
  };

  const getStatusText = () => {
    switch (status) {
      case 'compliant':
        return 'Compliant';
      case 'at-risk':
        return 'At Risk';
      case 'non-compliant':
        return 'Non-Compliant';
      default:
        return 'Unknown';
    }
  };

  const getStatusBadgeClass = () => {
    switch (status) {
      case 'compliant':
        return 'bg-green-100 text-green-800 hover:bg-green-200';
      case 'at-risk':
        return 'bg-amber-100 text-amber-800 hover:bg-amber-200';
      case 'non-compliant':
        return 'bg-red-100 text-red-800 hover:bg-red-200';
      default:
        return 'bg-blue-100 text-blue-800 hover:bg-blue-200';
    }
  };

  return (
    <Card className={`h-full hover:shadow-md transition-shadow ${className}`}>
      <CardHeader className="pb-2">
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <CardTitle className="text-sm font-medium leading-5 mb-1">{title}</CardTitle>
            <CardDescription className="text-xs text-gray-600 line-clamp-2">
              {description}
            </CardDescription>
          </div>
          <Badge className={`ml-2 text-xs font-medium px-2 py-1 ${getStatusBadgeClass()}`}>
            <span className="flex items-center gap-1">
              {getStatusIcon()}
              {getStatusText()}
            </span>
          </Badge>
        </div>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="flex items-center justify-between text-xs text-gray-500">
          <span className="font-medium">{standard}</span>
          <span>Due: {dueDate}</span>
        </div>
      </CardContent>
    </Card>
  );
};

const ComplianceChartAndFindings = ({ loading, findings }: FindingsProps) => {
  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <ComplianceStats />
      
      <DynamicAuditFindings 
        loading={loading} 
        findings={findings} 
      />
    </div>
  );
};

const ActionItemsAndFeed = ({ loading, actionItems, regulatoryNews }: { loading: boolean; actionItems: ActionItemType[]; regulatoryNews: RegulatoryNewsType[] }) => {
  if (loading) {
    return (
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
        <Card className="h-full">
          <CardHeader className="pb-2">
            <Skeleton className="h-5 w-24" />
            <Skeleton className="h-4 w-48 mt-1" />
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {[1, 2, 3].map(i => (
                <div key={i} className="p-3 rounded-md bg-gray-50">
                  <Skeleton className="h-4 w-full mb-2" />
                  <Skeleton className="h-3 w-3/4" />
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
        
        <Card className="h-full">
          <CardHeader className="pb-2">
            <Skeleton className="h-5 w-32" />
            <Skeleton className="h-4 w-48 mt-1" />
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {[1, 2, 3].map(i => (
                <div key={i}>
                  <Skeleton className="h-4 w-full mb-1" />
                  <Skeleton className="h-3 w-3/4" />
                  {i < 3 && <Separator className="mt-4" />}
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <ActionItems items={actionItems} />
      <RegulatoryFeed news={regulatoryNews} />
    </div>
  );
};

interface ActionItem {
  id: string;
  title: string;
  dueDate: string;
  priority: string;
  completed: boolean;
  category: string;
  description?: string;
  assignedTo?: string;
}

interface ActionItemsProps {
  items: ActionItem[];
}

const ActionItems: React.FC<ActionItemsProps> = ({ items }) => {
  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high':
        return 'bg-red-100 text-red-800';
      case 'medium':
        return 'bg-amber-100 text-amber-800';
      case 'low':
        return 'bg-green-100 text-green-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg font-medium flex items-center">
          <ListChecks className="h-5 w-5 mr-2 text-blue-600" />
          Action Items
        </CardTitle>
        <CardDescription>Pending compliance tasks</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-3">
          {items.map((item) => (
            <div key={item.id} className="p-3 rounded-md bg-gray-50 hover:bg-gray-100 transition-colors">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <p className="font-medium text-sm text-gray-900">{item.title}</p>
                  <div className="flex items-center gap-2 mt-1">
                    <Badge className={`text-xs px-2 py-0.5 ${getPriorityColor(item.priority)}`}>
                      {item.priority.charAt(0).toUpperCase() + item.priority.slice(1)}
                    </Badge>
                    <span className="text-xs text-gray-500">Due: {item.dueDate}</span>
                  </div>
                </div>
                <ChevronRight className="h-4 w-4 text-gray-400 mt-0.5" />
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};

interface NewsItem {
  id: string;
  title: string;
  source: string;
  date: string;
  category: string;
  link: string;
  isNew?: boolean;
}

interface RegulatoryFeedProps {
  news: NewsItem[];
}

const RegulatoryFeed: React.FC<RegulatoryFeedProps> = ({ news }) => {
  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'regulation':
        return <BookOpen className="h-4 w-4 text-blue-600" />;
      case 'safety':
        return <AlertTriangle className="h-4 w-4 text-amber-600" />;
      case 'security':
        return <Info className="h-4 w-4 text-red-600" />;
      default:
        return <Globe className="h-4 w-4 text-gray-600" />;
    }
  };

  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg font-medium flex items-center">
          <Globe className="h-5 w-5 mr-2 text-blue-600" />
          Regulatory News
        </CardTitle>
        <CardDescription>Latest regulatory updates</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {news.map((item, index) => (
            <div key={item.id}>
              <div className="flex items-start space-x-3">
                {getCategoryIcon(item.category)}
                <div className="flex-1 min-w-0">
                  <div className="flex items-center gap-2">
                    <p className="font-medium text-sm text-gray-900 truncate">{item.title}</p>
                    {item.isNew && (
                      <Badge className="bg-blue-100 text-blue-800 text-xs px-1.5 py-0.5">New</Badge>
                    )}
                  </div>
                  <p className="text-xs text-gray-500 mt-0.5">
                    {item.source} • {item.date}
                  </p>
                </div>
              </div>
              {index < news.length - 1 && <Separator className="mt-4" />}
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}; 