'use client'

import React, { ReactNode, useState, useMemo, useCallback, Suspense, lazy, useEffect } from 'react'
import { usePathname, useRouter } from 'next/navigation'
import Link from 'next/link'
import Sidebar from '@/components/layout/Sidebar'
import { Input } from '@/components/ui/input'
import { Button } from '@/components/ui/button'
import { 
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu'
import { HomeIcon, BellIcon, SearchIcon, BuildingIcon } from '@/components/ui/icons'
import HospitalSelector from '@/components/layout/HospitalSelector'


import { useHospital } from '@/lib/context/HospitalContext'
import { toast } from 'sonner'
import { useUser } from '@/lib/context/UserContext'

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card'
import { Progress } from '@/components/ui/progress'
import { Badge } from '@/components/ui/badge'
import { Separator } from '@/components/ui/separator'

// Icons
import {
  BadgeCheck,
  FileCheck,
  Calendar,
  UserCheck,
  AlertTriangle,
  BookOpen,
  CheckCircle2,
  Clock,
  ListChecks,
  ChevronRight,
  Globe,
  Info,
  AlertCircle,
} from 'lucide-react'

// Lazy load heavy components for better performance (only the ones we can verify work)
const LazyCompliance = lazy(() => import('@/components/dashboard/ComplianceStats').then(module => ({ default: module.ComplianceStats })))

// Loading fallback component
const ComponentLoader = React.memo(() => (
  <div className="flex items-center justify-center h-32">
    <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
  </div>
))
ComponentLoader.displayName = 'ComponentLoader'

// Enhanced BreadcrumbLink component with prefetching
const BreadcrumbLink = ({ 
  className, 
  children, 
  href,
  ...props 
}: React.AnchorHTMLAttributes<HTMLAnchorElement> & { href?: string }) => {
  if (href) {
    return (
      <Link 
        href={href}
        prefetch={true} // Enable prefetching for faster navigation
        className={`transition-colors hover:text-foreground flex items-center ${className || ''}`} 
        {...props}
      >
        {children}
      </Link>
    )
  }
  
  return (
    <a className={`transition-colors hover:text-foreground flex items-center ${className || ''}`} {...props}>
      {children}
    </a>
  )
}

// Header component with memoization
const Header = React.memo(() => {
  const { currentHospital } = useHospital();
  const { isSuperAdmin, currentUser } = useUser();
  
  // Memoized notifications to prevent re-creation
  const notifications = useMemo(() => [
    { id: 1, title: 'New audit finding requires response', time: '10m ago' },
    { id: 2, title: 'ISO 9001 compliance report due in 3 days', time: '1h ago' },
    { id: 3, title: 'Staff training compliance below target', time: '3h ago' },
  ], []);

  // Don't show anything if no hospital is selected
  if (!currentHospital) return null;

  return (
    <header className="bg-white border-b border-gray-200 py-4 px-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center">
          <h1 className="text-2xl font-bold text-gray-900">Global Health Guardian</h1>
          
          {/* Show hospital name for all users */}
          {currentHospital && (
            <div className="flex items-center ml-4">
              <div className="h-5 border-l border-gray-300 mx-3"></div>
              <div className="flex items-center text-primary font-medium">
                <BuildingIcon className="h-4 w-4 mr-2 text-primary" />
                <span>{currentHospital.name}</span>
              </div>
            </div>
          )}
        </div>
        
        <div className="flex items-center space-x-4">
          <div className="relative w-64">
            <Input 
              placeholder="Search compliance data..." 
              className="pl-10 pr-4 py-2 rounded-md border-gray-200 focus:border-blue-500 focus:ring focus:ring-blue-200 focus:ring-opacity-50"
            />
            <SearchIcon className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
          </div>
          
          {/* Only show the hospital selector for superadmins */}
          {isSuperAdmin && <HospitalSelector />}
          
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" size="icon" className="relative">
                <BellIcon className="h-5 w-5" />
                <span className="absolute top-0 right-0 h-2.5 w-2.5 bg-red-500 rounded-full border-2 border-white"></span>
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent className="w-80" align="end">
              <DropdownMenuLabel>Notifications</DropdownMenuLabel>
              <DropdownMenuSeparator />
              {notifications.map((notification) => (
                <DropdownMenuItem key={notification.id} className="p-0">
                  <div className="px-3 py-2 w-full cursor-pointer hover:bg-gray-50">
                    <div className="flex justify-between items-start">
                      <p className="font-medium text-sm text-gray-900">{notification.title}</p>
                      <span className="text-xs text-gray-500">{notification.time}</span>
                    </div>
                  </div>
                </DropdownMenuItem>
              ))}
              <DropdownMenuSeparator />
              <DropdownMenuItem className="cursor-pointer justify-center text-sm font-medium text-blue-600">
                View all notifications
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>
    </header>
  );
});

Header.displayName = 'Header';

// Simple breadcrumb components with memoization
const Breadcrumb = React.memo(({ className, ...props }: React.HTMLAttributes<HTMLElement>) => (
  <nav className={`flex flex-wrap items-center text-sm text-muted-foreground ${className || ''}`} {...props} />
))
Breadcrumb.displayName = 'Breadcrumb'

const BreadcrumbList = React.memo(({ className, ...props }: React.OlHTMLAttributes<HTMLOListElement>) => (
  <ol className={`flex flex-wrap items-center gap-1.5 ${className || ''}`} {...props} />
))
BreadcrumbList.displayName = 'BreadcrumbList'

const BreadcrumbItem = React.memo(({ className, ...props }: React.LiHTMLAttributes<HTMLLIElement>) => (
  <li className={`inline-flex items-center gap-1.5 ${className || ''}`} {...props} />
))
BreadcrumbItem.displayName = 'BreadcrumbItem'

const BreadcrumbPage = React.memo(({ className, ...props }: React.HTMLAttributes<HTMLSpanElement>) => (
  <span
    role="link"
    aria-disabled="true"
    aria-current="page"
    className={`text-foreground font-normal ${className || ''}`}
    {...props}
  />
))
BreadcrumbPage.displayName = 'BreadcrumbPage'

const BreadcrumbSeparator = React.memo(({ className, children, ...props }: React.HTMLAttributes<HTMLSpanElement>) => (
  <span
    role="presentation"
    className={className}
    {...props}
  >
    {children || '/'}
  </span>
))
BreadcrumbSeparator.displayName = 'BreadcrumbSeparator'

interface DashboardClientProps {
  children: ReactNode
}

export default function DashboardClient({ children }: DashboardClientProps) {
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false)
  const pathname = usePathname()
  
  // Performance monitoring
  React.useEffect(() => {
    const startTime = performance.now()
    
    // Cache warming removed to improve performance
    
    // Monitor component mount time
    const observer = new PerformanceObserver((list) => {
      list.getEntries().forEach((entry) => {
        if (entry.entryType === 'measure') {
          console.log(`🚀 Dashboard performance: ${entry.name} took ${entry.duration.toFixed(2)}ms`)
        }
      })
    })
    
    observer.observe({ entryTypes: ['measure'] })
    
    // Measure initial render time
    setTimeout(() => {
      const endTime = performance.now()
      performance.mark('dashboard-mounted')
      performance.measure('dashboard-mount-time', {
        start: startTime,
        end: endTime
      })
    }, 0)
    
    return () => observer.disconnect()
  }, [])
  
  // Memoized toggle function with performance tracking
  const toggleSidebar = useCallback(() => {
    performance.mark('sidebar-toggle-start')
    setSidebarCollapsed(!sidebarCollapsed)
    
    // Track sidebar toggle performance
    setTimeout(() => {
      performance.mark('sidebar-toggle-end')
      performance.measure('sidebar-toggle-time', 'sidebar-toggle-start', 'sidebar-toggle-end')
    }, 0)
  }, [sidebarCollapsed])
  
  // Memoized path segments to prevent re-computation
  const pathSegments = useMemo(() => 
    pathname.split('/').filter(Boolean), 
    [pathname]
  )
  
  // Memoized page title function with caching
  const getPageTitle = useCallback((path: string): string => {
    const titleCache = new Map([
      ['', 'Dashboard'],
      ['compliance', 'Compliance Management'],
      ['findings', 'Audit Findings'],
      ['reports', 'Compliance Reports'],
      ['documentation', 'Documentation'],
      ['regulations', 'Regulations'],
      ['schedule', 'Schedule'],
      ['training', 'Staff & Training'],
      ['hospitals', 'Hospital Management'],
      ['departments', 'Departments'],
      ['settings', 'Settings'],
      ['security', 'Security'],
      ['database', 'Database'],
      ['access', 'Access Control'],
      ['users', 'User Management'],
      ['admin', 'Administration']
    ])
    
    return titleCache.get(path) || path.charAt(0).toUpperCase() + path.slice(1)
  }, [])

  // Memoized breadcrumb items with performance optimization
  const breadcrumbItems = useMemo(() => {
    performance.mark('breadcrumb-calculation-start')
    
    const items = pathSegments.map((segment, index) => {
      const isLast = index === pathSegments.length - 1
      const href = `/${pathSegments.slice(0, index + 1).join('/')}`
      
      return {
        segment,
        isLast,
        href,
        title: getPageTitle(segment)
      }
    })
    
    performance.mark('breadcrumb-calculation-end')
    performance.measure('breadcrumb-calculation', 'breadcrumb-calculation-start', 'breadcrumb-calculation-end')
    
    return items
  }, [pathSegments, getPageTitle])

  // State for handling remediation actions with optimistic updates
  const [staffCompliance, setStaffCompliance] = useState(89)
  const [deadlinesCount, setDeadlinesCount] = useState(8)

  // Optimized remediation action handlers with performance tracking
  const handleStaffTrainingReminder = useCallback(() => {
    performance.mark('staff-training-reminder-start')
    toast.success('Staff compliance training reminders sent to all departments')
    setStaffCompliance(prev => Math.min(prev + 3, 100))
    performance.mark('staff-training-reminder-end')
    performance.measure('staff-training-reminder', 'staff-training-reminder-start', 'staff-training-reminder-end')
  }, [])

  const handleDeadlineExtension = useCallback(() => {
    performance.mark('deadline-extension-start')
    toast.success('Selected deadlines have been reviewed and prioritized')
    setDeadlinesCount(prev => Math.max(prev - 2, 0))
    performance.mark('deadline-extension-end')
    performance.measure('deadline-extension', 'deadline-extension-start', 'deadline-extension-end')
  }, [])

  const handleScheduleAudit = useCallback(() => {
    performance.mark('schedule-audit-start')
    toast.success('ISO 9001 compliance audit has been scheduled')
    performance.mark('schedule-audit-end')
    performance.measure('schedule-audit', 'schedule-audit-start', 'schedule-audit-end')
  }, [])

  return (
    <div className="flex h-screen overflow-hidden bg-gray-50">
      
      
      <Sidebar collapsed={sidebarCollapsed} onToggle={toggleSidebar} />
      
      <div className="flex-1 flex flex-col overflow-hidden">
        <Header />
        
        <div className="bg-white px-6 py-3 border-b border-gray-200">
          <Breadcrumb>
            <BreadcrumbList>
              <BreadcrumbItem>
                <BreadcrumbLink href="/dashboard">
                    <HomeIcon className="h-4 w-4 mr-1" />
                    Home
                  </BreadcrumbLink>
              </BreadcrumbItem>
              
              {breadcrumbItems.map((item, index) => {
                const isLast = item.isLast
                const href = item.href
                
                return (
                  <React.Fragment key={item.segment}>
                    <BreadcrumbSeparator />
                    <BreadcrumbItem>
                      {isLast ? (
                        <BreadcrumbPage>{item.title}</BreadcrumbPage>
                      ) : (
                        <BreadcrumbLink href={href}>{item.title}</BreadcrumbLink>
                      )}
                    </BreadcrumbItem>
                  </React.Fragment>
                )
              })}
            </BreadcrumbList>
          </Breadcrumb>
        </div>
        
        <main className="flex-1 overflow-y-auto p-6">
          <Suspense fallback={
            <div className="flex items-center justify-center h-full">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
            </div>
          }>
          {children}
          </Suspense>
        </main>
      </div>
    </div>
  )
}

// Components for dashboard
interface MetricsRowProps {
  staffCompliance: number;
  deadlinesCount: number;
  onStaffTrainingReminder: () => void;
  onDeadlineExtension: () => void;
  onScheduleAudit: () => void;
}

const MetricsRow: React.FC<MetricsRowProps> = ({
  staffCompliance,
  deadlinesCount,
  onStaffTrainingReminder,
  onDeadlineExtension,
  onScheduleAudit
}) => {
  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-5">
      <div className="lg:col-span-1">
        <ComplianceScore score={82} previousScore={78} />
      </div>
      
      <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
        <StatCard 
          title="Total Standards"
          value={217}
          icon={<BadgeCheck className="h-5 w-5 text-blue-600" />}
          trend={{
            value: 5,
            isPositive: true,
            label: "from last month"
          }}
          tooltip="Total number of regulatory standards across all frameworks that apply to your hospital operations"
        />
        
        <StatCard 
          title="Audits Completed"
          value={24}
          description="This Quarter"
          icon={<FileCheck className="h-5 w-5 text-green-600" />}
          trend={{
            value: 12,
            isPositive: true,
            label: "vs. last quarter"
          }}
          tooltip="Internal and external audits completed to verify compliance with regulatory standards"
          remediationAction={{
            label: "Schedule ISO 9001 Audit",
            action: onScheduleAudit
          }}
        />
        
        <StatCard 
          title="Upcoming Deadlines"
          value={deadlinesCount}
          icon={<Calendar className="h-5 w-5 text-amber-600" />}
          trend={{
            value: 2,
            isNeutral: true,
            label: "in next 14 days"
          }}
          tooltip="Critical compliance deadlines requiring immediate attention to maintain regulatory status"
          remediationAction={{
            label: "Review & Prioritize",
            action: onDeadlineExtension
          }}
        />
        
        <StatCard 
          title="Staff Compliance"
          value={`${staffCompliance}%`}
          icon={<UserCheck className="h-5 w-5 text-blue-600" />}
          trend={{
            value: 3,
            isPositive: false,
            label: "below target"
          }}
          tooltip="Percentage of staff who have completed all required compliance training modules"
          remediationAction={{
            label: "Send Training Reminders",
            action: onStaffTrainingReminder
          }}
        />
      </div>
    </div>
  );
};

interface ComplianceScoreProps {
  score: number;
  previousScore?: number;
}

const ComplianceScore: React.FC<ComplianceScoreProps> = ({ score, previousScore }) => {
  const getScoreColor = (value: number): string => {
    if (value >= 90) return 'text-green-500';
    if (value >= 70) return 'text-amber-500';
    return 'text-red-500';
  };
  
  const scoreTrend = previousScore ? score - previousScore : 0;
  
  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <CardTitle className="text-lg font-medium">Overall Compliance Score</CardTitle>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="flex items-center justify-center mb-4">
          <div className="w-32 h-32 flex items-center justify-center relative">
            <svg viewBox="0 0 100 100" className="w-full h-full">
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke="#E5E7EB"
                strokeWidth="10"
              />
              <circle
                cx="50"
                cy="50"
                r="45"
                fill="none"
                stroke={score >= 90 ? '#66BB6A' : score >= 70 ? '#FFC107' : '#F44336'}
                strokeWidth="10"
                strokeDasharray={`${2 * Math.PI * 45 * (score / 100)} ${2 * Math.PI * 45 * (1 - score / 100)}`}
                strokeDashoffset={2 * Math.PI * 45 * 0.25}
                transform="rotate(-90 50 50)"
              />
            </svg>
            <div className="absolute text-2xl font-bold">{score}%</div>
          </div>
        </div>
        
        <div className="text-center">
          <div className={`text-lg font-semibold mb-1 ${getScoreColor(score)}`}>
            {score >= 90 ? 'Excellent' : 
             score >= 80 ? 'Good' : 
             score >= 70 ? 'Average' : 
             score >= 60 ? 'Below Average' : 
             'Critical'}
          </div>
          
          {previousScore !== undefined && (
            <div className="flex items-center justify-center text-sm">
              <span className="text-gray-500 mr-1">Previous: {previousScore}%</span>
              <span className={
                scoreTrend > 0 ? "text-green-600 font-medium" : 
                scoreTrend < 0 ? "text-red-600 font-medium" : 
                "text-gray-500"
              }>
                ({scoreTrend > 0 ? '+' : ''}{scoreTrend}%)
              </span>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
};

interface TrendInfo {
  value: number;
  isPositive?: boolean;
  isNeutral?: boolean;
  label?: string;
}

interface RemediationAction {
  label: string;
  action: () => void;
}

interface StatCardProps {
  title: string;
  value: string | number;
  description?: string;
  icon?: React.ReactNode;
  trend?: TrendInfo;
  tooltip?: string;
  remediationAction?: RemediationAction;
}

const StatCard: React.FC<StatCardProps> = ({
  title,
  value,
  description,
  icon,
  trend,
  tooltip,
  remediationAction
}) => {
  return (
    <Card className="overflow-hidden h-full">
      <CardHeader className="flex flex-row items-center justify-between pb-2 space-y-0">
        <div className="flex items-center gap-2">
          <CardTitle className="text-sm font-medium text-gray-600">{title}</CardTitle>
          {tooltip && (
            <div className="group relative">
              <Info className="h-4 w-4 text-gray-400 hover:text-blue-500 cursor-help" />
              <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 -translate-y-2 hidden group-hover:block bg-white border border-gray-200 p-2 rounded shadow-lg z-10 w-48">
                <p className="text-xs text-gray-700">{tooltip}</p>
              </div>
            </div>
          )}
        </div>
        {icon && <div>{icon}</div>}
      </CardHeader>
      <CardContent className="pt-0">
        <div className="text-2xl font-bold">{value}</div>
        {description && (
          <p className="text-xs text-gray-500 mt-1">{description}</p>
        )}
        {trend && (
          <div className="flex items-center mt-2">
            <span className={
              trend.isPositive ? "text-xs font-medium text-green-600" : 
              trend.isNeutral ? "text-xs font-medium text-gray-500" : 
              "text-xs font-medium text-red-600"
            }>
              {trend.isPositive ? '+' : trend.isNeutral ? '' : '-'}{Math.abs(trend.value)}%
            </span>
            {trend.label && (
              <span className="text-xs text-gray-500 ml-1">
                {trend.label}
              </span>
            )}
          </div>
        )}
        
        {remediationAction && (
          <button 
            onClick={remediationAction.action} 
            className="mt-3 text-xs px-3 py-1.5 bg-blue-500 text-white rounded-md hover:bg-blue-600 transition-colors w-full"
          >
            {remediationAction.label}
          </button>
        )}
      </CardContent>
    </Card>
  );
};

const ComplianceCardsRow = () => {
  return (
    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-5">
      <ComplianceCard 
        title="HIPAA Privacy & Security"
        description="Patient data protection compliance"
        standard="HIPAA (USA)"
        status="compliant"
        dueDate="Annual Review: Sep 30, 2023"
      />
      
      <ComplianceCard 
        title="Quality Management System"
        description="Hospital management systems and processes"
        standard="ISO 9001:2015"
        status="partial"
        dueDate="Remediation due: Jun 15, 2023"
      />
      
      <ComplianceCard 
        title="Patient Safety Goals"
        description="International safety standards implementation"
        standard="JCI 7th Edition"
        status="non-compliant"
        dueDate="Action Required by May 31, 2023"
      />
    </div>
  );
};

interface ComplianceCardProps {
  title: string;
  description: string;
  standard: string;
  status: string;
  dueDate: string;
  className?: string;
}

const ComplianceCard: React.FC<ComplianceCardProps> = ({
  title,
  description,
  standard,
  status,
  dueDate,
  className
}) => {
  const getStatusIcon = () => {
    switch (status) {
      case 'compliant':
        return <CheckCircle2 className="h-5 w-5 text-green-600" />;
      case 'partial':
        return <AlertTriangle className="h-5 w-5 text-amber-600" />;
      case 'non-compliant':
        return <AlertCircle className="h-5 w-5 text-red-600" />;
      default:
        return null;
    }
  };
  
  const getStatusText = () => {
    switch (status) {
      case 'compliant':
        return 'Compliant';
      case 'partial':
        return 'Partially Compliant';
      case 'non-compliant':
        return 'Non-Compliant';
      default:
        return '';
    }
  };
  
  const getStatusBadgeClass = () => {
    switch (status) {
      case 'compliant':
        return 'bg-green-100 text-green-800';
      case 'partial':
        return 'bg-amber-100 text-amber-800';
      case 'non-compliant':
        return 'bg-red-100 text-red-800';
      default:
        return '';
    }
  };

  return (
    <Card className={`overflow-hidden h-full ${className || ''}`}>
      <div className="flex">
        <div className={
          status === 'compliant' ? 'bg-green-500 w-1 h-full' : 
          status === 'partial' ? 'bg-amber-500 w-1 h-full' : 
          'bg-red-500 w-1 h-full'
        }></div>
        <div className="flex-1">
          <CardHeader className="pb-2">
            <div className="flex items-start justify-between">
              <div>
                <CardTitle className="text-base font-medium">{title}</CardTitle>
                <CardDescription className="text-xs mt-1">{standard}</CardDescription>
              </div>
              {getStatusIcon()}
            </div>
          </CardHeader>
          <CardContent className="pt-0">
            {description && (
              <p className="text-sm text-gray-600 mb-3">{description}</p>
            )}
            <div className="flex items-center justify-between">
              <span className={`text-xs px-2 py-1 rounded ${getStatusBadgeClass()}`}>{getStatusText()}</span>
              {dueDate && (
                <span className="text-xs text-gray-500">Due: {dueDate}</span>
              )}
            </div>
          </CardContent>
        </div>
      </div>
    </Card>
  );
};

const ComplianceChartAndFindings = () => {
  const complianceData = [
    { name: 'ISO 9001', compliance: 87, required: 90 },
    { name: 'HIPAA', compliance: 94, required: 90 },
    { name: 'JCI', compliance: 78, required: 90 },
    { name: 'GDPR', compliance: 82, required: 90 },
    { name: 'FDA 21 CFR', compliance: 76, required: 90 },
    { name: 'OSHA', compliance: 88, required: 90 },
    { name: 'ISO 27001', compliance: 81, required: 90 },
    { name: 'ISO 14001', compliance: 79, required: 85 }
  ];
  
  const auditFindings = [
    {
      id: 'af1',
      title: 'Missing documentation for equipment maintenance',
      department: 'Facilities',
      severity: 'high',
      date: 'May 10, 2023',
      status: 'open',
    },
    {
      id: 'af2',
      title: 'Staff training records incomplete',
      department: 'Human Resources',
      severity: 'medium',
      date: 'May 8, 2023',
      status: 'in-progress',
    },
    {
      id: 'af3',
      title: 'Patient data access logs not maintained',
      department: 'IT Security',
      severity: 'critical',
      date: 'May 5, 2023',
      status: 'open',
    },
    {
      id: 'af4',
      title: 'Medication storage temperature log inconsistent',
      department: 'Pharmacy',
      severity: 'medium',
      date: 'May 3, 2023',
      status: 'resolved',
    }
  ];

  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-5">
      <div className="lg:col-span-2">
        <ComplianceChart data={complianceData} />
      </div>
      
      <div className="lg:col-span-1">
        <AuditFindings findings={auditFindings} />
      </div>
    </div>
  );
};

interface ChartDataPoint {
  name: string;
  compliance: number;
  required: number;
}

interface ComplianceChartProps {
  data: ChartDataPoint[];
}

const ComplianceChart: React.FC<ComplianceChartProps> = ({ data }) => {
  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle>Compliance Metrics</CardTitle>
        <CardDescription>Current compliance level vs. required threshold</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {data.map((item, index) => (
            <div key={index} className="space-y-2">
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <span className={`h-2 w-2 rounded-full mr-2 ${
                    item.compliance >= item.required ? 'bg-green-500' : 'bg-red-500'
                  }`} />
                  <span className="text-sm font-medium">{item.name}</span>
                </div>
                <div className="flex items-center">
                  <span className="text-sm font-medium mr-2">{item.compliance}%</span>
                  <span className="text-xs text-gray-500">/ {item.required}%</span>
                </div>
              </div>
              <div className="h-2 bg-gray-200 rounded-full">
                <div 
                  className={`h-full rounded-full ${
                    item.compliance >= item.required ? 'bg-green-500' : 'bg-red-500'
                  }`} 
                  style={{ width: `${item.compliance}%` }}
                />
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};

interface Finding {
  id: string;
  title: string;
  department: string;
  severity: string;
  status: string;
  date: string;
}

interface AuditFindingsProps {
  findings: Finding[];
}

const AuditFindings: React.FC<AuditFindingsProps> = ({ findings }) => {
  const getSeverityBadge = (severity: string) => {
    switch (severity) {
      case 'critical':
        return <Badge className="bg-red-700">Critical</Badge>;
      case 'high':
        return <Badge className="bg-red-500">High</Badge>;
      case 'medium':
        return <Badge className="bg-amber-500">Medium</Badge>;
      case 'low':
        return <Badge variant="outline" className="text-blue-600">Low</Badge>;
      default:
        return null;
    }
  };
  
  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'open':
        return <Badge variant="outline" className="text-gray-600 border-gray-300">Open</Badge>;
      case 'in-progress':
        return <Badge variant="outline" className="text-blue-600 border-blue-300">In Progress</Badge>;
      case 'resolved':
        return <Badge variant="outline" className="text-green-600 border-green-300">Resolved</Badge>;
      default:
        return null;
    }
  };

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle>Recent Audit Findings</CardTitle>
        <CardDescription>Last updated: Today at 09:34 AM</CardDescription>
      </CardHeader>
      <CardContent className="overflow-hidden">
        <div className="space-y-4">
          {findings.map((finding, index) => (
            <div key={finding.id}>
              {index > 0 && <Separator className="my-4" />}
              <div>
                <div className="flex items-center justify-between mb-1">
                  <h4 className="font-medium text-sm">{finding.title}</h4>
                  {getSeverityBadge(finding.severity)}
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-500">{finding.department} • {finding.date}</span>
                  {getStatusBadge(finding.status)}
                </div>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};

const ActionItemsAndFeed = () => {
  const actionItems = [
    {
      id: 'ai1',
      title: 'Update HIPAA compliance documentation',
      dueDate: 'Jun 15, 2023',
      priority: 'high',
      completed: false,
      category: 'Documentation',
    },
    {
      id: 'ai2',
      title: 'Complete staff GDPR training program',
      dueDate: 'Jun 20, 2023',
      priority: 'medium',
      completed: false,
      category: 'Training',
    },
    {
      id: 'ai3',
      title: 'Implement access control recommendations',
      dueDate: 'Jun 5, 2023',
      priority: 'low',
      completed: true,
      category: 'Security',
    },
    {
      id: 'ai4',
      title: 'Review patient consent procedures',
      dueDate: 'Jun 12, 2023',
      priority: 'medium',
      completed: false,
      category: 'Process',
    }
  ];
  
  const regulatoryNews = [
    {
      id: 'rn1',
      title: 'JCI announces 7th Edition Standard updates',
      source: 'Joint Commission International',
      date: 'May 12, 2023',
      category: 'Standards Update',
      link: '#',
      isNew: true,
    },
    {
      id: 'rn2',
      title: 'FDA releases new guidance on medical device reporting',
      source: 'U.S. FDA',
      date: 'May 10, 2023',
      category: 'Regulatory Guidance',
      link: '#',
    },
    {
      id: 'rn3',
      title: 'GDPR enforcement actions increasing in healthcare sector',
      source: 'EU Commission',
      date: 'May 7, 2023',
      category: 'Enforcement',
      link: '#',
    }
  ];

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-5">
      <ActionItems items={actionItems} />
      <RegulatoryFeed news={regulatoryNews} />
    </div>
  );
};

interface ActionItem {
  id: string;
  title: string;
  dueDate: string;
  priority: string;
  completed: boolean;
  category: string;
  description?: string;
  assignedTo?: string;
}

interface ActionItemsProps {
  items: ActionItem[];
}

const ActionItems: React.FC<ActionItemsProps> = ({ items }) => {
  const priorityClasses: Record<string, string> = {
    high: "text-red-600",
    medium: "text-amber-600",
    low: "text-blue-600",
  };

  return (
    <Card className="h-full">
      <CardHeader className="flex flex-row items-center justify-between pb-2">
        <div className="flex items-center space-x-2">
          <ListChecks className="h-5 w-5 text-blue-600" />
          <CardTitle>Compliance Action Items</CardTitle>
        </div>
        <Button variant="outline" size="sm">View All</Button>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="space-y-4">
          {items.map((item, index) => (
            <div key={item.id}>
              {index > 0 && <Separator className="my-4" />}
              <div className="flex items-start gap-3">
                <div className="mt-0.5">
                  {item.completed ? (
                    <CheckCircle2 className="h-5 w-5 text-green-600" />
                  ) : (
                    <div className={`h-5 w-5 rounded-full border-2 flex items-center justify-center ${priorityClasses[item.priority]}`}>
                      <div className={`h-2 w-2 rounded-full ${priorityClasses[item.priority]}`}></div>
                    </div>
                  )}
                </div>
                <div className="flex-1">
                  <h4 className={`font-medium text-sm ${item.completed && "line-through text-gray-500"}`}>
                    {item.title}
                  </h4>
                  <div className="flex items-center mt-1 text-xs text-gray-500">
                    <Clock className="h-3 w-3 mr-1" />
                    <span>Due: {item.dueDate}</span>
                    <span className="mx-1.5">•</span>
                    <span>{item.category}</span>
                  </div>
                </div>
                <Button 
                  variant={item.completed ? "ghost" : "outline"} 
                  size="sm" 
                  className="text-xs h-7 px-2"
                >
                  {item.completed ? "Completed" : "Mark Done"}
                </Button>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};

interface NewsItem {
  id: string;
  title: string;
  source: string;
  date: string;
  category: string;
  link: string;
  isNew?: boolean;
}

interface RegulatoryFeedProps {
  news: NewsItem[];
}

const RegulatoryFeed: React.FC<RegulatoryFeedProps> = ({ news }) => {
  return (
    <Card className="h-full">
      <CardHeader className="pb-2">
        <div className="flex items-center space-x-2">
          <Globe className="h-5 w-5 text-blue-600" />
          <CardTitle>Global Regulatory Updates</CardTitle>
        </div>
        <CardDescription>Latest standards and regulatory changes</CardDescription>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="space-y-4">
          {news.map((item, index) => (
            <div key={item.id}>
              {index > 0 && <Separator className="my-4" />}
              <div>
                <div className="flex items-start justify-between">
                  <h4 className="font-medium text-sm">{item.title}</h4>
                  {item.isNew && (
                    <Badge className="bg-blue-100 text-blue-800 hover:bg-blue-100">New</Badge>
                  )}
                </div>
                <div className="flex items-center mt-1">
                  <span className="text-xs text-gray-500">{item.source}</span>
                  <span className="mx-1.5 text-gray-300">•</span>
                  <span className="text-xs text-gray-500">{item.date}</span>
                </div>
                <div className="mt-2 flex items-center justify-between">
                  <span className="text-xs px-2 py-0.5 bg-blue-50 text-blue-600 rounded-full">
                    {item.category}
                  </span>
                  <a 
                    href={item.link} 
                    className="text-xs text-blue-600 hover:underline flex items-center"
                    target="_blank" 
                    rel="noreferrer"
                  >
                    <BookOpen className="h-3 w-3 mr-1" />
                    Read more
                  </a>
                </div>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}; 