'use client'

import React, { useState } from "react"
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardHeader, 
  CardTitle,
  CardFooter
} from "@/components/ui/card"
import { Button } from "@/components/ui/button"

// Simple Tabs components
interface TabsProps {
  defaultValue: string;
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

type TabChildProps = {
  activeTab?: string;
  setActiveTab?: (value: string) => void;
};

const Tabs: React.FC<TabsProps> = ({ defaultValue, children, className, ...props }) => {
  const [activeTab, setActiveTab] = useState(defaultValue)
  
  // Clone children and add the active tab state
  const enhancedChildren = React.Children.map(children, child => {
    if (React.isValidElement(child)) {
      return React.cloneElement(child as React.ReactElement<TabChildProps>, {
        activeTab,
        setActiveTab,
      })
    }
    return child
  })
  
  return (
    <div className={className} {...props}>
      {enhancedChildren}
    </div>
  )
}

interface TabsListProps {
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

const TabsList: React.FC<TabsListProps> = ({ children, className, ...props }) => {
  return (
    <div className={`inline-flex h-10 items-center justify-center rounded-md bg-muted p-1 text-muted-foreground ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

interface TabsTriggerProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  setActiveTab?: (value: string) => void;
  [key: string]: unknown;
}

const TabsTrigger: React.FC<TabsTriggerProps> = ({ value, children, className, activeTab, setActiveTab, ...props }) => {
  const isActive = activeTab === value
  
  return (
    <button
      className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium transition-all
        ${isActive ? 'bg-background text-foreground shadow-sm' : 'hover:text-foreground'} 
        ${className || ''}`}
      onClick={() => setActiveTab?.(value)}
      {...props}
    >
      {children}
    </button>
  )
}

interface TabsContentProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  [key: string]: unknown;
}

const TabsContent: React.FC<TabsContentProps> = ({ value, children, className, activeTab, ...props }) => {
  if (activeTab !== value) return null
  
  return (
    <div className={`mt-2 ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

export default function SettingsPage() {
  const [activeTab, setActiveTab] = useState("database")
  
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">System Settings</h1>
          <p className="text-muted-foreground">Configure system settings and preferences</p>
        </div>
      </div>

      <Tabs defaultValue="database" className="w-full">
        <TabsList>
          <TabsTrigger value="database" activeTab={activeTab} setActiveTab={setActiveTab}>
            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="mr-2 h-4 w-4">
              <ellipse cx="12" cy="5" rx="9" ry="3"/>
              <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3"/>
              <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"/>
            </svg>
            Database
          </TabsTrigger>
          <TabsTrigger value="api" activeTab={activeTab} setActiveTab={setActiveTab}>
            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="mr-2 h-4 w-4">
              <circle cx="12" cy="12" r="10"/>
              <path d="M12 2a14.5 14.5 0 0 0 0 20 14.5 14.5 0 0 0 0-20"/>
              <path d="M2 12h20"/>
            </svg>
            API
          </TabsTrigger>
          <TabsTrigger value="security" activeTab={activeTab} setActiveTab={setActiveTab}>
            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="mr-2 h-4 w-4">
              <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
              <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
            </svg>
            Security
          </TabsTrigger>
          <TabsTrigger value="maintenance" activeTab={activeTab} setActiveTab={setActiveTab}>
            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="mr-2 h-4 w-4">
              <rect width="20" height="16" x="2" y="4" rx="2"/>
              <path d="M20 8H4"/>
              <path d="M8 16h.01"/>
              <path d="M12 16h.01"/>
              <path d="M16 16h.01"/>
            </svg>
            Maintenance
          </TabsTrigger>
          <TabsTrigger value="demo" activeTab={activeTab} setActiveTab={setActiveTab}>
            <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="mr-2 h-4 w-4">
              <rect x="4" y="4" width="16" height="16" rx="2"/>
              <circle cx="12" cy="12" r="3"/>
              <path d="M12 7v2"/>
              <path d="M12 15v2"/>
              <path d="M7 12h2"/>
              <path d="M15 12h2"/>
            </svg>
            Demo Data
          </TabsTrigger>
        </TabsList>
        
        <TabsContent value="database" className="space-y-4 mt-4" activeTab={activeTab}>
          <Card>
            <CardHeader>
              <CardTitle>Database Connection</CardTitle>
              <CardDescription>
                Configure connection to the PHP/MySQL backend
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="db-host">Host</label>
                  <input 
                    id="db-host" 
                    className="w-full px-3 py-2 border rounded-md"
                    defaultValue="localhost"
                  />
                </div>
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="db-port">Port</label>
                  <input 
                    id="db-port" 
                    className="w-full px-3 py-2 border rounded-md"
                    defaultValue="3306"
                  />
                </div>
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium" htmlFor="db-name">Database Name</label>
                <input 
                  id="db-name" 
                  className="w-full px-3 py-2 border rounded-md"
                  defaultValue="health_guardian"
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="db-username">Username</label>
                  <input 
                    id="db-username" 
                    className="w-full px-3 py-2 border rounded-md"
                    placeholder="Enter username"
                  />
                </div>
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="db-password">Password</label>
                  <input 
                    id="db-password" 
                    type="password" 
                    className="w-full px-3 py-2 border rounded-md"
                    placeholder="Enter password"
                  />
                </div>
              </div>
            </CardContent>
            <CardFooter>
              <Button className="ml-auto">Save Database Settings</Button>
            </CardFooter>
          </Card>
        </TabsContent>
        
        <TabsContent value="api" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>API Configuration</CardTitle>
              <CardDescription>Configure API endpoints for communication with PHP backend</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <label className="text-sm font-medium" htmlFor="api-base-url">Base URL</label>
                <input 
                  id="api-base-url" 
                  className="w-full px-3 py-2 border rounded-md"
                  defaultValue="https://api.example.com"
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="api-timeout">Request Timeout (seconds)</label>
                  <input 
                    id="api-timeout" 
                    type="number" 
                    className="w-full px-3 py-2 border rounded-md"
                    defaultValue="30"
                  />
                </div>
                <div className="space-y-2">
                  <label className="text-sm font-medium" htmlFor="api-retry">Retry Attempts</label>
                  <input 
                    id="api-retry" 
                    type="number" 
                    className="w-full px-3 py-2 border rounded-md"
                    defaultValue="3"
                  />
                </div>
              </div>
              
              <div className="pt-2">
                <Button>Save API Settings</Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="security" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Security Settings</CardTitle>
              <CardDescription>Configure system security options</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <label className="text-sm font-medium" htmlFor="session-timeout">Session Timeout (minutes)</label>
                <input 
                  id="session-timeout" 
                  type="number" 
                  className="w-full px-3 py-2 border rounded-md"
                  defaultValue="30"
                />
              </div>
              
              <div className="flex items-center justify-between space-x-2 py-3 border-t">
                <div>
                  <div className="text-sm font-medium">Enforce Password Strength</div>
                  <p className="text-xs text-gray-500">Require strong passwords with numbers, symbols and uppercase</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input type="checkbox" className="sr-only peer" defaultChecked />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full rtl:peer-checked:after:-translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:start-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
              
              <div className="flex items-center justify-between space-x-2 py-3 border-t">
                <div>
                  <div className="text-sm font-medium">Enforce Two-Factor Authentication</div>
                  <p className="text-xs text-gray-500">Require 2FA for admin accounts</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input type="checkbox" className="sr-only peer" />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full rtl:peer-checked:after:-translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:start-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
              
              <div className="pt-2">
                <Button>Save Security Settings</Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="maintenance" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>System Maintenance</CardTitle>
              <CardDescription>Perform system maintenance tasks</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="border rounded-md p-4">
                  <h3 className="text-lg font-medium mb-2">System Cache</h3>
                  <p className="text-sm text-gray-500 mb-4">Clear the system cache to free up memory and resolve potential conflicts</p>
                  <Button variant="outline">Clear System Cache</Button>
                </div>
                
                <div className="border rounded-md p-4">
                  <h3 className="text-lg font-medium mb-2">System Diagnostics</h3>
                  <p className="text-sm text-gray-500 mb-4">Run system diagnostics to check for issues and optimize performance</p>
                  <Button variant="outline">Run Diagnostics</Button>
                </div>
              </div>
              
              <div className="border rounded-md p-4 mt-4">
                <h3 className="text-lg font-medium mb-2">Database Maintenance</h3>
                <p className="text-sm text-gray-500 mb-4">Optimize database performance and integrity</p>
                <div className="flex flex-wrap gap-2">
                  <Button variant="outline">Optimize Tables</Button>
                  <Button variant="outline">Check Integrity</Button>
                  <Button variant="outline">Repair Tables</Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="demo" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Demo Data Management</CardTitle>
              <CardDescription>Manage demo data for testing and demonstrations</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex items-center justify-between space-x-2 py-3 border-b">
                <div>
                  <div className="text-lg font-medium">Enable Demo Data</div>
                  <p className="text-sm text-gray-500">Load sample data for demonstration purposes</p>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input type="checkbox" className="sr-only peer" defaultChecked />
                  <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full rtl:peer-checked:after:-translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:start-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                </label>
              </div>
              
              <div className="border rounded-md p-4">
                <h3 className="text-lg font-medium mb-2">Available Demo Datasets</h3>
                <div className="space-y-2">
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="hospital-data" className="rounded" defaultChecked />
                    <label htmlFor="hospital-data">Hospital Sample Data</label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="compliance-data" className="rounded" defaultChecked />
                    <label htmlFor="compliance-data">Compliance Sample Data</label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="user-data" className="rounded" defaultChecked />
                    <label htmlFor="user-data">User Sample Data</label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <input type="checkbox" id="audit-data" className="rounded" defaultChecked />
                    <label htmlFor="audit-data">Audit Sample Data</label>
                  </div>
                </div>
                <div className="mt-4">
                  <Button>Apply Demo Data Settings</Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
      
      <div className="flex justify-end gap-2">
        <Button variant="outline">Cancel</Button>
        <Button>Save Changes</Button>
      </div>
    </div>
  )
} 