'use client'

import React, { useState } from "react"
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardHeader, 
  CardTitle 
} from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"

// Simple Tabs components
interface TabsProps {
  defaultValue: string;
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

type TabChildProps = {
  activeTab?: string;
  setActiveTab?: (value: string) => void;
};

const Tabs: React.FC<TabsProps> = ({ defaultValue, children, className, ...props }) => {
  const [activeTab, setActiveTab] = useState(defaultValue)
  
  // Clone children and add the active tab state
  const enhancedChildren = React.Children.map(children, child => {
    if (React.isValidElement(child)) {
      return React.cloneElement(child as React.ReactElement<TabChildProps>, {
        activeTab,
        setActiveTab,
      })
    }
    return child
  })
  
  return (
    <div className={className} {...props}>
      {enhancedChildren}
    </div>
  )
}

interface TabsListProps {
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

const TabsList: React.FC<TabsListProps> = ({ children, className, ...props }) => {
  return (
    <div className={`inline-flex h-10 items-center justify-center rounded-md bg-muted p-1 text-muted-foreground ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

interface TabsTriggerProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  setActiveTab?: (value: string) => void;
  [key: string]: unknown;
}

const TabsTrigger: React.FC<TabsTriggerProps> = ({ value, children, className, activeTab, setActiveTab, ...props }) => {
  const isActive = activeTab === value
  
  return (
    <button
      className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium transition-all
        ${isActive ? 'bg-background text-foreground shadow-sm' : 'hover:text-foreground'} 
        ${className || ''}`}
      onClick={() => setActiveTab?.(value)}
      {...props}
    >
      {children}
    </button>
  )
}

interface TabsContentProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  [key: string]: unknown;
}

const TabsContent: React.FC<TabsContentProps> = ({ value, children, className, activeTab, ...props }) => {
  if (activeTab !== value) return null
  
  return (
    <div className={`mt-2 ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

// Icons
const LockIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-5 w-5 text-blue-600 ${className}`}>
    <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
    <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
  </svg>
)

const ActivityIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-5 w-5 text-amber-600 ${className}`}>
    <path d="M22 12h-4l-3 9L9 3l-3 9H2"/>
  </svg>
)

const SearchIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-4 w-4 text-gray-500 ${className}`}>
    <circle cx="11" cy="11" r="8"/>
    <path d="m21 21-4.3-4.3"/>
  </svg>
)

interface ActivityLogItemProps {
  action: string;
  user: string;
  timestamp: string;
  ip: string;
  status: string;
  details?: string;
}

const ActivityLogItem: React.FC<ActivityLogItemProps> = ({ action, user, timestamp, ip, status, details }) => {
  return (
    <div className="py-3 border-b last:border-0">
      <div className="flex items-center justify-between">
        <div>
          <div className="font-medium text-sm">{action}</div>
          <div className="text-xs text-gray-500 mt-1">
            By {user} • {timestamp} • IP: {ip}
          </div>
        </div>
        <Badge className={
          status === 'success' 
            ? 'bg-green-100 text-green-800' 
            : status === 'warning'
              ? 'bg-amber-100 text-amber-800'
              : 'bg-red-100 text-red-800'
        }>
          {status.charAt(0).toUpperCase() + status.slice(1)}
        </Badge>
      </div>
      {details && (
        <div className="mt-2 text-xs text-gray-600 bg-gray-50 p-2 rounded">
          {details}
        </div>
      )}
    </div>
  )
}

export default function SecurityPage() {
  const [activeTab, setActiveTab] = useState("vulnerabilities")
  
  // Sample data
  const auditLogs = [
    {
      id: "log1",
      action: "User Login",
      user: "Dr. Sarah Williams",
      timestamp: "Jun 15, 2023 14:30:25",
      ip: "192.168.1.105",
      status: "success",
      details: "Login from Chrome/Windows"
    },
    {
      id: "log2",
      action: "Permission Changed",
      user: "Robert Davis (Administrator)",
      timestamp: "Jun 15, 2023 11:24:36",
      ip: "192.168.1.201",
      status: "success",
      details: "Changed John Smith role from Viewer to Auditor"
    },
    {
      id: "log3",
      action: "Failed Login Attempt",
      user: "Unknown",
      timestamp: "Jun 14, 2023 23:15:02",
      ip: "216.58.210.46",
      status: "error",
      details: "3 failed attempts with username: admin"
    },
    {
      id: "log4",
      action: "Document Accessed",
      user: "John Smith",
      timestamp: "Jun 14, 2023 15:42:18",
      ip: "192.168.1.105",
      status: "success",
      details: "Accessed document: HIPAA Policy.pdf"
    },
    {
      id: "log5",
      action: "User Account Created",
      user: "Robert Davis (Administrator)",
      timestamp: "Jun 13, 2023 09:11:45",
      ip: "192.168.1.201",
      status: "success",
      details: "Created account for Emily Johnson"
    },
    {
      id: "log6",
      action: "Suspicious Activity",
      user: "Michael Chen",
      timestamp: "Jun 12, 2023 19:26:03",
      ip: "203.0.113.45",
      status: "warning",
      details: "Multiple data export attempts"
    }
  ]
  
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Security Management</h1>
          <p className="text-muted-foreground">Monitor and manage system security</p>
        </div>
        <Badge className="bg-blue-100 text-blue-800 text-xs">
          New Feature
        </Badge>
      </div>

      <Tabs defaultValue="vulnerabilities" className="w-full">
        <TabsList>
          <TabsTrigger value="vulnerabilities" activeTab={activeTab} setActiveTab={setActiveTab}>Vulnerabilities</TabsTrigger>
          <TabsTrigger value="access" activeTab={activeTab} setActiveTab={setActiveTab}>Access Logs</TabsTrigger>
          <TabsTrigger value="compliance" activeTab={activeTab} setActiveTab={setActiveTab}>Compliance</TabsTrigger>
          <TabsTrigger value="settings" activeTab={activeTab} setActiveTab={setActiveTab}>Settings</TabsTrigger>
        </TabsList>
        
        <TabsContent value="vulnerabilities" className="space-y-4 mt-4" activeTab={activeTab}>
          <Card>
            <CardHeader>
              <CardTitle>Security Vulnerabilities</CardTitle>
              <CardDescription>Critical and high priority security issues</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="bg-red-50 border-l-4 border-red-500 p-4 rounded-md flex justify-between">
                  <div>
                    <div className="flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-red-500 mr-2">
                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                        <line x1="12" y1="9" x2="12" y2="13"/>
                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                      </svg>
                      <h4 className="font-medium text-red-800">Outdated Password Policy</h4>
                      <Badge variant="destructive" className="ml-2">Critical</Badge>
                    </div>
                    <p className="mt-1 text-sm">Password policy does not enforce complexity requirements.</p>
                  </div>
                  <Button variant="outline" size="sm">Details</Button>
                </div>
                
                <div className="bg-amber-50 border-l-4 border-amber-500 p-4 rounded-md flex justify-between">
                  <div>
                    <div className="flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-amber-500 mr-2">
                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                        <line x1="12" y1="9" x2="12" y2="13"/>
                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                      </svg>
                      <h4 className="font-medium text-amber-800">Missing Two-Factor Authentication</h4>
                      <Badge variant="outline" className="ml-2 bg-amber-100 text-amber-800 border-amber-300">High</Badge>
                    </div>
                    <p className="mt-1 text-sm">Admin accounts are not using two-factor authentication.</p>
                  </div>
                  <Button variant="outline" size="sm">Details</Button>
                </div>
                
                <div className="bg-green-50 border-l-4 border-green-500 p-4 rounded-md flex justify-between">
                  <div>
                    <div className="flex items-center">
                      <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-green-500 mr-2">
                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                        <polyline points="22 4 12 14.01 9 11.01"/>
                      </svg>
                      <h4 className="font-medium text-green-800">Database Encryption</h4>
                      <Badge variant="outline" className="ml-2 bg-green-100 text-green-800 border-green-300">Secure</Badge>
                    </div>
                    <p className="mt-1 text-sm">All sensitive data is encrypted both at rest and in transit.</p>
                  </div>
                  <Button variant="outline" size="sm">Details</Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="access" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Access Logs</CardTitle>
                  <CardDescription>Recent system access events</CardDescription>
                </div>
                <div className="flex gap-2">
                  <Button variant="outline" size="sm">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-1">
                      <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                      <polyline points="14 2 14 8 20 8"/>
                    </svg>
                    Export
                  </Button>
                  <Button variant="outline" size="sm">
                    <SearchIcon className="mr-1" />
                    Search
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="border-b pb-2">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="font-medium">Admin login from new location</p>
                      <p className="text-sm text-gray-500">User: admin@example.com</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm">April 28, 2025</p>
                      <p className="text-xs text-gray-500">09:12 AM</p>
                    </div>
                  </div>
                </div>
                
                <div className="border-b pb-2">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="font-medium">Failed login attempts</p>
                      <p className="text-sm text-gray-500">User: john.smith@hospital.org</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm">April 27, 2025</p>
                      <p className="text-xs text-gray-500">11:45 PM</p>
                    </div>
                  </div>
                </div>
                
                <div className="border-b pb-2">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="font-medium">User permission modified</p>
                      <p className="text-sm text-gray-500">Modified by: admin@example.com</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm">April 27, 2025</p>
                      <p className="text-xs text-gray-500">04:30 PM</p>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="compliance" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Security Compliance Status</CardTitle>
              <CardDescription>Security standards and regulatory compliance</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-md">
                  <div className="flex items-center">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-green-600 mr-3">
                      <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                    </svg>
                    <div>
                      <p className="font-medium">HIPAA Compliance</p>
                      <p className="text-sm text-gray-500">Health Insurance Portability and Accountability Act</p>
                    </div>
                  </div>
                  <Badge className="bg-green-100 text-green-800 border-green-300">Compliant</Badge>
                </div>
                
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-md">
                  <div className="flex items-center">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-amber-600 mr-3">
                      <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
                      <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                    </svg>
                    <div>
                      <p className="font-medium">GDPR Compliance</p>
                      <p className="text-sm text-gray-500">General Data Protection Regulation</p>
                    </div>
                  </div>
                  <Badge className="bg-amber-100 text-amber-800 border-amber-300">Partial</Badge>
                </div>
                
                <div className="flex items-center justify-between p-3 bg-gray-50 rounded-md">
                  <div className="flex items-center">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-5 w-5 text-green-600 mr-3">
                      <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                      <circle cx="9" cy="7" r="4"/>
                      <path d="M22 21v-2a4 4 0 0 0-3-3.87"/>
                      <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                    </svg>
                    <div>
                      <p className="font-medium">SOC 2 Compliance</p>
                      <p className="text-sm text-gray-500">System and Organization Controls</p>
                    </div>
                  </div>
                  <Badge className="bg-green-100 text-green-800 border-green-300">Compliant</Badge>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="settings" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Security Settings</CardTitle>
              <CardDescription>Configure system-wide security parameters</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-6">
                <div>
                  <h3 className="text-lg font-medium mb-2">Password Policy</h3>
                  <div className="grid gap-4 sm:grid-cols-2">
                    <div className="flex justify-between items-center p-3 border rounded-md">
                      <span>Minimum length</span>
                      <span className="font-medium">8 characters</span>
                    </div>
                    <div className="flex justify-between items-center p-3 border rounded-md">
                      <span>Require special characters</span>
                      <span className="font-medium">Yes</span>
                    </div>
                    <div className="flex justify-between items-center p-3 border rounded-md">
                      <span>Require numbers</span>
                      <span className="font-medium">Yes</span>
                    </div>
                    <div className="flex justify-between items-center p-3 border rounded-md">
                      <span>Password expiration</span>
                      <span className="font-medium">90 days</span>
                    </div>
                  </div>
                  <Button className="mt-4" size="sm">
                    Edit Password Policy
                  </Button>
                </div>
                
                <div className="pt-4 border-t">
                  <h3 className="text-lg font-medium mb-2">Two-Factor Authentication</h3>
                  <div className="flex items-center justify-between p-3 border rounded-md mb-2">
                    <span>Require for admin accounts</span>
                    <Badge>Enabled</Badge>
                  </div>
                  <Button size="sm">
                    Configure 2FA Settings
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
} 