'use client'

import React, { useState, useEffect } from 'react'
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardHeader, 
  CardTitle 
} from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { toast } from 'sonner'
import { useApiWithHospital } from '@/hooks/useApiWithHospital'

// Icons
const Plus = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="M12 5v14M5 12h14" />
  </svg>
)

const Edit = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="M17 3a2.85 2.83 0 1 1 4 4L7.5 20.5 2 22l1.5-5.5Z"/>
    <path d="m15 5 4 4"/>
  </svg>
)

const Trash = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="M3 6h18"/>
    <path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"/>
    <path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"/>
  </svg>
)

const ChevronDown = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="m6 9 6 6 6-6"/>
  </svg>
)

const ChevronUp = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="m18 15-6-6-6 6"/>
  </svg>
)

const Building = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <rect width="16" height="20" x="4" y="2" rx="2" ry="2" />
    <path d="M9 22v-4h6v4" />
    <path d="M8 6h.01" />
    <path d="M16 6h.01" />
    <path d="M12 6h.01" />
    <path d="M12 10h.01" />
    <path d="M12 14h.01" />
    <path d="M16 10h.01" />
    <path d="M16 14h.01" />
    <path d="M8 10h.01" />
    <path d="M8 14h.01" />
  </svg>
)

const User = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="M19 21v-2a4 4 0 0 0-4-4H9a4 4 0 0 0-4 4v2" />
    <circle cx="12" cy="7" r="4" />
  </svg>
)

const Mail = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <rect width="20" height="16" x="2" y="4" rx="2" />
    <path d="m22 7-8.97 5.7a1.94 1.94 0 0 1-2.06 0L2 7" />
  </svg>
)

const Key = () => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
    <path d="M21 2l-2 2m-7.61 7.61a5.5 5.5 0 1 1-7.778 7.778 5.5 5.5 0 0 1 7.777-7.777zm0 0L15.5 7.5m0 0 3 3L22 7l-3-3m-3.5 3.5L19 4" />
  </svg>
)

// Interface definitions
interface Hospital {
  id: number;
  name: string;
  location: string;
  logo_url?: string | null;
  created_at: string;
  updated_at: string;
  users: HospitalUser[];
}

interface HospitalUser {
  id: number;
  username: string;
  email: string;
}

interface HospitalFormData {
  name: string;
  location: string;
  logoUrl?: string;
  adminUsername: string;
  adminEmail: string;
  password?: string;
}

// Dialog component
const Dialog = ({ 
  isOpen, 
  onClose, 
  title, 
  children, 
  footer
}: {
  isOpen: boolean;
  onClose: () => void;
  title: string;
  children: React.ReactNode;
  footer?: React.ReactNode;
}) => {
  if (!isOpen) return null
  
  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black bg-opacity-50">
      <div className="relative bg-white rounded-lg shadow-lg w-full max-w-md p-6 max-h-[90vh] overflow-auto">
        <div className="flex justify-between items-center mb-4">
          <h3 className="text-lg font-semibold">{title}</h3>
          <button 
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
          >
            <svg 
              xmlns="http://www.w3.org/2000/svg" 
              width="24" 
              height="24" 
              viewBox="0 0 24 24" 
              fill="none" 
              stroke="currentColor" 
              strokeWidth="2" 
              strokeLinecap="round" 
              strokeLinejoin="round" 
              className="w-5 h-5"
            >
              <line x1="18" y1="6" x2="6" y2="18"></line>
              <line x1="6" y1="6" x2="18" y2="18"></line>
            </svg>
          </button>
        </div>
        <div className="my-4">
          {children}
        </div>
        {footer && (
          <div className="flex justify-end gap-2 mt-6">
            {footer}
          </div>
        )}
      </div>
    </div>
  )
}

// Form components
const FormField = ({ 
  label, 
  children, 
  error 
}: { 
  label: string; 
  children: React.ReactNode; 
  error?: string;
}) => (
  <div className="grid gap-2 mb-4">
    <label className="text-sm font-medium">{label}</label>
    {children}
    {error && <p className="text-xs text-red-500">{error}</p>}
  </div>
)

const Input = ({ 
  className = '', 
  ...props 
}: React.InputHTMLAttributes<HTMLInputElement>) => (
  <input 
    className={`w-full p-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 ${className}`}
    {...props}
  />
)

export default function HospitalManagementPage() {
  // State for hospital data
  const [hospitals, setHospitals] = useState<Hospital[]>([]);
  const [loading, setLoading] = useState(true);
  const [expandedHospital, setExpandedHospital] = useState<number | null>(null);
  const [dbConnectionError, setDbConnectionError] = useState<string | null>(null);
  
  // State for dialogs
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  
  // State for the currently selected hospital
  const [currentHospital, setCurrentHospital] = useState<Hospital | null>(null);
  
  // State for form data
  const [formData, setFormData] = useState<HospitalFormData>({
    name: '',
    location: '',
    logoUrl: '',
    adminUsername: '',
    adminEmail: '',
    password: '',
  });
  
  // State for form validation errors
  const [formErrors, setFormErrors] = useState<Partial<Record<keyof HospitalFormData, string>>>({});
  
  // API hook
  const { fetchWithHospital, postWithHospital } = useApiWithHospital();
  
  // Fetch hospitals on mount
  useEffect(() => {
    fetchHospitals();
  }, []);
  
  // Function to fetch hospitals
  const fetchHospitals = async () => {
    setLoading(true);
    setDbConnectionError(null);
    try {
      const data = await fetchWithHospital<Hospital[]>('/api/admin/hospitals');
      setHospitals(data);
    } catch (error) {
      console.error('Error fetching hospitals:', error);
      
      let errorMessage = 'Failed to load hospitals';
      
      // Check for specific error types
      if (typeof error === 'object' && error !== null) {
        const errorObj = error as any;
        if (errorObj.status === 503) {
          errorMessage = 'Database connection failed. Please make sure your database server is running.';
          setDbConnectionError('The application could not connect to the database server. Please ensure your MySQL server is running at localhost:3306 and try again.');
        } else if (errorObj.message) {
          errorMessage = `Failed to load hospitals: ${errorObj.message}`;
        } else if (errorObj.details) {
          errorMessage = `Failed to load hospitals: ${errorObj.details}`;
        }
      }
      
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };
  
  // Toggle hospital details
  const toggleHospitalDetails = (hospitalId: number) => {
    if (expandedHospital === hospitalId) {
      setExpandedHospital(null);
    } else {
      setExpandedHospital(hospitalId);
    }
  };
  
  // Handle form input change
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    
    // Clear error for this field when user types
    if (formErrors[name as keyof HospitalFormData]) {
      setFormErrors(prev => ({ ...prev, [name]: undefined }));
    }
  };
  
  // Validate form
  const validateForm = (): boolean => {
    const errors: Partial<Record<keyof HospitalFormData, string>> = {};
    
    if (!formData.name) errors.name = 'Hospital name is required';
    if (!formData.location) errors.location = 'Location is required';
    if (!formData.adminUsername) errors.adminUsername = 'Admin username is required';
    
    if (!formData.adminEmail) {
      errors.adminEmail = 'Admin email is required';
    } else if (!/\S+@\S+\.\S+/.test(formData.adminEmail)) {
      errors.adminEmail = 'Please enter a valid email address';
    }
    
    setFormErrors(errors);
    return Object.keys(errors).length === 0;
  };
  
  // Reset form
  const resetForm = () => {
    setFormData({
      name: '',
      location: '',
      logoUrl: '',
      adminUsername: '',
      adminEmail: '',
      password: '',
    });
    setFormErrors({});
  };
  
  // Open add hospital dialog
  const openAddDialog = () => {
    resetForm();
    setIsAddDialogOpen(true);
  };
  
  // Open edit hospital dialog
  const openEditDialog = (hospital: Hospital) => {
    setCurrentHospital(hospital);
    setFormData({
      name: hospital.name,
      location: hospital.location,
      logoUrl: hospital.logo_url || '',
      // We don't include admin details in edit mode as we're not editing the admin user
      adminUsername: '',
      adminEmail: '',
    });
    setIsEditDialogOpen(true);
  };
  
  // Open delete confirmation dialog
  const openDeleteDialog = (hospital: Hospital) => {
    setCurrentHospital(hospital);
    setIsDeleteDialogOpen(true);
  };
  
  // Handle add hospital
  const handleAddHospital = async () => {
    if (!validateForm()) return;
    
    console.log('Submitting hospital form with data:', formData);
    
    try {
      const response = await postWithHospital<{
        hospital: Hospital;
        adminUser: HospitalUser;
        temporaryPassword: string;
      }>('/api/admin/hospitals', formData);
      
      console.log('Hospital creation successful, response:', response);
      setHospitals(prev => [...prev, response.hospital]);
      toast.success('Hospital and admin user created successfully');
      
      // Close the dialog and reset form after successful creation
      setIsAddDialogOpen(false);
      resetForm();
    } catch (error) {
      console.error('Error creating hospital:', error);
      // Show more specific error message if available
      if (error instanceof Error) {
        toast.error(`Failed to create hospital: ${error.message}`);
      } else if (typeof error === 'object' && error !== null) {
        // Handle our custom error object
        const errorObj = error as any;
        const errorMessage = errorObj.message || errorObj.details || 'Unknown error';
        toast.error(`Failed to create hospital: ${errorMessage}`);
      } else {
        toast.error('Failed to create hospital. Check console for details.');
      }
    }
  };
  
  // Handle edit hospital
  const handleEditHospital = async () => {
    if (!currentHospital) return;
    
    // Only validate name and location for edit
    if (!formData.name || !formData.location) {
      setFormErrors({
        name: !formData.name ? 'Hospital name is required' : undefined,
        location: !formData.location ? 'Location is required' : undefined,
      });
      return;
    }
    
    try {
      const response = await fetchWithHospital<Hospital>(`/api/admin/hospitals/${currentHospital.id}`, {
        method: 'PUT',
        body: JSON.stringify({
          name: formData.name,
          location: formData.location,
          logoUrl: formData.logoUrl,
        }),
      });
      
      setHospitals(prev => 
        prev.map(h => h.id === currentHospital.id ? response : h)
      );
      
      setIsEditDialogOpen(false);
      toast.success('Hospital updated successfully');
    } catch (error) {
      console.error('Error updating hospital:', error);
      toast.error('Failed to update hospital');
    }
  };
  
  // Handle delete hospital
  const handleDeleteHospital = async () => {
    if (!currentHospital) return;
    
    try {
      await fetchWithHospital(`/api/admin/hospitals/${currentHospital.id}`, {
        method: 'DELETE',
      });
      
      setHospitals(prev => prev.filter(h => h.id !== currentHospital.id));
      setIsDeleteDialogOpen(false);
      toast.success('Hospital deleted successfully');
    } catch (error) {
      console.error('Error deleting hospital:', error);
      toast.error('Failed to delete hospital');
    }
  };
  
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-bold">Hospital Management</h1>
        <Button onClick={openAddDialog} className="flex items-center gap-2">
          <Plus />
          <span>Add Hospital</span>
        </Button>
      </div>
      
      <Card>
        <CardHeader>
          <CardTitle>Hospitals</CardTitle>
          <CardDescription>
            Manage the hospitals in your healthcare system
          </CardDescription>
        </CardHeader>
        <CardContent>
          {dbConnectionError ? (
            <div className="py-8 px-4 text-center">
              <div className="bg-red-50 border border-red-200 rounded-md p-4 mb-4">
                <h3 className="text-lg font-medium text-red-800 mb-2">Database Connection Error</h3>
                <p className="text-red-700 mb-4">{dbConnectionError}</p>
                <div className="flex justify-center">
                  <Button 
                    onClick={fetchHospitals}
                    className="flex items-center gap-2"
                  >
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4">
                      <path d="M21 12a9 9 0 0 0-9-9 9.75 9.75 0 0 0-6.74 2.74L3 8" />
                      <path d="M3 3v5h5" />
                      <path d="M3 12a9 9 0 0 0 9 9 9.75 9.75 0 0 0 6.74-2.74L21 16" />
                      <path d="M16 21h5v-5" />
                    </svg>
                    <span>Try Again</span>
                  </Button>
                </div>
              </div>
              <div className="bg-gray-50 border border-gray-200 rounded-md p-4">
                <h4 className="font-medium text-gray-700 mb-2">Troubleshooting Steps</h4>
                <ol className="list-decimal text-left pl-5 space-y-2 text-gray-600">
                  <li>Ensure MySQL service is running on your system</li>
                  <li>Check that the database credentials in .env file are correct</li>
                  <li>Verify that the database 'audit_db' exists</li>
                  <li>Make sure the MySQL server is accessible at localhost:3306</li>
                </ol>
              </div>
            </div>
          ) : loading ? (
            <div className="py-8 text-center text-gray-500">Loading hospitals...</div>
          ) : hospitals.length === 0 ? (
            <div className="py-8 text-center text-gray-500">
              No hospitals found. Click the "Add Hospital" button to create your first hospital.
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50 text-xs font-medium text-gray-500 uppercase tracking-wider">
                  <tr>
                    <th className="px-4 py-3 text-left">Hospital</th>
                    <th className="px-4 py-3 text-left">Location</th>
                    <th className="px-4 py-3 text-left">Admin User</th>
                    <th className="px-4 py-3 text-center">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {hospitals.map((hospital) => (
                    <React.Fragment key={hospital.id}>
                      <tr className="hover:bg-gray-50">
                        <td className="px-4 py-3">
                          <div className="flex items-center gap-2">
                            <Building />
                            <span className="font-medium">{hospital.name}</span>
                          </div>
                        </td>
                        <td className="px-4 py-3">{hospital.location}</td>
                        <td className="px-4 py-3">
                          <Button
                            variant="ghost"
                            size="sm"
                            className="flex items-center gap-2"
                            onClick={() => toggleHospitalDetails(hospital.id)}
                          >
                            {(hospital.users?.length || 0) > 0 ? (
                              <Badge variant="outline" className="text-xs">
                                {hospital.users.length} user{hospital.users.length !== 1 ? 's' : ''}
                              </Badge>
                            ) : (
                              <Badge variant="outline" className="text-xs bg-red-50 text-red-700">
                                No admin
                              </Badge>
                            )}
                            {expandedHospital === hospital.id ? <ChevronUp /> : <ChevronDown />}
                          </Button>
                        </td>
                        <td className="px-4 py-3">
                          <div className="flex justify-center gap-2">
                            <Button variant="ghost" size="sm" onClick={() => openEditDialog(hospital)}>
                              <Edit />
                            </Button>
                            <Button variant="ghost" size="sm" onClick={() => openDeleteDialog(hospital)}>
                              <Trash />
                            </Button>
                          </div>
                        </td>
                      </tr>
                      
                      {expandedHospital === hospital.id && (
                        <tr>
                          <td colSpan={4} className="bg-gray-50 p-4">
                            <div className="space-y-4">
                              <h3 className="font-medium">Admin Users</h3>
                              
                              {(hospital.users?.length || 0) === 0 ? (
                                <div className="text-sm text-gray-500">
                                  No admin users found for this hospital.
                                </div>
                              ) : (
                                <div className="space-y-3">
                                  {(hospital.users || []).map(user => (
                                    <div key={user.id} className="bg-white p-3 rounded-md border">
                                      <div className="flex items-start gap-4">
                                        <div className="flex-1">
                                          <div className="flex items-center gap-2 mb-1">
                                            <User />
                                            <span className="font-medium">{user.username}</span>
                                          </div>
                                          <div className="flex items-center gap-2 text-gray-500 text-sm">
                                            <Mail />
                                            <span>{user.email}</span>
                                          </div>
                                        </div>
                                      </div>
                                    </div>
                                  ))}
                                </div>
                              )}
                            </div>
                          </td>
                        </tr>
                      )}
                    </React.Fragment>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </CardContent>
      </Card>
      
      {/* Add Hospital Dialog */}
      <Dialog
        isOpen={isAddDialogOpen}
        onClose={() => {
          setIsAddDialogOpen(false);
          resetForm();
        }}
        title="Add New Hospital"
        footer={
          <>
            <Button
              variant="outline"
              onClick={() => {
                setIsAddDialogOpen(false);
                resetForm();
              }}
            >
              Cancel
            </Button>
            <Button onClick={handleAddHospital}>
              Create Hospital
            </Button>
          </>
        }
      >
        <div className="space-y-4">
          <div>
            <h3 className="text-sm font-medium mb-2">Hospital Information</h3>
            <FormField label="Hospital Name" error={formErrors.name}>
              <Input
                name="name"
                value={formData.name}
                onChange={handleInputChange}
                placeholder="Enter hospital name"
              />
            </FormField>
            
            <FormField label="Location" error={formErrors.location}>
              <Input
                name="location"
                value={formData.location}
                onChange={handleInputChange}
                placeholder="City, State"
              />
            </FormField>
            
            <FormField label="Logo URL (Optional)">
              <Input
                name="logoUrl"
                value={formData.logoUrl}
                onChange={handleInputChange}
                placeholder="https://example.com/logo.png"
              />
            </FormField>
          </div>
          
          <div className="pt-4 border-t border-gray-200">
            <h3 className="text-sm font-medium mb-2">Admin User</h3>
            <p className="text-xs text-gray-500 mb-4">
              This user will be the administrator for this hospital and will have full access to manage the hospital's compliance data.
            </p>
            
            <FormField label="Admin Username" error={formErrors.adminUsername}>
              <Input
                name="adminUsername"
                value={formData.adminUsername}
                onChange={handleInputChange}
                placeholder="admin_username"
              />
            </FormField>
            
            <FormField label="Admin Email" error={formErrors.adminEmail}>
              <Input
                name="adminEmail"
                value={formData.adminEmail}
                onChange={handleInputChange}
                type="email"
                placeholder="admin@hospital.com"
              />
            </FormField>
            
            <FormField label="Password (Optional)">
              <Input
                name="password"
                value={formData.password}
                onChange={handleInputChange}
                type="password"
                placeholder="Leave blank to generate automatically"
              />
              <p className="text-xs text-gray-500 mt-1">
                If left blank, a secure password will be generated automatically.
              </p>
            </FormField>
          </div>
        </div>
      </Dialog>
      
      {/* Edit Hospital Dialog */}
      <Dialog
        isOpen={isEditDialogOpen}
        onClose={() => setIsEditDialogOpen(false)}
        title="Edit Hospital"
        footer={
          <>
            <Button
              variant="outline"
              onClick={() => setIsEditDialogOpen(false)}
            >
              Cancel
            </Button>
            <Button onClick={handleEditHospital}>
              Update Hospital
            </Button>
          </>
        }
      >
        <div className="space-y-4">
          <FormField label="Hospital Name" error={formErrors.name}>
            <Input
              name="name"
              value={formData.name}
              onChange={handleInputChange}
              placeholder="Enter hospital name"
            />
          </FormField>
          
          <FormField label="Location" error={formErrors.location}>
            <Input
              name="location"
              value={formData.location}
              onChange={handleInputChange}
              placeholder="City, State"
            />
          </FormField>
          
          <FormField label="Logo URL (Optional)">
            <Input
              name="logoUrl"
              value={formData.logoUrl}
              onChange={handleInputChange}
              placeholder="https://example.com/logo.png"
            />
          </FormField>
          
          <p className="text-sm text-gray-500 italic">
            Note: Editing admin users is not supported through this interface. Please contact your system administrator if you need to modify admin users.
          </p>
        </div>
      </Dialog>
      
      {/* Delete Hospital Dialog */}
      <Dialog
        isOpen={isDeleteDialogOpen}
        onClose={() => setIsDeleteDialogOpen(false)}
        title="Delete Hospital"
        footer={
          <>
            <Button
              variant="outline"
              onClick={() => setIsDeleteDialogOpen(false)}
            >
              Cancel
            </Button>
            <Button
              onClick={handleDeleteHospital}
              className="bg-red-600 hover:bg-red-700"
            >
              Delete Hospital
            </Button>
          </>
        }
      >
        <div className="space-y-4">
          <p className="text-red-600 font-medium">
            Are you sure you want to delete this hospital? This action cannot be undone.
          </p>
          
          <div className="bg-red-50 p-4 rounded-md border border-red-100">
            <p className="text-sm text-red-800 font-medium mb-2">
              This will permanently delete:
            </p>
            <ul className="text-sm text-red-700 list-disc list-inside space-y-1">
              <li>All hospital data and records</li>
              <li>All associated compliance reports</li>
              <li>All audit findings and schedules</li>
              <li>All admin users for this hospital</li>
              <li>All training events and documentation</li>
            </ul>
          </div>

          {currentHospital && (
            <div className="bg-gray-50 p-4 rounded-md border">
              <p className="font-medium mb-1">{currentHospital.name}</p>
              <p className="text-sm text-gray-500">{currentHospital.location}</p>
            </div>
          )}
        </div>
      </Dialog>
    </div>
  )
} 