'use client'

import React, { useState } from "react"
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardHeader, 
  CardTitle 
} from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"

// Simple Tabs components
interface TabsProps {
  defaultValue: string;
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

type TabChildProps = {
  activeTab?: string;
  setActiveTab?: (value: string) => void;
};

const Tabs: React.FC<TabsProps> = ({ defaultValue, children, className, ...props }) => {
  const [activeTab, setActiveTab] = useState(defaultValue)
  
  // Clone children and add the active tab state
  const enhancedChildren = React.Children.map(children, child => {
    if (React.isValidElement(child)) {
      return React.cloneElement(child as React.ReactElement<TabChildProps>, {
        activeTab,
        setActiveTab,
      })
    }
    return child
  })
  
  return (
    <div className={className} {...props}>
      {enhancedChildren}
    </div>
  )
}

interface TabsListProps {
  children: React.ReactNode;
  className?: string;
  [key: string]: unknown;
}

const TabsList: React.FC<TabsListProps> = ({ children, className, ...props }) => {
  return (
    <div className={`inline-flex h-10 items-center justify-center rounded-md bg-muted p-1 text-muted-foreground ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

interface TabsTriggerProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  setActiveTab?: (value: string) => void;
  [key: string]: unknown;
}

const TabsTrigger: React.FC<TabsTriggerProps> = ({ value, children, className, activeTab, setActiveTab, ...props }) => {
  const isActive = activeTab === value
  
  return (
    <button
      className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium transition-all
        ${isActive ? 'bg-background text-foreground shadow-sm' : 'hover:text-foreground'} 
        ${className || ''}`}
      onClick={() => setActiveTab?.(value)}
      {...props}
    >
      {children}
    </button>
  )
}

interface TabsContentProps {
  value: string;
  children: React.ReactNode;
  className?: string;
  activeTab?: string;
  [key: string]: unknown;
}

const TabsContent: React.FC<TabsContentProps> = ({ value, children, className, activeTab, ...props }) => {
  if (activeTab !== value) return null
  
  return (
    <div className={`mt-2 ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 ${className || ''}`} {...props}>
      {children}
    </div>
  )
}

// Icons
const KeyIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-4 w-4 ${className}`}>
    <path d="M21 2l-2 2m-7.61 7.61a5.5 5.5 0 1 1-7.778 7.778 5.5 5.5 0 0 1 7.777-7.777zm0 0L15.5 7.5m0 0l3 3L22 7l-3-3m-3.5 3.5L19 4"/>
  </svg>
)

const RefreshIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-4 w-4 ${className}`}>
    <path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"/>
    <path d="M21 3v5h-5"/>
    <path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"/>
    <path d="M8 16H3v5"/>
  </svg>
)

const EyeIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-4 w-4 ${className}`}>
    <path d="M2 12s3-7 10-7 10 7 10 7-3 7-10 7-10-7-10-7Z"/>
    <circle cx="12" cy="12" r="3"/>
  </svg>
)

const TrashIcon = ({ className = '' }: { className?: string }) => (
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className={`h-4 w-4 ${className}`}>
    <path d="M3 6h18"/>
    <path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"/>
    <path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"/>
  </svg>
)

export default function AccessPage() {
  const [activeTab, setActiveTab] = useState("roles")
  
  // Sample data
  const apiKeys = [
    {
      id: "key1",
      name: "Backend Integration",
      key: "aud7f9a8s7df98as7fd9asdfasdf8f",
      created: "Jun 15, 2023",
      lastUsed: "Jun 15, 2023 10:42:18",
      status: "active",
      permissions: ["read", "write"]
    },
    {
      id: "key2",
      name: "Mobile App",
      key: "87a9s6df987as6d98f7a698df69f87",
      created: "May 20, 2023",
      lastUsed: "Jun 14, 2023 16:35:02",
      status: "active",
      permissions: ["read"]
    },
    {
      id: "key3",
      name: "Data Export Service",
      key: "as87df6as98d7f6a98s7df69a87sdf",
      created: "Apr 10, 2023",
      lastUsed: "Jun 10, 2023 09:17:45",
      status: "active",
      permissions: ["read"]
    },
    {
      id: "key4",
      name: "Reporting Tool",
      key: "j8k7l6j5k4j3h2g1f2d3s4a5d6f7g8",
      created: "Mar 05, 2023",
      lastUsed: "Jun 01, 2023 14:22:09",
      status: "inactive",
      permissions: ["read"]
    }
  ]
  
  const accessTokens = [
    {
      id: "token1",
      user: "Dr. Sarah Williams",
      token: "eyJhbGciOiJIUzI1...",
      created: "Jun 15, 2023 14:30:00",
      expires: "Jun 15, 2023 16:30:00",
      status: "active"
    },
    {
      id: "token2",
      user: "John Smith",
      token: "eyJhbGciOiJIUzI1...",
      created: "Jun 15, 2023 10:15:00",
      expires: "Jun 15, 2023 12:15:00",
      status: "expired"
    },
    {
      id: "token3",
      user: "Robert Davis",
      token: "eyJhbGciOiJIUzI1...",
      created: "Jun 15, 2023 09:45:00",
      expires: "Jun 15, 2023 11:45:00",
      status: "expired"
    }
  ]
  
  const accessLogs = [
    {
      id: "log1",
      endpoint: "/api/v1/reports",
      method: "GET",
      user: "Mobile App",
      timestamp: "Jun 15, 2023 14:42:18",
      status: 200,
      responseTime: "86ms"
    },
    {
      id: "log2",
      endpoint: "/api/v1/hospitals",
      method: "GET",
      user: "Backend Integration",
      timestamp: "Jun 15, 2023 14:38:55",
      status: 200,
      responseTime: "124ms"
    },
    {
      id: "log3",
      endpoint: "/api/v1/users",
      method: "POST",
      user: "Backend Integration",
      timestamp: "Jun 15, 2023 14:36:30",
      status: 201,
      responseTime: "178ms"
    },
    {
      id: "log4",
      endpoint: "/api/v1/documents",
      method: "GET",
      user: "Data Export Service",
      timestamp: "Jun 15, 2023 14:30:12",
      status: 200,
      responseTime: "95ms"
    },
    {
      id: "log5",
      endpoint: "/api/v1/auth/token",
      method: "POST",
      user: "Anonymous",
      timestamp: "Jun 15, 2023 14:25:41",
      status: 401,
      responseTime: "65ms"
    }
  ]
  
  // Show hidden key toggle
  const [showKey, setShowKey] = useState<Record<string, boolean>>({})
  
  const toggleShowKey = (keyId: string): void => {
    setShowKey(prev => ({
      ...prev,
      [keyId]: !prev[keyId]
    }))
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Access Control</h1>
          <p className="text-muted-foreground">Manage API access and authentication tokens</p>
        </div>
      </div>

      <Tabs defaultValue="roles" className="w-full">
        <TabsList>
          <TabsTrigger value="roles" activeTab={activeTab} setActiveTab={setActiveTab}>Roles & Permissions</TabsTrigger>
          <TabsTrigger value="policies" activeTab={activeTab} setActiveTab={setActiveTab}>Access Policies</TabsTrigger>
          <TabsTrigger value="audit" activeTab={activeTab} setActiveTab={setActiveTab}>Access Audit</TabsTrigger>
          <TabsTrigger value="settings" activeTab={activeTab} setActiveTab={setActiveTab}>Settings</TabsTrigger>
        </TabsList>
        
        <TabsContent value="roles" className="space-y-4 mt-4" activeTab={activeTab}>
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Roles & Permissions</CardTitle>
                  <CardDescription>Manage user roles and their associated permissions</CardDescription>
                </div>
                <div className="flex gap-2">
                  <Button variant="outline">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-1">
                      <circle cx="11" cy="11" r="8"/>
                      <path d="m21 21-4.3-4.3"/>
                    </svg>
                    Search Roles
                  </Button>
                  <Button>
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-1">
                      <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                      <circle cx="9" cy="7" r="4"/>
                      <path d="M22 21v-2a4 4 0 0 0-3-3.87"/>
                      <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                      <path d="M23 11h-2"/>
                    </svg>
                    Add Role
                  </Button>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <table className="w-full">
                <thead>
                  <tr>
                    <th className="text-left pb-3">Role Name</th>
                    <th className="text-left pb-3">Description</th>
                    <th className="text-left pb-3">Users</th>
                    <th className="text-left pb-3">Permission Level</th>
                    <th className="text-left pb-3">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  <tr>
                    <td className="py-3">
                      <div className="flex items-center">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-2 text-blue-600">
                          <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                        </svg>
                        Super Administrator
                      </div>
                    </td>
                    <td className="py-3">System-wide full access</td>
                    <td className="py-3">2</td>
                    <td className="py-3">
                      <Badge className="bg-blue-100 text-blue-700 border-blue-300">Full Access</Badge>
                    </td>
                    <td className="py-3">
                      <div className="flex gap-1">
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                            <polyline points="14 2 14 8 20 8"/>
                          </svg>
                        </Button>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
                            <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                          </svg>
                        </Button>
                      </div>
                    </td>
                  </tr>
                  <tr>
                    <td className="py-3">
                      <div className="flex items-center">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-2 text-blue-600">
                          <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                          <circle cx="9" cy="7" r="4"/>
                        </svg>
                        Hospital Administrator
                      </div>
                    </td>
                    <td className="py-3">Hospital-specific administration</td>
                    <td className="py-3">8</td>
                    <td className="py-3">
                      <Badge className="bg-green-100 text-green-700 border-green-300">Hospital Admin</Badge>
                    </td>
                    <td className="py-3">
                      <div className="flex gap-1">
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                            <polyline points="14 2 14 8 20 8"/>
                          </svg>
                        </Button>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
                            <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                          </svg>
                        </Button>
                      </div>
                    </td>
                  </tr>
                  <tr>
                    <td className="py-3">
                      <div className="flex items-center">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-2 text-blue-600">
                          <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                          <circle cx="9" cy="7" r="4"/>
                        </svg>
                        Compliance Manager
                      </div>
                    </td>
                    <td className="py-3">Compliance reporting and management</td>
                    <td className="py-3">15</td>
                    <td className="py-3">
                      <Badge className="bg-amber-100 text-amber-700 border-amber-300">Standard</Badge>
                    </td>
                    <td className="py-3">
                      <div className="flex gap-1">
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <path d="M14.5 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V7.5L14.5 2z"/>
                            <polyline points="14 2 14 8 20 8"/>
                          </svg>
                        </Button>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <rect width="18" height="11" x="3" y="11" rx="2" ry="2"/>
                            <path d="M7 11V7a5 5 0 0 1 10 0v4"/>
                          </svg>
                        </Button>
                      </div>
                    </td>
                  </tr>
                </tbody>
              </table>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="policies" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>Access Policies</CardTitle>
                  <CardDescription>Configure role-based access control policies</CardDescription>
                </div>
                <Button>
                  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 mr-1">
                    <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>
                  </svg>
                  Add Policy
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <table className="w-full">
                <thead>
                  <tr>
                    <th className="text-left pb-3">Policy Name</th>
                    <th className="text-left pb-3">Applied To</th>
                    <th className="text-left pb-3">Resource</th>
                    <th className="text-left pb-3">Permission</th>
                    <th className="text-left pb-3">Status</th>
                    <th className="text-left pb-3">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  <tr>
                    <td className="py-3 font-medium">Hospital Data Access</td>
                    <td className="py-3">Hospital Administrators</td>
                    <td className="py-3">Hospital Records</td>
                    <td className="py-3">Read/Write</td>
                    <td className="py-3">
                      <Badge className="bg-green-100 text-green-700">Active</Badge>
                    </td>
                    <td className="py-3">
                      <div className="flex gap-1">
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                          </svg>
                        </Button>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <TrashIcon />
                        </Button>
                      </div>
                    </td>
                  </tr>
                  <tr>
                    <td className="py-3 font-medium">Compliance Document Access</td>
                    <td className="py-3">Compliance Managers</td>
                    <td className="py-3">Compliance Documents</td>
                    <td className="py-3">Read</td>
                    <td className="py-3">
                      <Badge className="bg-green-100 text-green-700">Active</Badge>
                    </td>
                    <td className="py-3">
                      <div className="flex gap-1">
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round" className="h-4 w-4 text-gray-500">
                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"/>
                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"/>
                          </svg>
                        </Button>
                        <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                          <TrashIcon />
                        </Button>
                      </div>
                    </td>
                  </tr>
                </tbody>
              </table>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="api-keys" className="space-y-4 mt-4" activeTab={activeTab}>
          <div className="flex justify-end">
            <Button>
              <KeyIcon className="mr-2" />
              Generate New API Key
            </Button>
          </div>
          
          <Card>
            <CardHeader>
              <CardTitle>API Keys</CardTitle>
              <CardDescription>Manage API keys for external integrations</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <tr>
                      <th className="px-4 py-3 text-left">Name</th>
                      <th className="px-4 py-3 text-left">API Key</th>
                      <th className="px-4 py-3 text-left">Created</th>
                      <th className="px-4 py-3 text-left">Last Used</th>
                      <th className="px-4 py-3 text-left">Permissions</th>
                      <th className="px-4 py-3 text-right">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {apiKeys.map((key) => (
                      <tr key={key.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3 font-medium">
                          <div className="flex items-center">
                            {key.name}
                            <Badge className={`ml-2 ${
                              key.status === 'active' 
                                ? 'bg-green-100 text-green-800' 
                                : 'bg-gray-100 text-gray-800'
                            }`}>
                              {key.status === 'active' ? 'Active' : 'Inactive'}
                            </Badge>
                          </div>
                        </td>
                        <td className="px-4 py-3 font-mono text-xs">
                          {showKey[key.id] ? key.key : '•••••••••••••••••••••••••••••••••'}
                        </td>
                        <td className="px-4 py-3 text-sm">{key.created}</td>
                        <td className="px-4 py-3 text-sm">{key.lastUsed}</td>
                        <td className="px-4 py-3">
                          <div className="flex gap-1">
                            {key.permissions.includes('read') && (
                              <Badge className="bg-blue-100 text-blue-800">Read</Badge>
                            )}
                            {key.permissions.includes('write') && (
                              <Badge className="bg-purple-100 text-purple-800">Write</Badge>
                            )}
                          </div>
                        </td>
                        <td className="px-4 py-3 text-right">
                          <div className="flex justify-end space-x-2">
                            <Button variant="ghost" size="sm" onClick={() => toggleShowKey(key.id)}>
                              <EyeIcon />
                            </Button>
                            <Button variant="ghost" size="sm">
                              <RefreshIcon />
                            </Button>
                            <Button variant="ghost" size="sm" className="text-red-500">
                              <TrashIcon />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="access-tokens" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>Access Tokens</CardTitle>
              <CardDescription>Manage user access tokens</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <tr>
                      <th className="px-4 py-3 text-left">User</th>
                      <th className="px-4 py-3 text-left">Token</th>
                      <th className="px-4 py-3 text-left">Created</th>
                      <th className="px-4 py-3 text-left">Expires</th>
                      <th className="px-4 py-3 text-left">Status</th>
                      <th className="px-4 py-3 text-right">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {accessTokens.map((token) => (
                      <tr key={token.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3 font-medium">{token.user}</td>
                        <td className="px-4 py-3 font-mono text-xs truncate max-w-[240px]">
                          {showKey[token.id] ? token.token : '•••••••••••••••••••'}
                        </td>
                        <td className="px-4 py-3 text-sm">{token.created}</td>
                        <td className="px-4 py-3 text-sm">{token.expires}</td>
                        <td className="px-4 py-3">
                          <Badge className={
                            token.status === 'active' 
                              ? 'bg-green-100 text-green-800' 
                              : 'bg-gray-100 text-gray-800'
                          }>
                            {token.status.charAt(0).toUpperCase() + token.status.slice(1)}
                          </Badge>
                        </td>
                        <td className="px-4 py-3 text-right">
                          <div className="flex justify-end space-x-2">
                            <Button variant="ghost" size="sm" onClick={() => toggleShowKey(token.id)}>
                              <EyeIcon />
                            </Button>
                            <Button variant="ghost" size="sm" className="text-red-500">
                              <TrashIcon />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
        
        <TabsContent value="access-logs" activeTab={activeTab}>
          <Card className="mt-4">
            <CardHeader>
              <CardTitle>API Access Logs</CardTitle>
              <CardDescription>Monitor API access and usage</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <tr>
                      <th className="px-4 py-3 text-left">Endpoint</th>
                      <th className="px-4 py-3 text-left">Method</th>
                      <th className="px-4 py-3 text-left">User</th>
                      <th className="px-4 py-3 text-left">Timestamp</th>
                      <th className="px-4 py-3 text-left">Status</th>
                      <th className="px-4 py-3 text-left">Response Time</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {accessLogs.map((log) => (
                      <tr key={log.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3 font-mono text-xs">{log.endpoint}</td>
                        <td className="px-4 py-3">
                          <Badge className={
                            log.method === 'GET' 
                              ? 'bg-blue-100 text-blue-800' 
                              : log.method === 'POST'
                                ? 'bg-green-100 text-green-800'
                                : log.method === 'PUT'
                                  ? 'bg-amber-100 text-amber-800'
                                  : 'bg-red-100 text-red-800'
                          }>
                            {log.method}
                          </Badge>
                        </td>
                        <td className="px-4 py-3 text-sm">{log.user}</td>
                        <td className="px-4 py-3 text-sm">{log.timestamp}</td>
                        <td className="px-4 py-3">
                          <Badge className={
                            log.status >= 200 && log.status < 300
                              ? 'bg-green-100 text-green-800' 
                              : log.status >= 400 && log.status < 500
                                ? 'bg-amber-100 text-amber-800'
                                : 'bg-red-100 text-red-800'
                          }>
                            {log.status}
                          </Badge>
                        </td>
                        <td className="px-4 py-3 text-sm">{log.responseTime}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              
              <div className="flex justify-between items-center mt-6">
                <div className="text-sm text-gray-500">
                  Showing 5 of 2,457 logs
                </div>
                <div className="flex items-center space-x-2">
                  <Button variant="outline" size="sm">
                    Previous
                  </Button>
                  <Button variant="outline" size="sm">
                    Next
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
} 